#include <vector>
#include "mbed.h"
#include "SerialRPCInterface.h"

#define DAC_WRITE_UPDATE_REG 0x03
// Create a USB serial port 
Serial pc(USBTX, USBRX);
// SPI info for the DAC
SPI spi(p11, p12, p13); // mosi, miso, sclk
DigitalOut cs(p14); // Chip select, active low

// pin info for the frequency counter
DigitalOut clear_data(p21);     // Clears data, active low
DigitalOut CSAL(p16);           // Selects the A Lower storage register (LSB of register A), active low
DigitalOut CSAU(p17);           // Selects the A Upper storage register (MSB of register A), active low
DigitalOut CSBL(p18);           // Selects the B Lower storage register (LSB of register B), active low
DigitalOut CSBU(p19);           // Selects the b Upper storage register (MSB of register B), active low
DigitalOut store_data(p20);     // Stores all data to the appropriate storage register, stores on the rising edge
BusIn raw_data(p29, p28, p27, p26, p25, p24, p23, p22);     // Mbed reads the output of the selected storage register

// Define the base integration time and response time delay. These get written over by RPCs
float integration_time = 100;  // How long we integrate for in ms
float response_time_delay = 50; // Time delay to account for response time in ms

// Define the sequence length and set length and their respective counting varialbes for the "for" loop
int   sequence_length = 1;
int   set_length = 10;

// Define variables used for the DAC
vector<unsigned int> voltage_vector;// Voltage vector to be filled in the function "voltage_vector_function"

// Defines RPC variables for the above
RPCVariable<float> rpcintegration_time(&integration_time, "integration_time");
RPCVariable<float> rpcresponse_time_delay(&response_time_delay, "response_time_delay");
RPCVariable<int> rpcsequence_length(&sequence_length, "sequence_length");
RPCVariable<int> rpcset_length(&set_length, "set_length");

// Defines RPC functions and makes them callable
void XPS(Arguments *input, Reply *output);
void voltage_vector_function(Arguments *input, Reply *output);

RPCFunction rpcXPS(&XPS, "XPS");
RPCFunction rpcvoltage_vector_function(&voltage_vector_function, "voltage_vector_function");

// Define the voltage vetor function, which receives the voltage vector from matlab 
// the vector is stored as 8 bit integers. However, each voltage is 16 bit integers. Therefore, when
// we call the voltage vector during XPS, we simply call 2n and 2n+1 of the voltage vetor, which also makes up the 
// two 8 bit int variables we feed into the DAC
void voltage_vector_function(Arguments *in, Reply *out)
{
    voltage_vector.resize(set_length*sizeof(int16_t)); // resize the vector to fit what is expected
    for(int i = 0;i<set_length*sizeof(int16_t);){
        if(pc.readable()){                             // check for character in the serial port
        voltage_vector[i] = pc.getc();                 // get character from serial port
        i = i+1;
        }
    }  
}


// Start of XPS function, callable with RPCs
void XPS(Arguments *in, Reply *out)
{
    //initiate variables to store data
    int16_t   counts[sequence_length];
    int data_store[2][sequence_length];
    
    
    // initiate first voltage from voltage_vector to the DAC
    cs = 0;
    spi.write(DAC_WRITE_UPDATE_REG);
    spi.write(voltage_vector[1]);
    spi.write(voltage_vector[0]);
    cs = 1; 
    
    // Time delay to stabilize
    wait_ms(response_time_delay);
    // Prepare the frequency counter
    store_data.write(1);
    clear_data.write(1);
    CSAL.write(1);
    CSAU.write(1);
    CSBL.write(1);
    CSBU.write(1);


    for(int m =0; m<set_length; m++) {
        // Begins with first point
        // Voltage was set at line 70 
        // Set new voltage and prepare to write to the DAC
        cs = 0;
        spi.write(DAC_WRITE_UPDATE_REG);
        spi.write(voltage_vector[2*m+1]);
        spi.write(voltage_vector[2*m]);
        cs = 1;
        
        wait_ms(response_time_delay);
        // Get one set of data to be transferred
        for (int n = 0; n<sequence_length; n++) {
            
            clear_data.write(0);        // Stop and clear the counter
            store_data.write(0);        // prep storage pin to store on rising edge
            clear_data.write(1);        // Start counter again
            
            wait_ms(integration_time);  // Delay for the integration time

            store_data.write(1);        // Store the data on the rising edge of the store_data pin
            
            // Store the data from the AL storage register
            CSAL.write(0);
            data_store[0][n] = raw_data;
            CSAL.write(1);
            
            wait_us(1);
            
            // Store the data from the AU storage register
            CSAU.write(0);
            data_store[1][n] =  raw_data;
            CSAU.write(1);
            
            // We have the option to use the BL and BU registers to extend the number of counts we can read,
            // but I have omitted those for speed and simplicity. 
        }
        // combine LSB and MSB of the A register and prepare to print to MatLab
        for (int n = 0; n<sequence_length; n++) {
            data_store[1][n] = data_store[1][n]<<8;
            counts[n] = data_store[1][n]| data_store[0][n];
        }
        // Print to MatLab
        
        for (int i = 0; i < sequence_length*sizeof(int16_t); i++) {
            pc.putc( *((char*)counts + i) );
        }
       
        
    }
}


// Main loop, used to prepare the DAC and call RPCs
int main()
{
    // Set baud rate for serial port
    pc.baud(115200);
    
    // Prepare DAC
    // DAC must be deselected
    cs = 1;
    spi.format(8,2);
    spi.frequency(10000);
    // Start with a software full reset
    cs = 0;
    spi.write(0x0F);
    spi.write(0);
    spi.write(0);
    cs = 1;
    wait(0.5);
    // Write to command register
    // set voltage range to 0-10V
    cs = 0;
    spi.write(0x04);
    spi.write(0x00);
    spi.write(0x21);
    cs = 1;
    
    wait(0.5);

    // prepare RPC read
    char buf[256], outbuf[256];
    while(1) {
        pc.gets(buf, 256);
        RPC::call(buf, outbuf);
        // Print the response
        pc.printf("%s\n", outbuf);
    }

}
