#include <vector>
#include "mbed.h"
#include "SerialRPCInterface.h"

#define ADS1115_ADDR     (0x90) // ADS1115 base address 0x18<<1
#define ADS1115_REG_CONF (0x01) // Configuration Register
#define ADS1115_REG_CONV (0x00) // Conversion Register
#define ADS1115_REG_HI_THRESH (0x03) // High Thresh Register
#define ADS1115_REG_LOW_THRESH (0x02) // Low Thresh REgister

#define DAC_WRITE_UPDATE_REG 0x03
// Create a USB serial port 
Serial pc(USBTX, USBRX);
I2C i2c(p9, p10);   // SDA, SCL on ADS1115

SPI spi(p11, p12, p13); // mosi, miso, sclk
DigitalOut cs(p14); // Chip select, active low

DigitalOut reset_mbed(p15);
reset_mbed = 1;
// Define the base integration time and response time delay. These get written over by RPCs
float integration_time = 100;  // How long we integrate for in ms
float response_time_delay = 50; // Time delay to account for response time in ms

// Define the sequence length and set length and their respective counting varialbes for the "for" loop
int   sequence_length = 1;
int   set_length = 10;

char  write_config[3];
char  read_conv[2];
int ii;
// Define variables used for the DAC
vector<unsigned int> voltage_vector;// Voltage vector to be filled in the function "voltage_vector_function"

// Defines RPC variables for the above
RPCVariable<float> rpcintegration_time(&integration_time, "integration_time");
RPCVariable<float> rpcresponse_time_delay(&response_time_delay, "response_time_delay");
RPCVariable<int> rpcsequence_length(&sequence_length, "sequence_length");
RPCVariable<int> rpcset_length(&set_length, "set_length");

// Defines RPC functions and makes them callable
void AES(Arguments *input, Reply *output);
void voltage_vector_function(Arguments *input, Reply *output);
void Mbed_reset(Arguments *input, Reply *output);


RPCFunction rpcAES(&AES, "AES");
RPCFunction rpcvoltage_vector_function(&voltage_vector_function, "voltage_vector_function");

// Define the voltage vetor function, which receives the voltage vector from matlab 
// the vector is stored as 8 bit integers. However, each voltage is 16 bit integers. Therefore, when
// we call the voltage vector during AES, we simply call 2n and 2n+1 of the voltage vetor, which also makes up the 
// two 8 bit int variables we feed into the DAC
void voltage_vector_function(Arguments *in, Reply *out)
{
    voltage_vector.resize(set_length*sizeof(int16_t)); // resize the vector to fit what is expected
    for(int i = 0;i<set_length*sizeof(int16_t);){
        if(pc.readable()){                             // check for character in the serial port
        voltage_vector[i] = pc.getc();                 // get character from serial port
        i = i+1;
        }
    }  
}


// Start of AES function, callable with RPCs
void AES(Arguments *in, Reply *out)
{
    //initiate variables to store data
    int16_t   counts[sequence_length];
    char data_store[2][sequence_length];
    
    // Prepare the ADC as necessay
    write_config[0] = ADS1115_REG_CONF; // Register address
    write_config[1] = 0x0A;
    write_config[2] = 0xE4; // Set to E0 for ALRT off and E4 for ALRT on
    i2c.write(ADS1115_ADDR, write_config, 3, 0); // address,data,length,repeat

    for(int m =0; m<set_length; m++) {
        // Begins with first point
        // Set new voltage and prepare to write to the DAC
        cs = 0;
        spi.write(DAC_WRITE_UPDATE_REG);
        spi.write(voltage_vector[2*m+1]);
        spi.write(voltage_vector[2*m]);
        cs = 1;
        
        wait_ms(response_time_delay);
        // Get one set of data to be transferred
            for(int n=0;n<sequence_length;n++)
                {       
                    wait_ms(integration_time);           
                    // Read data from the ADS1115
                    write_config[0] = ADS1115_REG_CONV;
                    i2c.write(ADS1115_ADDR, write_config, 1, 0); // no stop
                    i2c.read(ADS1115_ADDR, read_conv, 2, 0);  
                      
                    // Store the data for later conversion
                    data_store[0][n] = read_conv[0];
                    data_store[1][n] = read_conv[1];
                }
        
            for(int n = 0; n<sequence_length;n++)
                {
                    // Run 2's compliment
        
                    ii = int(data_store[0][n]);
                        if (ii > 127)
                            {
                                counts[n] = 256*data_store[0][n]+data_store[1][n];
                                counts[n] = counts[n] - 65536;
                            }
                        else 
                            {
                                counts[n] = 256*data_store[0][n]+data_store[1][n];
                            }                    
                }
        // Print to MatLab as two 8 bit characters
        for (int i = 0; i < sequence_length*sizeof(int16_t); i++) {
            pc.putc( *((char*)counts + i) );
        }
       
        
    }
}


// Main loop, used to prepare the DAC and call RPCs
int main()
{
    // Set baud rate for serial port
    pc.baud(115200);
    
    // Prepare DAC
    // DAC must be deselected
    cs = 1;
    spi.format(8,2);
    spi.frequency(10000);
    // Start with a software full reset
    cs = 0;
    spi.write(0x0F);
    spi.write(0);
    spi.write(0);
    cs = 1;
    wait(0.5);
    // Write to command register
    // set voltage range to 0-10V
    cs = 0;
    spi.write(0x04);
    spi.write(0x00);
    spi.write(0x21);
    cs = 1;
   
    wait(0.5);

    // prepare RPC read
    char buf[256], outbuf[256];
    while(1) {
        pc.gets(buf, 256);
        RPC::call(buf, outbuf);
        // Print the response
        pc.printf("%s\n", outbuf);
    }

}
