#ifndef SNAKE_H
#define SNAKE_H

#include "mbed.h"
#include "N5110.h"
#include "Gamepad.h"

/** Snake class

@brief Class for the sanke game
@brief It includes the sanke,ball,socres and collision_check

@author Qi Minghong

@date April 2019

@code
///////// pre-processor directives ////////
#include "mbed.h"
#include "Gamepad.h"
#include "N5110.h"
#include "Snake.h"
#include "Engine.h"
#include "Finger.h"

/////////////// objects ///////////////
N5110 lcd(PTC9,PTC0,PTC7,PTD2,PTD1,PTC11);
Gamepad pad;
Snake snake;
Engine engine;
Finger finger;

///////////// prototypes ///////////////
void init();
void run();
void over();

///////////// functions ////////////////
int main()
{
    int fps = 8;  // frames per second
    snake.hscore = 0;
    lcd.init();
    while(1){    
      init();
      pad.tone(900.0,0.5);
      engine.welcome(pad,lcd);  // show welcome display, waiting for the user to start
      engine.menu(pad,lcd);  // show the select display, waiting for the user to select
    // choose game and game loop
    if(engine.game == 0){
      while (snake.over == 0) {
        run(); // run the snake game
        while (pad.check_event(Gamepad::START_PRESSED) == true){
            engine.pause(pad,lcd); // pause the game
        }
        wait(engine.p/fps);  // and wait for one frame period
      }
     over();  // show gameover display, waiting for the user to restart
     }else if(engine.game == 1){
         finger.run(lcd,pad); // run the finger-geuss game
         }
    }
}

// this function draws each frame on the LCD
void run()
{
    // clear screen, re-draw and refresh
    lcd.clear();
    int length = snake.get_length();
    int direction = engine.get_direction(pad);
    snake.check_eat(pad);
    snake.draw(lcd,length);
    snake.update(direction,length);
    snake.check_over(lcd);
    lcd.refresh();
}

// initialies all classes and libraries
void init()
{
    // need to initialise LCD and Gamepad
    pad.init();    
    snake.init(2,3);
    engine.init();
    finger.init();
    lcd.clear();
}

//gameover screen
void over()
{
    int s = snake.get_score(); // get the scores to show at the gameover display
    int hs = snake.hscore;
    engine.gameover(pad,lcd,s,hs);

}
@endcode
*/ 

class Snake
{

public:
    /** Constructor */
    Snake();
    /** Destructor */
    ~Snake();
    /** Get the score
    *
    *   This function gets the score of user.
    */
    int get_score();
    /** Get the length
    *
    *   This function gets the length of the sanke.
    */
    int get_length();
    int over;
    int hscore;
    /** Initialise the components of snake
    *
    *   This function innitialises the the components of snake.
    *   @param  size - the size of the snake
    *   @param  length - the length of the snake
    */
    void init(int size,int length);
    /** Update the data
    *
    *   This function gets the new position of the snake.
    *   @param  direction - the direction of the snake
    *   @param  length - the length of the snake
    */
    void update(int direction, int length);
    /** Draw the snake and ball
    *
    *   This function draws the snake and ball.
    *   @param  lcd - N5110 library
    *   @param  length - the length of the snake
    */
    void draw(N5110 &lcd, int length);
    /** Check eat
    *
    *   This function checks whether the snake touthes the ball.
    *   @param  pad - Gamepad library
    */
    void check_eat(Gamepad &pad);
    /** Check over
    *
    *   This function checks whether the snake runs into the wall or itself.
    *   @param  lcd - N5110 library
    */
    void check_over(N5110 &lcd);

private:
    int _size;
    int _x [100];
    int _y [100];
    int _score;
    int _length;
    int ball_x;
    int ball_y; 
    void ball(int length);

};
#endif