#ifndef ENGINE_H
#define ENGINE_H

#include "mbed.h"
#include "N5110.h"
#include "Gamepad.h"
#include "Snake.h"
#include "Finger.h"

/** Engine class

@brief Class for the structure of the game
@brief It includes the control input, menu and other display parts(welcome,select,gameover)

@author Qi Minghong

@date April 2019

@code
///////// pre-processor directives ////////
#include "mbed.h"
#include "Gamepad.h"
#include "N5110.h"
#include "Snake.h"
#include "Engine.h"
#include "Finger.h"

/////////////// objects ///////////////
N5110 lcd(PTC9,PTC0,PTC7,PTD2,PTD1,PTC11);
Gamepad pad;
Snake snake;
Engine engine;
Finger finger;

///////////// prototypes ///////////////
void init();
void run();
void over();

///////////// functions ////////////////
int main()
{
    int fps = 8;  // frames per second
    snake.hscore = 0;
    lcd.init();
    while(1){    
      init();
      pad.tone(900.0,0.5);
      engine.welcome(pad,lcd);  // show welcome display, waiting for the user to start
      engine.menu(pad,lcd);  // show the select display, waiting for the user to select
    // choose game and game loop
    if(engine.game == 0){
      while (snake.over == 0) {
        run(); // run the snake game
        while (pad.check_event(Gamepad::START_PRESSED) == true){
            engine.pause(pad,lcd); // pause the game
        }
        wait(engine.p/fps);  // and wait for one frame period
      }
     over();  // show gameover display, waiting for the user to restart
     }else if(engine.game == 1){
         finger.run(lcd,pad); // run the finger-geuss game
         }
    }
}

// this function draws each frame on the LCD
void run()
{
    // clear screen, re-draw and refresh
    lcd.clear();
    int length = snake.get_length();
    int direction = engine.get_direction(pad);
    snake.check_eat(pad);
    snake.draw(lcd,length);
    snake.update(direction,length);
    snake.check_over(lcd);
    lcd.refresh();
}

// initialies all classes and libraries
void init()
{
    // need to initialise LCD and Gamepad
    pad.init();    
    snake.init(2,3);
    engine.init();
    finger.init();
    lcd.clear();
}

//gameover screen
void over()
{
    int s = snake.get_score(); // get the scores to show at the gameover display
    int hs = snake.hscore;
    engine.gameover(pad,lcd,s,hs);

}
@endcode
*/ 

class Engine
{
    
    public:
    /** Constructor */
    Engine();
    /** Destructor */
    ~Engine();
    int direction;
    int game;
    /** Get the direction
    *
    *   This function gets the direction of the snake (using the gamepad).
    *   @param  pad - Gamepad library
    */
    int get_direction(Gamepad &pad);
    /** Get the selection
    *
    *   This function gets the selection of user (using the gamepad).
    *   @param  pad - Gamepad library
    */
    int get_select(Gamepad &pad);
    /** Initialise the components
    *
    *   This function initialises the components of engine.
    */
    void init();
    /** Welcome display
    *
    *   This function prints the welcome display and wait for user to select.
    *   @param  pad - Gamepad library
    *   @param  lcd - N5110 library
    */
    void welcome(Gamepad &pad, N5110 &lcd);
    /** Gameover display
    *
    *   This function prints the gameover display and wait for user to select.
    *   @param  pad - Gamepad library
    *   @param  lcd - N5110 library
    *   @param  score - the score of user
    *   @param  hscore - the max score user had achieved
    */
    void gameover(Gamepad &pad, N5110 &lcd, int score, int hscore);
    /** Select the difficulty
    *
    *   This function prints the select display and wait for user to select the speed of snake.
    *   @param  pad - Gamepad library
    *   @param  lcd - N5110 library
    */
    void select(Gamepad &pad, N5110 &lcd);
    /** Pause the game
    *
    *   This function prints the select display and wait for user to restart game.
    *   @param  pad - Gamepad library
    *   @param  lcd - N5110 library
    */
    void pause(Gamepad &pad, N5110 &lcd);
    /** Menu
    *
    *   This function prints the menu display and wait for user to select the game.
    *   @param  pad - Gamepad library
    *   @param  lcd - N5110 library
    */
    void menu(Gamepad &pad, N5110 &lcd);
    /** Setting
    *
    *   This function prints the setting display and wait for user to control the light or contrast of LED.
    *   @param  pad - Gamepad library
    *   @param  lcd - N5110 library
    */
    void setting(Gamepad &pad, N5110 &lcd);
    float p;
    
    private:
    int s;
    

};

#endif