/// @copyright
/// ========================================================================={{{
/// Copyright (c) 2013-2016 WizziLab                                           /
/// All rights reserved                                                        /
/// =========================================================================}}}
/// @endcopyright

//  =======================================================================
/// @file       wc_deserializer.c
/// @brief      Exemple of WizziCom serial transport layer parser.
///             Can be used as entry-point/scheduler for modem_ref
//  =======================================================================
#include "modem_ref.h"

// Depending on Host needs/capacities, choose variable-size buffer allocation style
#if 1 // static buffer alloc
    #define MAX_RX_BUFFER_SIZE              (256)
    #define GET_BUFFER(_s)                  rxbuf
    #define FREE_BUFFER(_b)
    static u8 rxbuf[MAX_RX_BUFFER_SIZE];
#else // Dynamic alloc
    #define GET_BUFFER(_s)                  MALLOC(_s);
    #define FREE_BUFFER(_b)                 FREE(_b);
#endif

enum { S_SYNC0, S_SYNC1, S_SIZE, S_SEQU, S_FLOWID, S_DATA };

//======================================================================
// wc_deserializer
//----------------------------------------------------------------------
/// @brief  Parse serial flow character-wise and extract packets from
///         WizziCom serial transport layer (WC).
///         It calls modem_input on complete payloads.
/// @param  c           : received character.
/// @param  init        : 0 when parsing, 1 to init/reinit parser.
/// @return flowid of successfuly parsed packet, 0 otherwise.
/// @note   This function must be called on every received character on
///         modem serial link
/// @note   This function should not be directly called from (serial)
///         ISR as it is the entry point for all driver/callback execution.
/// @note   If WC deserialization is performed by other means, and that
///         full packets are available, one should do direct calls to
///         modem_input
//======================================================================
public int wc_deserializer(char c, u8 init)
{
    static u8 state,rx_ptr,rx_size,rx_flowid;
    static u8* rx_data;

    if (init) { state = S_SYNC0; }
    // WC Packet decap'
    switch(state)
    {
        case S_SYNC0:
            state   = (c == WC_SYNC_BYTE_0)?S_SYNC1:S_SYNC0;
            rx_ptr  = 0;
            rx_data = (u8*)NULL;
            return 0;
        case S_SYNC1:
            state = (c == WC_SYNC_BYTE_1)?S_SIZE:
                    (c == WC_SYNC_BYTE_0)?S_SYNC1:S_SYNC0;
            return 0;
        case S_SIZE:
            state       = S_SEQU;
            rx_size     = (u8)c;
            if (rx_size) rx_data = GET_BUFFER(rx_size);
            return 0;
        case S_SEQU:
            state       = S_FLOWID;
            //rx_sequ     = (u8)c;
            return 0;
        case S_FLOWID:
            state       = (rx_size!=0)?S_DATA:S_SYNC0;
            rx_flowid   = (u8)c;
            break;
        case S_DATA:
            state       = (rx_ptr >= rx_size)?S_SYNC0:S_DATA;
            rx_data[rx_ptr++]  = (u8)c;
            break;
        default:
            break;
    }

    if (rx_size == rx_ptr)
    {
        modem_input(rx_flowid,rx_data,rx_size);
        if (rx_data) { FREE_BUFFER(rx_data); }
    }
    return rx_flowid;
}
