/// @copyright
/// ========================================================================={{{
/// Copyright (c) 2012-2013 WizziLab                                           /
/// All rights reserved                                                        /
///                                                                            /
/// IMPORTANT: This Software may not be modified, copied or distributed unless /
/// embedded on a WizziLab product. Other than for the foregoing purpose, this /
/// Software and/or its documentation may not be used, reproduced, copied,     /
/// prepared derivative works of, modified, performed, distributed, displayed  /
/// or sold for any purpose. For the sole purpose of embedding this Software   /
/// on a WizziLab product, copy, modification and distribution of this         /
/// Software is granted provided that the following conditions are respected:  /
///                                                                            /
/// *  Redistributions of source code must retain the above copyright notice,  /
///    this list of conditions and the following disclaimer                    /
///                                                                            /
/// *  Redistributions in binary form must reproduce the above copyright       / 
///    notice, this list of conditions and the following disclaimer in the     /
///    documentation and/or other materials provided with the distribution.    /
///                                                                            /
/// *  The name of WizziLab can not be used to endorse or promote products     /
///    derived from this software without specific prior written permission.   /
///                                                                            /
/// THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS        /
/// "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED  /
/// TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR /
/// PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR          /
/// CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,      /
/// EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,        /
/// PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA,            /
/// OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY     /
/// OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING    /
/// NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS         /
/// SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.               /
/// WIZZILAB HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT, UPDATES,       /
/// ENHANCEMENTS OR MODIFICATIONS.                                             /
///                                                                            /
/// Should you have any questions regarding your right to use this Software,   /
/// contact WizziLab at www.wizzilab.com.                                      /
///                                                                            /
/// =========================================================================}}}
/// @endcopyright

//	====================================================================
/// @file           hal_types.h
/// @brief          platform types definitions
/// @ingroup        HAL
/// @defgroup HAL_TYPES      HAL Types
/// @{
//	====================================================================

#ifndef __HAL_TYPES_H__
#define __HAL_TYPES_H__

//======================================================================
// Common Types
//======================================================================
typedef signed long  s32;
typedef signed short s16;
typedef signed char  s8;

typedef unsigned long  u32;
typedef unsigned short u16;
typedef unsigned char  u8;

typedef volatile signed long  vs32;
typedef volatile signed short vs16;
typedef volatile signed char  vs8;

typedef volatile unsigned long  vu32;
typedef volatile unsigned short vu16;
typedef volatile unsigned char  vu8;

typedef signed long  const sc32;
typedef signed short const sc16;
typedef signed char  const sc8;

typedef unsigned long  const uc32;
typedef unsigned short const uc16;
typedef unsigned char  const uc8;

typedef volatile signed long  const vsc32;
typedef volatile signed short const vsc16;
typedef volatile signed char  const vsc8;

typedef volatile unsigned long  const vuc32;
typedef volatile unsigned short const vuc16;
typedef volatile unsigned char  const vuc8;

typedef unsigned int  uint;
typedef int  sint;

typedef volatile unsigned int  vuint;
typedef volatile int  vsint;

typedef unsigned int const  cuint;
typedef int const  csint;

//typedef unsigned int size_t;
//typedef unsigned long int uint32_t;

typedef struct {
    s16 I;
    s16 Q;
} complex_t;

typedef struct {
    u8 m;
    u8 e;
} mant_exp_t;

typedef struct {
    s16 x;
    s16 y;
    s16 z;
} v3d_t;

//======================================================================
// Common Enum / Defines
//======================================================================
#ifndef bool
 //typedef u8 bool;
#endif

#ifndef false
 #define false 0
#endif
#ifndef true
 #define true 1
#endif
#ifndef FALSE
 #define FALSE 0
#endif
#ifndef TRUE
 #define TRUE 1
#endif

#ifndef NULL
 #define NULL       (void*)0
#endif

#define typeof(s) __typeof__(s)

#define U8_MAX     ((u8)255)
#define S8_MAX     ((s8)127)
#define S8_MIN     ((s8)-128)
#define U16_MAX    ((u16)65535u)
#define S16_MAX    ((s16)32767)
#define S16_MIN    ((s16)-32768)
#define U32_MAX    ((u32)4294967295uL)
#define S32_MAX    ((s32)2147483647)
#define S32_MIN    ((s32)-2147483648)

//======================================================================
// Common Endianness cast & conversion Macros
//======================================================================
#define HAL_U16_B0(a)           ((u8)(((u16)(a) >>  0) & 0xFF))
#define HAL_U16_B1(a)           ((u8)(((u16)(a) >>  8) & 0xFF))
#define HAL_U16_LO(a)           ((u8)((a) >> 0))
#define HAL_U16_HI(a)           ((u8)((a) >> 8))
#define HAL_TO_U16(a,b)         ((u16)((u16)(a) << 8)|(u16)(b))
#define HAL_TO_U32(a,b,c,d)     ((u32)((u32)(a) <<24)|((u32)(b) <<16)|((u32)(c) << 8)|(u32)(d))
#define HAL_U32_LO(a)           ((u16)((a) >> 0))
#define HAL_U32_HI(a)           ((u16)((a) >> 16))
#define HAL_U32_B0(a)           ((u8)(((u32)(a) >>  0) & 0xFF))
#define HAL_U32_B1(a)           ((u8)(((u32)(a) >>  8) & 0xFF))
#define HAL_U32_B2(a)           ((u8)(((u32)(a) >> 16) & 0xFF))
#define HAL_U32_B3(a)           ((u8)(((u32)(a) >> 24) & 0xFF))
// This should generate a REV instruction
#define HAL_U32_BYTE_SWAP(a)    (   (((a) & 0x000000ff) << 24) |   \
                                    (((a) & 0x0000ff00) <<  8) |   \
                                    (((a) & 0x00ff0000) >>  8) |   \
                                    (((a) & 0xff000000) >> 24))
// This should generate a REV16 instruction
#define HAL_U16_BYTE_SWAP(a)    ((((a) & 0x00ff) << 8) | (((a) & 0xff00) >> 8))
// Byte array deposit
#define HAL_U16_DEPOSIT(a)      HAL_U16_B0(a),HAL_U16_B1(a)
#define HAL_U24_DEPOSIT(a)      HAL_U32_B0(a),HAL_U32_B1(a),HAL_U32_B2(a)
#define HAL_U32_DEPOSIT(a)      HAL_U32_B0(a),HAL_U32_B1(a),HAL_U32_B2(a),HAL_U32_B3(a)
#define HAL_U16_BE_DEPOSIT(a)   HAL_U16_B1(a),HAL_U16_B0(a)
#define HAL_U24_BE_DEPOSIT(a)   HAL_U32_B2(a),HAL_U32_B1(a),HAL_U32_B0(a)
#define HAL_U32_BE_DEPOSIT(a)   HAL_U32_B3(a),HAL_U32_B2(a),HAL_U32_B1(a),HAL_U32_B0(a)
#define HAL_U32_PUSH(p,a)       do{ \
    *(p)++ = HAL_U32_B0(a); \
    *(p)++ = HAL_U32_B1(a); \
    *(p)++ = HAL_U32_B2(a); \
    *(p)++ = HAL_U32_B3(a); } while(0)
#define HAL_U16_PUSH(p,a)       do{ \
    *(p)++ = HAL_U16_B0(a); \
    *(p)++ = HAL_U16_B1(a); } while(0)
#define HAL_U32_BE_PUSH(p,a)       do{ \
    *(p)++ = HAL_U32_B3(a); \
    *(p)++ = HAL_U32_B2(a); \
    *(p)++ = HAL_U32_B1(a); \
    *(p)++ = HAL_U32_B0(a); } while(0)
#define HAL_U16_BE_PUSH(p,a)       do{ \
    *(p)++ = HAL_U16_B1(a); \
    *(p)++ = HAL_U16_B0(a); } while(0)

// Aliases
//======================================================================
#define U32_BYTE_SWAP(a)        HAL_U32_BYTE_SWAP(a)
#define U16_BYTE_SWAP(a)        HAL_U16_BYTE_SWAP(a)
#define U16_MSB(a)              HAL_U16_B1(a)
#define U16_LSB(a)              HAL_U16_B0(a)
#define U32_MSB(a)              HAL_U32_HI(a)
#define U32_LSB(a)              HAL_U32_LO(a)
#define U16_TO_BE(a)            U16_BYTE_SWAP(a)
#define U32_TO_BE(a)            U32_BYTE_SWAP(a)

#ifndef htonl
    #define htonl(a)            U32_BYTE_SWAP(a)
#endif

#ifndef ntohl
    #define ntohl(a)            htonl((a))
#endif

#ifndef htons
    #define htons(a)            U16_BYTE_SWAP(a)
#endif

#ifndef ntohs
    #define ntohs(a)            htons((a))
#endif

# define weak_alias(name, aliasname) _weak_alias (name, aliasname)
# define _weak_alias(name, aliasname) \
  extern __typeof (name) aliasname __attribute__ ((weak, alias (#name)));

//======================================================================
// Attributes and macros
//======================================================================
#define TYPEDEF_STRUCT_PACKED   typedef struct __attribute__((packed))

#define IN_SECTION(sec)         __attribute__((section(#sec)))

#define WEAK_SYM                __attribute__((weak))

#define IN_RAM                  __attribute__((section(".data"),no_instrument_function))

#define NOINSTRUMENT            __attribute__((no_instrument_function))

#define _UNUSED                 __attribute__((unused))

#define NOINLINE                __attribute__((noinline))
#define INLINE                  static inline __attribute__((always_inline,no_instrument_function))

#define sizeofof(_s,_f)         sizeof(((_s*)0)->_f) // size of structure field
#define typeofof(_s,_f)         typeof(((_s*)0)->_f) // type of structure field

//======================================================================
// hal_callback_t
//----------------------------------------------------------------------
/// @typedef hal_callback_t
/// signal : an event enumerator
/// param  : pointer to the parameter structure or buffer
/// looks like : void callback(u16 signal, u16 plen, void* param)
//======================================================================
typedef void (*hal_callback_t)(u16, void*);

#define private static
#define protected
#define public
#define global

/// @} HAL_TYPES

#endif // __HAL_TYPES_H__