/// @copyright
/// ========================================================================={{{
/// Copyright (c) 2012-2020 WizziLab                                           /
/// All rights reserved                                                        /
///                                                                            /
/// =========================================================================}}}
/// @endcopyright

//  =======================================================================
/// @file           alp_helpers.h
/// @defgroup       ALP
/// @brief          ALP helpers functions
//  =======================================================================

#ifndef __ALP_PAYLOAD_H__
#define __ALP_PAYLOAD_H__

#include "hal_types.h"

//======================================================================
//======================================================================
//  ALP Helpers
//======================================================================
//======================================================================

// ALP payload (chained list)
// Each element contains ONE ALP OP
typedef struct _alp_payload_t {
    u32             len;
    _alp_payload_t* next;
    u8              d[1];
} alp_payload_t;

// Utils
alp_payload_t*  alp_payload_new(u32 size);
void            alp_payload_free(alp_payload_t* alp);
void            alp_payload_print_nb(void);
alp_payload_t*  alp_payload_append(alp_payload_t* alp_1, alp_payload_t* alp_2);
int             alp_payload_get_err(alp_payload_t* alp);
alp_payload_t*  alp_payload_parse(u8* p, int length);
int             alp_payload_parse_chunk(alp_payload_t* alp, alp_parsed_chunk_t* r);
void            alp_payload_print(alp_payload_t* alp);
u32             alp_payload_to_buf(alp_payload_t* alp, u8* buf, u32 max);

//======================================================================
/// @brief  Parses one OP from a payload.
///         Returns at first matching OP.
///         Let payload untouched.
/// @param  alp             alp_payload_t*  Payload to parse from
/// @param  op              u8                  OP to look for
/// @param  buffer          void*               Parsed OP
/// @return                 int                 Size of parsed OP
/// @revent                 NONE
//======================================================================
alp_payload_t*  alp_payload_get(alp_payload_t* alp, u8 op);

//======================================================================
/// @brief  Extract one OP from a payload.
///         Returns at first matching OP.
///         Removes parsed OP from the payload.
/// @param  alp             alp_payload_t**     Payload to extract from
/// @param  op              u8                  OP to look for
/// @return                 alp_payload_t*      Extracted OP
/// @revent                 NONE
//======================================================================
alp_payload_t*  alp_payload_extract(alp_payload_t** alp, u8 op);

//======================================================================
/// @brief  Parses one OP from a payload.
///         Returns at first matching OP.
///         Removes parsed OP from the payload.
///         Copy payload data into provider buffer
/// @param  alp             alp_payload_t**     Payload to extract from
/// @param  op              u8                  OP to look for
/// @param  data            void*               Parsed OP data buffer
/// @return                 bool                true if OP found else false
/// @revent                 NONE
//======================================================================
bool alp_payload_extract_data(alp_payload_t** alp, u8 op, void* data);

//======================================================================
/// @brief  Creates malloc'ed ALP payload
/// @param  alp             alp_payload_t*  Payload to append the ALP message to. A new payload will be malloc'ed if NULL
/// @param  ...
/// @return                 alp_payload_t*  New ALP payload
/// @revent                 NONE
//======================================================================
alp_payload_t* alp_payload_root_auth(    alp_payload_t* alp);
alp_payload_t* alp_payload_root_auth_enc(alp_payload_t* alp, u8* challenge);
alp_payload_t* alp_payload_root_sign(    alp_payload_t* alp, u8* iv);
alp_payload_t* alp_payload_nop(          alp_payload_t* alp, u8 group);
alp_payload_t* alp_payload_forward(      alp_payload_t* alp, void* itf);
alp_payload_t* alp_payload_tag(          alp_payload_t* alp, u8 tag);
alp_payload_t* alp_payload_rsp_tag(      alp_payload_t* alp, u8 tag, u8 eop, u8 err);
alp_payload_t* alp_payload_rsp_f_data(   alp_payload_t* alp, u8 fid, void* data, u32 offset, u32 length);
//alp_payload_t* alp_payload_rsp_f_data_fs(alp_payload_t* alp, u8 fid, u32 offset, u32 length);
alp_payload_t* alp_payload_rsp_status(   alp_payload_t* alp, u8 action, s8 status);
alp_payload_t* alp_payload_rsp_fprop(    alp_payload_t* alp, u8 fid, alp_file_header_t* hdr);
alp_payload_t* alp_payload_f_wr_data(    alp_payload_t* alp, u8 fid, void* data, u32 offset, u32 length, u8 group);
alp_payload_t* alp_payload_f_wr_data_nr( alp_payload_t* alp, u8 fid, void* data, u32 offset, u32 length, u8 group);
alp_payload_t* alp_payload_f_rd_data(    alp_payload_t* alp, u8 fid, u32 offset, u32 length, u8 group);
alp_payload_t* alp_payload_f_flush(      alp_payload_t* alp, u8 fid, u8 group);
alp_payload_t* alp_payload_f_declare(    alp_payload_t* alp, u8 fid, alp_file_header_t* hdr);
alp_payload_t* alp_payload_f_touch(      alp_payload_t* alp, u8 fid, u32 offset, u32 length, u8 group);
alp_payload_t* alp_payload_activate_itf( alp_payload_t* alp, u8 type, u8 nb_dev, u8 ifid, u8 flags, u8 enable);
alp_payload_t* alp_payload_urcc_en(      alp_payload_t* alp, u8 type, u8 ifid, u8 val);
alp_payload_t* alp_payload_query(        alp_payload_t* alp, u8 fid, void* data, u32 offset, u32 length, u8 group);
alp_payload_t* alp_payload_qbreak(       alp_payload_t* alp, u8 fid, void* data, u32 offset, u32 length, u8 group);
alp_payload_t* alp_payload_qbreak_eq(    alp_payload_t* alp, u8 fid, u32 data, u32 offset, u32 length, u8 group);
alp_payload_t* alp_payload_qbreak_ne(    alp_payload_t* alp, u8 fid, u32 data, u32 offset, u32 length, u8 group);
alp_payload_t* alp_payload_qbreak_gt(    alp_payload_t* alp, u8 fid, u32 data, u32 offset, u32 length, u8 group);
alp_payload_t* alp_payload_qbreak_gte(   alp_payload_t* alp, u8 fid, u32 data, u32 offset, u32 length, u8 group);
alp_payload_t* alp_payload_qbreak_lt(    alp_payload_t* alp, u8 fid, u32 data, u32 offset, u32 length, u8 group);
alp_payload_t* alp_payload_qbreak_lte(   alp_payload_t* alp, u8 fid, u32 data, u32 offset, u32 length, u8 group);
alp_payload_t* alp_payload_qbreak_eq_msk(alp_payload_t* alp, u8 fid, u32 mask, u32 data, u32 offset, u32 length, u8 group);


#endif // __ALP_PAYLOAD_H__