/// @copyright
/// ========================================================================={{{
/// Copyright (c) 2012-2020 WizziLab                                           /
/// All rights reserved                                                        /
///                                                                            /
/// Should you have any questions regarding your right to use this Software,   /
/// contact WizziLab at www.wizzilab.com.                                      /
/// =========================================================================}}}
/// @endcopyright

//  =======================================================================
/// @file           alp.h
/// @brief          Wizzilab's specific ALP definitions
/// @defgroup       ALP
//  =======================================================================

#ifndef __ALP_H__
#define __ALP_H__

// D7A spec
#include "alp_spec.h"
// helpers
#include "alp_helpers.h"
#include "alp_payload.h"

// D7AActP Substitution modes:
// ----------------------------------------------------------------------
// MODE1: replace FID= actp.fid, offset=0,             length=(actp.fid).length
// MODE2: replace FID= actp.fid, offset=actp.offset,   length=actp.length
// ======================================================================
typedef enum {
    ACTP_SUBST_NONE = 0,
    ACTP_SUBST_MODE1,
    ACTP_SUBST_MODE2,
} alp_actp_substitution_mode_t;

typedef union {
    struct {
        u8 fid;
        u8 error;
    };
} alp_evt_done_evp_t;

typedef enum {
    ALP_ITF_FID_COM0,
    ALP_ITF_FID_COM1,
    ALP_ITF_FID_COM2,
    ALP_ITF_FID_COM_HACK_LIMIT = ALP_ITF_FID_COM2,
    ALP_ITF_FID_BLE,
} alp_itf_fixed_fids_t;

typedef enum {
    ALP_ROLE_ROOT       = 0xFF,
    ALP_ROLE_USER       = 0x38,
    ALP_ROLE_GUEST      = 0x07,
} alp_role_t;

typedef enum {
    ALP_RPOL_ONESHOT        = 0,
    ALP_RPOL_ONESHOT_RETRY  = 1,
    ALP_RPOL_FIFO_FAST      = 2,
    ALP_RPOL_FIFO_SLOW      = 3,
    ALP_RPOL_SINGLE_FAST    = 4,
    ALP_RPOL_SINGLE_SLOW    = 5,
    ALP_RPOL_ONESHOT_STICKY = 6,
    ALP_RPOL_RFU_7          = 7,
    ALP_RPOL_QTY,
} alp_rpol_t;

typedef enum {
    ALP_BOFF_NONE           = 0,
    ALP_BOFF_LINEAR         = 1,
    ALP_BOFF_EXP            = 2,

    ALP_BOFF_QTY
} alp_rpol_backoff_proc_t;


// Event parameter for ALP_EVT_URC
typedef union {
    struct {
        u8 type;
        u8 ifid;
        u8 value;
    };
} alp_urc_evp_t;

TYPEDEF_STRUCT_PACKED {
    u8 ifid;
    u8 report_freq;
} alp_urc_lqual_params_t;

TYPEDEF_STRUCT_PACKED {
    u8 ifid;
    u8 max_consecutive_errors;
} alp_urc_ldown_params_t;

typedef enum {
    /// Unsollicited message
    ALP_MSG_UNS,
    /// Deprecated
    ALP_MSG_ERR,
    /// Command message XXX deprecated BUT (mis)used in ALP Interfaces.
    ALP_MSG_CMD,
    /// Response message
    ALP_MSG_RES,
    /// Interface Termination message (used for TAG EOP handing)
    ALP_MSG_DONE,
} alp_msg_t;


//======================================================================
//======================================================================
//  D7A Interface
//======================================================================
//======================================================================

#include "d7a_1x.h"

//======================================================================
// alp_itf_d7a_cfg_t
//----------------------------------------------------------------------
// @brief ALP D7A interface configuration structure
//======================================================================
TYPEDEF_STRUCT_PACKED
{
    u8 type;
    d7a_sp_cfg_t cfg;

} alp_itf_d7a_cfg_t;

//======================================================================
// alp_itf_lwan_cfg_t
//----------------------------------------------------------------------
// @brief ALP LoRaWAN interface configuration structure
//======================================================================
TYPEDEF_STRUCT_PACKED
{
    u8 type;

} alp_itf_lwan_cfg_t;

//======================================================================
// alp_itf_com_cfg_t
//----------------------------------------------------------------------
// @brief ALP COM interface configuration structure
//======================================================================
TYPEDEF_STRUCT_PACKED
{
    u8 type;
    u8 com;

} alp_itf_com_cfg_t;

//======================================================================
// alp_itf_ble_cfg_t
//----------------------------------------------------------------------
// @brief ALP BLE (dummy) interface configuration structure
//======================================================================
TYPEDEF_STRUCT_PACKED
{
    u8 type;

} alp_itf_ble_cfg_t;

// XXX When declaring an interface file, use this size.
#define ALP_ITF_MAX_SIZE    (sizeof(alp_itf_d7a_cfg_t)) // Biggest is d7a itf for now

//======================================================================
// alp_itf_cfg_t
//----------------------------------------------------------------------
// @brief ALP Generic interface configuration structure
//======================================================================
TYPEDEF_STRUCT_PACKED
{
    u8 type;

} alp_itf_cfg_t;


//======================================================================
// @brief ALP Generic OP structures
//======================================================================
TYPEDEF_STRUCT_PACKED {
    u8 op;
    u8 id;
} alp_op_tag_t;

TYPEDEF_STRUCT_PACKED {
    u8 op;
    u8 type;
    u8 length; // D7 istat length is < 64
    d7a_sp_res_t istat;
} alp_op_istat_d7_t;

TYPEDEF_STRUCT_PACKED {
    u8 op;
    u8 type;
    u8 length; // Always 1
    s8 err;
} alp_op_eopistat_t;

TYPEDEF_STRUCT_PACKED {
    u8 op;
    u8 id; // Action ID
    s8 err;
} alp_op_status_t;

TYPEDEF_STRUCT_PACKED {
    u8 op;
    u8 fid;
    u8 offset; // offset is < 64
    u8 length; // payload length is < 64
    u8 data[63];
} alp_op_rsp_f_data_t;

#define ALP_D7A_ISTAT_NONE  0
#define ALP_D7A_ISTAT_UNS   (1<<0)
#define ALP_D7A_ISTAT_RESP  (1<<1)
#define ALP_D7A_ISTAT_EOP   (1<<2)


//======================================================================
// alp_itf_d7a_cfg_size
//----------------------------------------------------------------------
/// @brief  Get size of the D7A interface configuration
/// @param  cfg pointer to the configuration
/// @return int configuration size
//======================================================================
_public int alp_itf_d7a_cfg_size(u8* cfg);

//======================================================================
// alp_itf_size
//----------------------------------------------------------------------
/// @brief  Get size of the interface configuration
/// @param  itf pointer to the configuration
/// @return int configuration size
//======================================================================
_public int alp_itf_size(void* itf);


#endif // __ALP_H__