/// @copyright
/// ========================================================================={{{
/// Copyright (c) 2017 WizziLab                                                /
/// All rights reserved                                                        /
///                                                                            /
/// IMPORTANT: This Software may not be modified, copied or distributed unless /
/// embedded on a WizziLab product. Other than for the foregoing purpose, this /
/// Software and/or its documentation may not be used, reproduced, copied,     /
/// prepared derivative works of, modified, performed, distributed, displayed  /
/// or sold for any purpose. For the sole purpose of embedding this Software   /
/// on a WizziLab product, copy, modification and distribution of this         /
/// Software is granted provided that the following conditions are respected:  /
///                                                                            /
/// *  Redistributions of source code must retain the above copyright notice,  /
///    this list of conditions and the following disclaimer                    /
///                                                                            /
/// *  Redistributions in binary form must reproduce the above copyright       /
///    notice, this list of conditions and the following disclaimer in the     /
///    documentation and/or other materials provided with the distribution.    /
///                                                                            /
/// *  The name of WizziLab can not be used to endorse or promote products     /
///    derived from this software without specific prior written permission.   /
///                                                                            /
/// THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS        /
/// "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED  /
/// TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR /
/// PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR          /
/// CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,      /
/// EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,        /
/// PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA,            /
/// OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY     /
/// OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING    /
/// NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS         /
/// SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.               /
/// WIZZILAB HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT, UPDATES,       /
/// ENHANCEMENTS OR MODIFICATIONS.                                             /
///                                                                            /
/// Should you have any questions regarding your right to use this Software,   /
/// contact WizziLab at www.wizzilab.com.                                      /
///                                                                            /
/// =========================================================================}}}
/// @endcopyright
///
/// =======================================================================
///
/// @file           kal_crypto.c
/// @brief          Crypto Utilities
///
/// =======================================================================

#include "WizziDebug.h"
#include "kal_crypto.h"

// ======================================================================
//
//
//                        SHA-2 256 Tool-suite. 
//          (From Brad Conte's Licence-free implementation)
//
//
// ======================================================================

// DBL_INT_ADD treats two unsigned ints a and b as one 64-bit integer and adds c to it
#define DBL_INT_ADD(a,b,c) if (a > 0xffffffff - (c)) ++b; a += c;
#define ROTLEFT(a,b) (((a) << (b)) | ((a) >> (32-(b))))
#define ROTRIGHT(a,b) (((a) >> (b)) | ((a) << (32-(b))))

#define CH(x,y,z) (((x) & (y)) ^ (~(x) & (z)))
#define MAJ(x,y,z) (((x) & (y)) ^ ((x) & (z)) ^ ((y) & (z)))
#define EP0(x) (ROTRIGHT(x,2) ^ ROTRIGHT(x,13) ^ ROTRIGHT(x,22))
#define EP1(x) (ROTRIGHT(x,6) ^ ROTRIGHT(x,11) ^ ROTRIGHT(x,25))
#define SIG0(x) (ROTRIGHT(x,7) ^ ROTRIGHT(x,18) ^ ((x) >> 3))
#define SIG1(x) (ROTRIGHT(x,17) ^ ROTRIGHT(x,19) ^ ((x) >> 10))

// =======================================================================
// kal_sha256_ctx_t
// -----------------------------------------------------------------------
/// SHA256 context
// =======================================================================
typedef struct 
{
    u8 data[64];
    uint datalen;
    uint bitlen[2];
    uint state[8];

} kal_sha256_ctx_t;

kal_sha256_ctx_t* g_kal_sha_ctx = (kal_sha256_ctx_t*)NULL;

//======================================================================
// k_kal_sha
//----------------------------------------------------------------------
/// @brief Unique constant table used for SHA256 
//======================================================================
const uint k_kal_sha[64] = 
{
   0x428a2f98,0x71374491,0xb5c0fbcf,0xe9b5dba5,0x3956c25b,0x59f111f1,0x923f82a4,0xab1c5ed5,
   0xd807aa98,0x12835b01,0x243185be,0x550c7dc3,0x72be5d74,0x80deb1fe,0x9bdc06a7,0xc19bf174,
   0xe49b69c1,0xefbe4786,0x0fc19dc6,0x240ca1cc,0x2de92c6f,0x4a7484aa,0x5cb0a9dc,0x76f988da,
   0x983e5152,0xa831c66d,0xb00327c8,0xbf597fc7,0xc6e00bf3,0xd5a79147,0x06ca6351,0x14292967,
   0x27b70a85,0x2e1b2138,0x4d2c6dfc,0x53380d13,0x650a7354,0x766a0abb,0x81c2c92e,0x92722c85,
   0xa2bfe8a1,0xa81a664b,0xc24b8b70,0xc76c51a3,0xd192e819,0xd6990624,0xf40e3585,0x106aa070,
   0x19a4c116,0x1e376c08,0x2748774c,0x34b0bcb5,0x391c0cb3,0x4ed8aa4a,0x5b9cca4f,0x682e6ff3,
   0x748f82ee,0x78a5636f,0x84c87814,0x8cc70208,0x90befffa,0xa4506ceb,0xbef9a3f7,0xc67178f2
};

//======================================================================
// kal_sha256_init
//----------------------------------------------------------------------
/// @brief Initialize an SHA Hash generation. To be called before any
///        other kal_sha256_update/final functions.
//======================================================================
void kal_sha256_init(void)
{
    // Garbage collection
    if (g_kal_sha_ctx) FREE(g_kal_sha_ctx);
    g_kal_sha_ctx = (kal_sha256_ctx_t*) MALLOC(sizeof(kal_sha256_ctx_t));
    g_kal_sha_ctx->datalen = 0;
    g_kal_sha_ctx->bitlen[0] = 0;
    g_kal_sha_ctx->bitlen[1] = 0;
    g_kal_sha_ctx->state[0] = 0x6a09e667;
    g_kal_sha_ctx->state[1] = 0xbb67ae85;
    g_kal_sha_ctx->state[2] = 0x3c6ef372;
    g_kal_sha_ctx->state[3] = 0xa54ff53a;
    g_kal_sha_ctx->state[4] = 0x510e527f;
    g_kal_sha_ctx->state[5] = 0x9b05688c;
    g_kal_sha_ctx->state[6] = 0x1f83d9ab;
    g_kal_sha_ctx->state[7] = 0x5be0cd19;
}

//======================================================================
// kal_sha256_transform
//----------------------------------------------------------------------
/// @brief SHA core hashing function.
/// @param data stream (awaits 64 valid bytes)
//======================================================================
static void kal_sha256_transform(u8 data[])
{
    uint a,b,c,d,e,f,g,h,i,j,t1,t2,m[64];

    for (i=0,j=0; i < 16; ++i, j += 4)
        m[i] = (data[j] << 24) | (data[j+1] << 16) | (data[j+2] << 8) | (data[j+3]);
    for ( ; i < 64; ++i)
        m[i] = SIG1(m[i-2]) + m[i-7] + SIG0(m[i-15]) + m[i-16];

    a = g_kal_sha_ctx->state[0];
    b = g_kal_sha_ctx->state[1];
    c = g_kal_sha_ctx->state[2];
    d = g_kal_sha_ctx->state[3];
    e = g_kal_sha_ctx->state[4];
    f = g_kal_sha_ctx->state[5];
    g = g_kal_sha_ctx->state[6];
    h = g_kal_sha_ctx->state[7];

    for (i = 0; i < 64; ++i) {
        t1 = h + EP1(e) + CH(e,f,g) + k_kal_sha[i] + m[i];
        t2 = EP0(a) + MAJ(a,b,c);
        h = g;
        g = f;
        f = e;
        e = d + t1;
        d = c;
        c = b;
        b = a;
        a = t1 + t2;
    }

    g_kal_sha_ctx->state[0] += a;
    g_kal_sha_ctx->state[1] += b;
    g_kal_sha_ctx->state[2] += c;
    g_kal_sha_ctx->state[3] += d;
    g_kal_sha_ctx->state[4] += e;
    g_kal_sha_ctx->state[5] += f;
    g_kal_sha_ctx->state[6] += g;
    g_kal_sha_ctx->state[7] += h;
}

//======================================================================
// kal_sha256_update
//----------------------------------------------------------------------
/// @brief Used to 'push' new data into the hash calculation.
/// @param data pointer to the (char) data stream to hash.
/// @param len data stream length in bytes
//======================================================================
void kal_sha256_update(u8 data[], uint len)
{
   uint i;

   for (i=0; i < len; ++i) {
      g_kal_sha_ctx->data[g_kal_sha_ctx->datalen] = data[i];
      g_kal_sha_ctx->datalen++;
      if (g_kal_sha_ctx->datalen == 64) {
         kal_sha256_transform(g_kal_sha_ctx->data);
         DBL_INT_ADD(g_kal_sha_ctx->bitlen[0],g_kal_sha_ctx->bitlen[1],512);
         g_kal_sha_ctx->datalen = 0;
      }
   }
}

//======================================================================
// kal_sha256_final
//----------------------------------------------------------------------
/// @brief To be called when all data has been pushed into the hash
///        generator. Finalize and outputs resulting SHA hash.
/// @param hash Pointer to the Output (char) buffer. Fills 32-bytes.
//======================================================================
void kal_sha256_final(u8 hash[])
{
    uint i;

    i = g_kal_sha_ctx->datalen;

    // Pad whatever data is left in the buffer.
    if (g_kal_sha_ctx->datalen < 56) {
        g_kal_sha_ctx->data[i++] = 0x80;
        while (i < 56)
            g_kal_sha_ctx->data[i++] = 0x00;
    }
    else {
        g_kal_sha_ctx->data[i++] = 0x80;
        while (i < 64)
            g_kal_sha_ctx->data[i++] = 0x00;
        kal_sha256_transform(g_kal_sha_ctx->data);
        memset(g_kal_sha_ctx->data,0,56);
    }

    // Append to the padding the total message's length in bits and transform.
    DBL_INT_ADD(g_kal_sha_ctx->bitlen[0],g_kal_sha_ctx->bitlen[1],g_kal_sha_ctx->datalen * 8);
    g_kal_sha_ctx->data[63] = g_kal_sha_ctx->bitlen[0];
    g_kal_sha_ctx->data[62] = g_kal_sha_ctx->bitlen[0] >> 8;
    g_kal_sha_ctx->data[61] = g_kal_sha_ctx->bitlen[0] >> 16;
    g_kal_sha_ctx->data[60] = g_kal_sha_ctx->bitlen[0] >> 24;
    g_kal_sha_ctx->data[59] = g_kal_sha_ctx->bitlen[1];
    g_kal_sha_ctx->data[58] = g_kal_sha_ctx->bitlen[1] >> 8;
    g_kal_sha_ctx->data[57] = g_kal_sha_ctx->bitlen[1] >> 16;
    g_kal_sha_ctx->data[56] = g_kal_sha_ctx->bitlen[1] >> 24;
    kal_sha256_transform(g_kal_sha_ctx->data);

    // Since this implementation uses little endian byte ordering and SHA uses big endian,
    // reverse all the bytes when copying the final state to the output hash.
    for (i=0; i < 4; ++i) {
        hash[i]    = (g_kal_sha_ctx->state[0] >> (24-i*8)) & 0x000000ff;
        hash[i+4]  = (g_kal_sha_ctx->state[1] >> (24-i*8)) & 0x000000ff;
        hash[i+8]  = (g_kal_sha_ctx->state[2] >> (24-i*8)) & 0x000000ff;
        hash[i+12] = (g_kal_sha_ctx->state[3] >> (24-i*8)) & 0x000000ff;
        hash[i+16] = (g_kal_sha_ctx->state[4] >> (24-i*8)) & 0x000000ff;
        hash[i+20] = (g_kal_sha_ctx->state[5] >> (24-i*8)) & 0x000000ff;
        hash[i+24] = (g_kal_sha_ctx->state[6] >> (24-i*8)) & 0x000000ff;
        hash[i+28] = (g_kal_sha_ctx->state[7] >> (24-i*8)) & 0x000000ff;
    }
    FREE(g_kal_sha_ctx);
    g_kal_sha_ctx = (kal_sha256_ctx_t*)NULL;
}