/// @copyright
/// ========================================================================={{{
/// Copyright (c) 20XX                                                         /
/// All rights reserved                                                        /
///                                                                            /
/// IMPORTANT: This Software may not be modified, copied or distributed unless /
/// embedded on a WizziLab product. Other than for the foregoing purpose, this /
/// Software and/or its documentation may not be used, reproduced, copied,     /
/// prepared derivative works of, modified, performed, distributed, displayed  /
/// or sold for any purpose. For the sole purpose of embedding this Software   /
/// on a WizziLab product, copy, modification and distribution of this         /
/// Software is granted provided that the following conditions are respected:  /
///                                                                            /
/// *  Redistributions of source code must retain the above copyright notice,  /
///    this list of conditions and the following disclaimer                    /
///                                                                            /
/// *  Redistributions in binary form must reproduce the above copyright       /
///    notice, this list of conditions and the following disclaimer in the     /
///    documentation and/or other materials provided with the distribution.    /
///                                                                            /
/// *  The name of WizziLab can not be used to endorse or promote products     /
///    derived from this software without specific prior written permission.   /
///                                                                            /
/// THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS        /
/// "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED  /
/// TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR /
/// PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR          /
/// CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,      /
/// EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,        /
/// PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA,            /
/// OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY     /
/// OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING    /
/// NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS         /
/// SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.               /
/// WIZZILAB HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT, UPDATES,       /
/// ENHANCEMENTS OR MODIFICATIONS.                                             /
///                                                                            /
/// Should you have any questions regarding your right to use this Software,   /
/// contact WizziLab at www.wizzilab.com.                                      /
///                                                                            /
/// =========================================================================}}}
/// @endcopyright

//  =======================================================================
/// @file           kal_codec.h
/// @brief          KAL utilities s
//  =======================================================================

#ifndef _KAL_CODEC_H_
#define _KAL_CODEC_H_

#include "hal_types.h"

// ======================================================================
//
//
//                  ASCII to Binary Codec Toolkits 
//
//
// ======================================================================

//======================================================================
// hex2ascii
//----------------------------------------------------------------------
/// @brief  Convert hexadecimal number to ascii
/// @param  char* in        input buffer (hex)
/// @param  char* out      output buffer (ascii)
/// @param  u16 len         length of inoput buffer in bytes
//======================================================================
void hex2ascii( char* in, char* out, u16 length);

//======================================================================
// itoa
//----------------------------------------------------------------------
/// @brief  Converts an integer value to a null-terminated string using the specified base and stores the result
/// in the array given by result parameter.
/// @param  int value       integer to convert
/// @param  char* result    converted string
/// @param  int base        base used for conversion 2, 8, 10, 16
/// @return A pointer to the resulting null-terminated string, same as parameter result
//======================================================================
char* itoa(int value, char* result, int base);

//======================================================================
// atoitok
//----------------------------------------------------------------------
/// @brief  Extracts from a ASCII string a value (decimal or hex) and
///         convert it to an integer value.
///         - Blank characters are skipped
///         - hexa values must begin with '0x' or '0X'
///         - '-' sign is handled
///         - returns after one value has been parsed or null character
///           or CR character has been found.
/// @param  p pointer to pointer to the string to be parsed. The pointer
///         to string is modified: at the end of the call it points to
///         the character following the last parsed one.
//======================================================================
int atoitok(u8 **p);

//======================================================================
// kal_atoi
//----------------------------------------------------------------------
/// @brief  Extracts from a ASCII string a decimal value and
///         convert it to an integer value.
///         '-' sign is handled
/// @param  s               char*                   string to convert
/// @retval                 s32                     integer value
//======================================================================
s32 kal_atoi(u8* s);

//======================================================================
// kal_atoi_float
//----------------------------------------------------------------------
/// @brief  Extracts from a ASCII string a float value and
///         convert it to an integer value * 10^number_of_decimals.
///         '-' and '+' signs are handled
/// @param  s               char*                   string to convert
/// @param  d               u8                      Number of decimals
/// @retval                 s32                     integer value
//======================================================================
s32 kal_atoi_float(u8* s, u8 d);

//======================================================================
// kal_atoi_hex
//----------------------------------------------------------------------
/// @brief  Extracts from a ASCII string a hex value and
///         convert it to an integer value.
/// @param  s               char*                   string to convert
/// @retval                 u32                     integer value
//======================================================================
u32 kal_atoi_hex(u8* s);

/// Length of the Base64-encoded string based on the input binary length.
#define KAL_BASE64_ENCODE_LEN(l)        (((((l) + 2) / 3) * 4) + 1)

/// Length of the binary based on the real Base64-encoded string length.
#define KAL_BASE64_DECODE_LEN(l)        ((l) * 3 / 4)

//======================================================================
// kal_base64_strlen
//----------------------------------------------------------------------
/// @brief  Real length (excluding padding) of Base64-encoded string
///         Note : the input string always has 4x chars, if not return 0
/// @param  in              char*                   string to decode
/// @retval                 u16                     string length
//======================================================================
u16 kal_base64_strlen(const char* in);

//======================================================================
// kal_base64_encode
//----------------------------------------------------------------------
/// @brief  Encode Base64-encoded buffer. The output buffer is supposed to have enough space
///         Beware, the stream encoding is Big Endian.
/// @param  out             char*                   string result
/// @param  in              u8*                     binary buffer to encode
/// @param  len             u16                     length of the binary buffer in bytes
/// @retval                 void 
//======================================================================
void kal_base64_encode(char *out, u8* in, u16 len);

//======================================================================
// kal_base64_decode
//----------------------------------------------------------------------
/// @brief  Decode Base64-encoded buffer. The output buffer is supposed to have enough space
/// @param  out             u8*                     binary buffer result
/// @param  in              char*                   string to decode
/// @param  len             u16                     string length, excluding padding
/// @retval                 void 
//======================================================================
void kal_base64_decode(u8 *out, const char* in, u16 len);

//======================================================================
// kal_tolower
//----------------------------------------------------------------------
/// @brief  Changes inplace to lower character a non-constant string
/// @param  s               u8*                     String to transform
/// @retval                 void
//======================================================================
void kal_tolower(u8* s);

//======================================================================
// kal_toupper
//----------------------------------------------------------------------
/// @brief  Changes inplace to upper character a non-constant string
/// @param  s               u8*                     String to transform
/// @retval                 void
//======================================================================
void kal_toupper(u8* s);

//======================================================================
// kal_crc8
//----------------------------------------------------------------------
/// @brief  Generate crc8
/// @param  in          u8*                     input buffer
/// @param  len         u32                     input buffer length in bytes
/// @retval             void
//======================================================================
u8 kal_crc8(u8* in, u32 len);

// =======================================================================
// kal_ctf_t
// -----------------------------------------------------------------------
/// D7A compressed time format
// =======================================================================
typedef union
{
    // bit access fields
    struct {
        /// Mantissa
        u8 mant : 5;
        /// Exponent
        u8 exp  : 3;
    } bf;

    // byte access
    u8 byte;

} kal_ctf_t;

//======================================================================
// kal_ctf_encode
//----------------------------------------------------------------------
/// @brief  Compress u32 to D7A Compressed Time format (CTF).
///         The ceil flag is used to define rounding, so that 
///         kal_ctf_decode(kal_ctf_encode(val, FALSE) <= val (floor)
///         kal_ctf_decode(kal_ctf_encode(val, TRUE)  >= val (ceiling)
/// @param  val         u32                     value to encode
/// @param  ceil        u8                      ceil value when TRUE
/// @retval             kal_ctf_t               compressed value 
//======================================================================
kal_ctf_t kal_ctf_encode(u32 val, u8 ceil);

// =======================================================================
// kal_ctf_decode
// -----------------------------------------------------------------------
/// @brief  Decompress from Compressed Time Format to u32
/// @param  ctf         kal_ctf_t           compressed value in CTF
/// @retval             u32                 decode result
// =======================================================================
u32 kal_ctf_decode(kal_ctf_t ctf);

#endif // _KAL_CODEC_H_

