/// ======================================================================
///
/// Copyright (C) 20XX WizziLab
/// All Rights Reserved
///
/// =======================================================================
///
/// @file           d7a_fs_1x.h
/// @brief          D7A 1.x File System definitions
/// @defgroup D7A   D7A 1.x File System definitions
/// @{
/// =======================================================================

#ifndef __D7A_1X_FS_H__
#define __D7A_1X_FS_H__

#include "d7a_1x.h"

// =======================================================================
// d7a_fs_id_t
// -----------------------------------------------------------------------
/// D7A 1.x File System Identifiers (need to be defs for auto linker generation)
// =======================================================================
#define D7A_FID_UID                    0
#define D7A_FID_FACTORY_SETTINGS       1
#define D7A_FID_FIRMWARE_VERSION       2
#define D7A_FID_DEVICE_CAPACITY        3
#define D7A_FID_DEVICE_STATUS          4
#define D7A_FID_ENG                    5
#define D7A_FID_VID                    6
#define D7A_FID_RFU_07                 7
#define D7A_FID_PHY_CFG                8
#define D7A_FID_PHY_STATUS             9
#define D7A_FID_DLL_CFG               10
#define D7A_FID_DLL_STATUS            11
#define D7A_FID_RFU_12                12
#define D7A_FID_NWL_SECURITY          13
#define D7A_FID_NWL_KEY               14
#define D7A_FID_NWL_SSR               15
#define D7A_FID_NWL_STATUS            16
#define D7A_FID_TRL_STATUS            17
#define D7A_FID_SEL_CFG               18
#define D7A_FID_SEL_STATUS            19
#define D7A_FID_RFU_20                20
#define D7A_FID_RFU_21                21
#define D7A_FID_RFU_22                22
#define D7A_FID_LOCATION_DATA         23
#define D7A_FID_ROOT_KEY              24
#define D7A_FID_USER_KEY              25
#define D7A_FID_CHALLENGE             26
#define D7A_FID_SENSOR_DESCRIPTION    27
#define D7A_FID_RTC                   28
#define D7A_FID_TIME_STAMP            29
#define D7A_FID_RFU_30                30
#define D7A_FID_RFU_31                31
#define D7A_FID_ACCESS_PROFILE_0      32
#define D7A_FID_ACCESS_PROFILE_1      33
#define D7A_FID_ACCESS_PROFILE_2      34
#define D7A_FID_ACCESS_PROFILE_3      35
#define D7A_FID_ACCESS_PROFILE_4      36
#define D7A_FID_ACCESS_PROFILE_5      37
#define D7A_FID_ACCESS_PROFILE_6      38
#define D7A_FID_ACCESS_PROFILE_7      39
#define D7A_FID_ACCESS_PROFILE_8      40
#define D7A_FID_ACCESS_PROFILE_9      41
#define D7A_FID_ACCESS_PROFILE_A      42
#define D7A_FID_ACCESS_PROFILE_B      43
#define D7A_FID_ACCESS_PROFILE_C      44
#define D7A_FID_ACCESS_PROFILE_D      45
#define D7A_FID_ACCESS_PROFILE_E      46
#define D7A_FID_ACCESS_PROFILE_F      47
#define D7A_FID_RFU_48                48
#define D7A_FID_RFU_49                49
#define D7A_FID_RFU_50                50
#define D7A_FID_RFU_51                51
#define D7A_FID_RFU_52                52
#define D7A_FID_RFU_53                53
#define D7A_FID_RFU_54                54
#define D7A_FID_RFU_55                55
#define D7A_FID_RFU_56                56
#define D7A_FID_RFU_57                57
#define D7A_FID_RFU_58                58
#define D7A_FID_RFU_59                59
#define D7A_FID_RFU_60                60
#define D7A_FID_RFU_61                61
#define D7A_FID_RFU_62                62
#define D7A_FID_RFU_63                63


// =======================================================================
/// Fixed file sizes
// =======================================================================
#define D7A_FS_ROOT_KEY_SIZE            16
#define D7A_FS_USER_KEY_SIZE            16

// =======================================================================
/// FILE ACCESS TYPES
// =======================================================================
#define D7A_FS_OPENBAR                  0b11110110
#define D7A_FS_STANDARD                 0b11110100
#define D7A_FS_READ_ONLY                0b11100100
#define D7A_FS_USER_RO                  0b11100000
#define D7A_FS_ROOT                     0b11000000
#define D7A_FS_ROOT_RO                  0b10000000
#define D7A_FS_ROOT_WO                  0b01000000

/// =======================================================================
/// =======================================================================
///
/// 0. UID
///
/// =======================================================================
/// =======================================================================

// =======================================================================
// d7a_uid_t
// -----------------------------------------------------------------------
/// UID File structure
// =======================================================================
TYPEDEF_STRUCT_PACKED
{
    /// UID is a u64 in big endian
    u8 uid[D7A_UID_LEN];

} d7a_uid_t;
#define D7A_FS_UID_SIZE                 sizeof(d7a_uid_t)

/// =======================================================================
/// =======================================================================
///
/// 1. FACTORY SETTINGS
///
/// =======================================================================
/// =======================================================================

// =======================================================================
// d7a_factory_settings_t
// -----------------------------------------------------------------------
/// Factory Settingd File structure
// =======================================================================
TYPEDEF_STRUCT_PACKED
{
    /// Calibrated frequency offset to be applied on the central frequency
    s16 fof;
    /// Reception Insertion Loss (dB)
    s8  iloss_rx;
    /// Transmission Insertion Loss (dB)
    s8  iloss_tx;
    /// Power supply voltage in V * 10
    u8  vdd;

} d7a_factory_settings_t;
#define D7A_FS_FACTORY_SETTINGS_SIZE        sizeof(d7a_factory_settings_t)

/// =======================================================================
/// =======================================================================
///
/// 3. DEVICE CAPACITY
///
/// =======================================================================
/// =======================================================================

// =======================================================================
// d7a_device_t
// -----------------------------------------------------------------------
/// Enumerator of the device types
// =======================================================================
typedef enum
{
    /// Endpoint
    D7A_DEVICE_EP = 0,
    /// Subcontroller
    D7A_DEVICE_SC,
    /// Gateway
    D7A_DEVICE_GW,
    /// Quantity
    D7A_DEVICE_QTY,

} d7a_device_t;

#define D7A_FS_BMP(b)                   (1 << (b))
// =======================================================================
// d7a_device_capacity_t
// -----------------------------------------------------------------------
/// Device Capacity File structure
// =======================================================================
TYPEDEF_STRUCT_PACKED
{
    /// Supported ISM bands bitmap
    u8  ism_bands_bmp;
    /// Supported Channel Classes and Coding Schemes bitmap
    u8  ch_class_cs_bmp;
    /// Device type
    u8  device_type;
    /// Supported Network Layer Security Methods
    u16 nls_supported;
    /// Deprecated : Active Network Layer Security Methods
    u16 rfu;
    /// Min supported EIRP of the device in dBm
    s8  eirp_min;
    /// Max supported EIRP of the device in dBm
    s8  eirp_max;
    /// Total D7A File System Volatile memory
    u32 ram_total;
    /// Total D7A File System Permanent memory
    u32 flash_total;
    /// Maximum number of PHY inputs (>= 1)
    u16 miso_nb_max;

} d7a_device_capacity_t;
#define D7A_FS_DEVICE_CAPACITY_SIZE     sizeof(d7a_device_capacity_t)

/// =======================================================================
/// =======================================================================
///
/// 4. DEVICE STATUS
///
/// =======================================================================
/// =======================================================================

// =======================================================================
// d7a_device_status_t
// -----------------------------------------------------------------------
/// Device Status File structure
// =======================================================================
TYPEDEF_STRUCT_PACKED
{
    /// Error Code Bitmap
    u8 error_code_bmp;
    /// Available D7A File System Volatile memory
    u32 ram_available;
    /// Available D7A File System Permanent memory
    u32 flash_available;

} d7a_device_status_t;
#define D7A_FS_DEVICE_STATUS_SIZE       sizeof(d7a_device_status_t)

/// =======================================================================
/// =======================================================================
///
/// 5. ENG
///
/// =======================================================================
/// =======================================================================

// =======================================================================
// d7a_eng_mode_t
// -----------------------------------------------------------------------
/// Enumerator of the Engineering modes
// =======================================================================
typedef enum
{
    /// Normal Mode
    /// param : not used
    D7A_ENG_MODE_NORMAL         = 0,
    /// Spectrum Control Protocol flooding
    /// param[0] : access class
    /// param[1] : flooding duration in seconds (CTF)
    D7A_ENG_MODE_SCP            = 2,
    /// Beacon Test Mode. Discard duty and CSMA-CA
    /// param : not used
    D7A_ENG_MODE_BEACON         = 3,
    /// Sensitivity Test Mode. Fixed NF/ECCAO sensotivity is set according to the parameters
    /// param[0] : Noise Floor / or ECCAO
    /// param[1] : param[0] is, when O = NF, when 1 = ECCAO
    D7A_ENG_MODE_SENSITIVITY    = 4,
} d7a_eng_mode_t;

// =======================================================================
// d7a_eng_t
// -----------------------------------------------------------------------
/// Engineering File structure
// =======================================================================
TYPEDEF_STRUCT_PACKED
{
    /// Run/Running
    u8 run;
    /// Engineering Mode
    u8 mode;
    /// Parameter, depends on the mode (see enumerator)
    u8 param[2];

} d7a_eng_t;
#define D7A_FS_ENG_SIZE       sizeof(d7a_eng_t)

/// =======================================================================
/// =======================================================================
///
/// 6. VID
///
/// =======================================================================
/// =======================================================================

//======================================================================
/// VID Control
//======================================================================
typedef union
{
    struct {
        /// BTAG is a 6-bit address tag passed in the background broadcast frames
        u8 btag     : 6;
        /// Btag acceptance condition (0 : differ, 1 : equal)
        u8 bteq     : 1;
        /// Declare myself using VID instead of UID
        u8 en      :  1;
    } bf;

    /// byte access
    u8 byte;

} d7a_vid_ctrl_t;

// =======================================================================
// d7a_vid_t
// -----------------------------------------------------------------------
/// VID File structure
// =======================================================================
TYPEDEF_STRUCT_PACKED
{
    /// VID is a u16 in big endian
    u8 vid[D7A_VID_LEN];
    /// VID control fields
    d7a_vid_ctrl_t ctrl;

} d7a_vid_t;
#define D7A_FS_VID_SIZE                 sizeof(d7a_vid_t)

/// =======================================================================
/// =======================================================================
///
/// 8. PHY CONFIGURATION
///
/// =======================================================================
/// =======================================================================

// =======================================================================
// d7a_ext_t
// -----------------------------------------------------------------------
/// Enumerator of the PHY channel extensions
// =======================================================================
typedef enum
{
    /// Nominal D7A channels (FSK modulation)
    D7A_EXT_FSK = 0,
    /// Other D7A channels
    D7A_EXT_OTHER = 1,

} d7a_ext_t;

// =======================================================================
// d7a_class_t
// -----------------------------------------------------------------------
/// Enumerator of the nominal PHY channel classes
// =======================================================================
typedef enum
{
    /// Lo-Rate, 9.6 kbps, 4.8 kHz deviation
    D7A_CLASS_LO = 0,
    /// Reserved
    D7A_CLASS_RFU,
    /// Normal-Rate, 55.555 kbps, 50 kHz deviation
    D7A_CLASS_NORMAL,
    /// Hi-Rate, 166.667 kbps, 41.667 kHz deviation
    D7A_CLASS_HI,
    /// Quantity
    D7A_CLASS_QTY,

} d7a_class_t;

// =======================================================================
// d7a_class_ext_t
// -----------------------------------------------------------------------
/// Enumerator of the Extended channel classes
// =======================================================================
typedef enum
{
    /// LoRa modulation
    D7A_CLASS_EXT_LORA = 0,
    /// Reserved
    D7A_CLASS_EXT_RFU_1,
    /// Sigfox / UNB modulation
    D7A_CLASS_EXT_UNB,
    /// Decawave / UWB modulation (PHY 802.15.4a)
    D7A_CLASS_EXT_UWB,
    /// Quantity
    D7A_CLASS_EXT_QTY,

} d7a_class_ext_t;

// =======================================================================
// d7a_band_t
// -----------------------------------------------------------------------
/// Enumerator of the PHY channel bands
// =======================================================================
typedef enum
{
    /// Universal Band 0.0 to 1684 MHz
    D7A_BAND_000    = 0,
    /// RFU (band 315 MHz ?)
    D7A_BAND_RFU_1  = 1,
    /// Band 433.060 to 434.785 MHz
    D7A_BAND_433    = 2,
    /// Band 863.000 to 870.000 MHz
    D7A_BAND_868    = 3,
    /// Band 902.000 to 928.000 MHz
    D7A_BAND_915    = 4,
    /// RFU
    D7A_BAND_RFU_5  = 5,
    /// Band 2400.000 to 2480.000 MHz
    D7A_BAND_2400   = 6,
    /// Band 3.1 to 10.6 GHz exclusively for UWB usage
    D7A_BAND_UWB    = 7,
    /// Quantity
    D7A_BAND_QTY

} d7a_band_t;

// =======================================================================
// d7a_cs_t
// -----------------------------------------------------------------------
/// Enumerator of the PHY channel coding schemes
// =======================================================================
typedef enum
{
    /// Nominal - PN9 encoding
    /// Extension LoRa - LoRaWAN mode 0 (BW = 125 KHz, SF = 12, FEC 4/5)
    /// Extension UNB - TBD
    /// Extension UWB - rate 110 kbps
    D7A_CS_0 = 0,
    /// Nominal - 1/2 FEC + punc 1/4 + PN9 encoding
    /// Extension LoRa - LoRaWAN mode 1 (BW = 125 KHz, SF = 11, FEC 4/5)
    /// Extension UNB - TBD
    /// Extension UWB - rate 850 kbps
    D7A_CS_1,
    /// Nominal - 1/2 FEC + PN9 encoding
    /// Extension LoRa - LoRaWAN mode 2 (BW = 125 KHz, SF = 10, FEC 4/5)
    /// Extension UNB - TBD
    /// Extension UWB - rate 6.8 Mbps
    D7A_CS_2,
    /// Nominal - RFU
    /// Extension LoRa - LoRaWAN mode 3 (BW = 125 KHz, SF =  8, FEC 4/5)
    /// Extension UNB - TBD
    /// Extension UWB - TBD
    D7A_CS_3,
    /// Quantity
    D7A_CS_QTY,

} d7a_cs_t;

#define D7A_CLASS(a)        (a+4)

//======================================================================
/// PHY Duty configuration
//======================================================================
typedef union
{
    struct {
        /// Window type (1: sliding, 0 : fixed)
        u8 sliding  : 1;
        /// RFU
        u8 rfu      : 3;
        /// When the widow type is sliding, number of sub-windows.
        /// This parameter is used to manage properly the duty
        /// limit without having to record every unitary TX event.
        /// Instead the duty is always limited to (sub_nb-1)/sub_nb
        /// and the duty credit never exceeds 1/sub_nb of the maximum
        /// available duty over the window (sliding sub-window).
        /// The bigger is sub_nb, the closer the credited duty is to
        /// the duty limit, but the smaller is the immediate available
        /// credit. For regular transmission of small packets, sub_nb
        /// can be kept high (> 10). For rare transmissions of long
        /// packets, sub_nb shall be kept small (< 5).
        u8 sub_nb   : 4;
    } bf;

    /// word access
    u8 word;

} d7a_phy_duty_cfg_t;

// =======================================================================
// d7a_phy_cfg_t
// -----------------------------------------------------------------------
/// PHY Configuration File structure
// =======================================================================
TYPEDEF_STRUCT_PACKED
{
    /// Preamble length depending on channel class
    u8 preamble[D7A_CLASS_QTY];
    /// Maximum authorized EIRP (dBm)
    s8 eirp_limit;
    /// TX duty cycle window in 1/10 of sec
    d7a_ctf_t duty_win;
    /// Duty management configuration
    d7a_phy_duty_cfg_t duty_cfg;
    /// TX Duty cycle limit in 1/1024 (full scale is 1024)
    u16 duty_limit;

} d7a_phy_cfg_t;
#define D7A_FS_PHY_CFG_SIZE             sizeof(d7a_phy_cfg_t)

/// =======================================================================
/// =======================================================================
///
/// 9. PHY STATUS
///
/// =======================================================================
/// =======================================================================

/// Maximum number of monitored channels
#define D7A_PHY_MONITORED_CHANNELS              (32)

//======================================================================
/// PHY Channel Status Identifier
//======================================================================
typedef union
{
    struct {
        /// Channel index (maximum 1039 indexes for ISM 915, use 11 bits)
        u16 idx   : 11;
        /// duty limit (0 : duty limit applies, 1 : no duty limit)
        u16 duty  :  1;
        /// Channel bandwidth (1 : 25 kHz, 0 : 200 kHz)
        u16 lo    :  1;
        /// Band
        u16 band  :  3;
    } bf;

    /// word access
    u16 word;

} d7a_phy_status_ch_id_t;

//======================================================================
/// PHY Channel Monitoring Status
//======================================================================
TYPEDEF_STRUCT_PACKED
{
    // Identifier
    d7a_phy_status_ch_id_t id;
    /// Noise floor in -dBm
    u8 nf;

} d7a_phy_status_ch_t;

// =======================================================================
// d7a_phy_status_t
// -----------------------------------------------------------------------
/// PHY Status File structure
// =======================================================================
TYPEDEF_STRUCT_PACKED
{
    /// Time since power on in seconds
    u32 up_time;
    /// RX time since power on in seconds
    u32 rx_time;
    /// TX time since power on in seconds
    u32 tx_time;
    /// TX Duty cycle in 1/1024 (full scale is 1024)
    u16 tx_duty;
    /// Number of monitored channels
    u8  ch_nb;
    /// Channel Status List
    d7a_phy_status_ch_t ch[D7A_PHY_MONITORED_CHANNELS];
    /// Channel duty credit in compressed format (Ti)
    d7a_ctf_t ch_credit[D7A_PHY_MONITORED_CHANNELS];
    /// Channel duty
    d7a_ctf_t ch_duty[D7A_PHY_MONITORED_CHANNELS];

} d7a_phy_status_t;

#define D7A_FS_PHY_STATUS_SIZE          sizeof(d7a_phy_status_t)

/// =======================================================================
/// =======================================================================
///
/// 10. DLL CONFIGURATION
///
/// =======================================================================
/// =======================================================================

// =======================================================================
// d7a_nf_auto_t
// -----------------------------------------------------------------------
/// DLL Noise Floor Autoscaling
// =======================================================================
typedef enum
{
    /// No Autoscaling (default).
    /// Always use the Noise Floor and Noise Floor Variance defined in the Access Profile
    D7A_NF_AUTO_NO = 0,
    /// Slow RSSI Variation.
    /// Noise Floor and Noise Floor variance are computed based on regular RSSI measurements
    /// with a forget factor defined by the associated parameter
    D7A_NF_AUTO_SLOW  = 1,
    // Max number of methods
    D7A_NF_QTY  = 4,

} d7a_nf_auto_t;

// =======================================================================
// d7a_area_t
// -----------------------------------------------------------------------
/// DLL Area Enumerator
// =======================================================================
typedef enum
{
    // Unclassified (the Moon, Mars...)
    // Reserved for application specific configurations
    D7A_AREA_REGION_0   = 0x00,

    // Region 1 (Europe, Russia, Middle East)
    D7A_AREA_REGION_1   = 0x10,
    // Default area governed by ECC / CEPT / ITU regulations
    // European Union, Norway (?), UK (?), Switzerland (?)
    D7A_AREA_ECC        = D7A_AREA_REGION_1,
    // Russia
    D7A_AREA_RUSSIA,
    // United Arab Emirates
    D7A_AREA_UAE,

    // 0x18 to 0x1F Reserved for application specific configurations

    // Region 2 (Americas)
    D7A_AREA_REGION_2   = 0x20,
    // Default area governed by FCC regulations
    // US, Canada, Chile, Argentina, Columbia,...
    D7A_AREA_FCC        = D7A_AREA_REGION_2,
    // Brazil
    D7A_AREA_BRAZIL,

    // 0x28 to 0x2F Reserved for application specific configurations

    // Region 3 (Asia Pacific)
    D7A_AREA_REGION_3   = 0x30,
    // Default area governed by Asia-Pacific regulations (seems does not exist yet)
    D7A_AREA_ASIA       = D7A_AREA_REGION_3,
    // Australia and New Zealand
    D7A_AREA_AU_NZ,
    // India
    D7A_AREA_INDIA,
    // Pakistan
    D7A_AREA_PAKISTAN,
    // Korea
    D7A_AREA_KOREA,
    // Japan
    D7A_AREA_JAPAN,
    // China
    D7A_AREA_CHINA,
    // Hong Kong
    D7A_AREA_HK,
    // Singapore
    D7A_AREA_SINGAPORE,
    // Malaysia
    D7A_AREA_MALAYSIA,
    // Indonesia
    D7A_AREA_INDONESIA,

    // Area enumerator mask
    D7A_AREA_MASK   = 0x3f,

} d7a_area_t;

// =======================================================================
// d7a_cond_t
// -----------------------------------------------------------------------
/// D7A comparison conditions
// =======================================================================
typedef enum
{
    /// <, =, > (always true)
    D7A_COND_ANY                       = 0,
    /// <, >
    D7A_COND_NE                        = 1,
    /// =
    D7A_COND_EQ                        = 2,
    /// <
    D7A_COND_LT                        = 3,
    /// <, =
    D7A_COND_LTE                       = 4,
    /// >
    D7A_COND_GT                        = 5,
    /// >, =
    D7A_COND_GTE                       = 6,
} d7a_cond_t;
// =======================================================================
// d7a_dll_cfg_flags_t
// -----------------------------------------------------------------------
/// DLL control flags structure
// =======================================================================
typedef union
{
    // bit access fields
    struct {
        // NF method
        u8 nf               :2;
        // btag filter condition (0 = disable)
        u8 btag_cond        :3;
        // 5.x - 6.x timout compatibility (REMOVEME)
        u8 comp_5x          :1;
        // RFU
        u8 rfu              :2;
    } bf;
    // byte access
    u8 byte;
} d7a_dll_cfg_flags_t;
// =======================================================================
// d7a_dll_cfg_flags2_t
// -----------------------------------------------------------------------
/// DLL control flags2 structure
// =======================================================================
typedef union
{
    // bit access fields
    struct {
        // Restart scheduler on TX done (main XCL)
        u8 on_tx_0          :1;
        // Restart scheduler on RX done (main XCL)
        u8 on_rx_0          :1;
        // Restart scheduler on TX done (1st secondary XCL)
        u8 on_tx_1          :1;
        // Restart scheduler on RX done (1st secondary XCL)
        u8 on_rx_1          :1;
        // Restart scheduler on TX done (2nd secondary XCL)
        u8 on_tx_2          :1;
        // Restart scheduler on RX done (2nd secondary XCL)
        u8 on_rx_2          :1;
        // Restart scheduler on TX done (3rd secondary XCL)
        u8 on_tx_3          :1;
        // Restart scheduler on RX done (3rd secondary XCL)
        u8 on_rx_3          :1;
    } bf;
    // byte access
    u8 byte;
} d7a_dll_cfg_flags2_t;
#define D7A_DLL_SCAN_XCL_NB     4
//======================================================================
/// DLL active access class scan structure
//======================================================================
TYPEDEF_STRUCT_PACKED
{
    /// Active scan access class
    d7a_xcl_t xcl;
    /// Scan timeout
    d7a_ctf_t scan_to;
} d7a_xcl_cfg_t;
// =======================================================================
// d7a_dll_cfg_t
// -----------------------------------------------------------------------
/// DLL Configuration File structure
// =======================================================================
TYPEDEF_STRUCT_PACKED
{
    /// Main active scan access class
    d7a_xcl_t xcl;
    /// Main active access class scan timeout
    d7a_ctf_t scan_to;
    /// Area identifier.
    u8  area;
    /// Control (Bis) bitfield
    d7a_dll_cfg_flags2_t ctrl2;
    /// Noise Floor Computation Method Control Bitfield
    d7a_dll_cfg_flags_t ctrl;
    /// Noise Floor Method Parameter (depends on Method ID)
    u8  nf_param;
    /// Secondary active scan access classes
    d7a_xcl_cfg_t xtra[D7A_DLL_SCAN_XCL_NB-1];
} d7a_dll_cfg_t;
#define D7A_FS_DLL_CFG_SIZE             sizeof(d7a_dll_cfg_t)

/// =======================================================================
/// =======================================================================
///
/// 11. DLL STATUS
///
/// =======================================================================
/// =======================================================================

// =======================================================================
// d7a_dll_status_t
// -----------------------------------------------------------------------
/// DLL status File structure
// =======================================================================
TYPEDEF_STRUCT_PACKED
{
    /// Last RX level [-dBm]
    u8  rxlev_last;
    /// Last link budget [dB]
    u8  lb_last;
    /// Measured noise floor [-dBm]
    u8  nf;
    /// Last RX channel header
    d7a_ch_header_t ch_header;
    /// Last RX channel index
    u16 ch_idx;
    /// scan timeout ratio in 1/1024th
    u16 scan_ratio;
    /// scan count since boot
    u32 scan_cnt;
    /// scan timeout count since boot (timeout on detected channel acivity)
    u32 scan_to_cnt;

} d7a_dll_status_t;
#define D7A_FS_DLL_STATUS_SIZE          sizeof(d7a_dll_status_t)

/// =======================================================================
/// =======================================================================
///
/// 12. NWL ROUTING
///
/// =======================================================================
/// =======================================================================

// =======================================================================
// d7a_nwl_routing_mode_t
// -----------------------------------------------------------------------
/// NWL routing modes
// =======================================================================
typedef enum
{
    /// Default routing type
    D7A_NWL_ROUTING_MODE_DEFAULT = 0,
    /// Qty
    D7A_NWL_ROUTING_MODE_QTY

} d7a_nwl_routing_mode_t;

// =======================================================================
// d7a_nwl_routing_entry_t
// -----------------------------------------------------------------------
/// NWL Single Routing entry
// =======================================================================
TYPEDEF_STRUCT_PACKED
{
    /// Destination node to be reached
    d7a_addressee_t dst;
    /// Intermediary node to use
    d7a_addressee_t itm;

} d7a_nwl_routing_entry_t;

// =======================================================================
// d7a_fs_nwl_routing_t
// -----------------------------------------------------------------------
/// NWL Routing File structure
// =======================================================================
#define D7A_FS_NWL_ROUTING_SIZE(n)          (sizeof(u8) + (n) * sizeof(d7a_nwl_routing_entry_t))
#define d7a_nwl_routing_t(n) \
struct __attribute__((packed)) \
{ \
    u8 routing;\
    d7a_nwl_routing_entry_t entry[(n)];\
}

/// =======================================================================
/// =======================================================================
///
/// 13. NWL SECURITY
///
/// =======================================================================
/// =======================================================================
// =======================================================================
// d7a_nwl_nls_status_bmp_t
// -----------------------------------------------------------------------
/// NWL Security status bitmap
// =======================================================================
typedef union
{
    // bit access fields
    struct {
        /// SSR filter enable
        u8 ssr_en   : 1;
        /// SSR filter overflow
        u8 ssr_ovf  : 1;
        /// RFU
        u8 rfu      : 6;
    } bf;

    // byte access
    u8 byte;

} d7a_nls_status_bmp_t;

// =======================================================================
// d7a_nwl_fs_security_t
// -----------------------------------------------------------------------
/// NWL Securty fil definition
// =======================================================================
TYPEDEF_STRUCT_PACKED
{
    /// Security state
    d7a_nwl_security_t sec;
    /// Keyring ID
    u8 ringid[4];
    /// NLS status flags
    d7a_nls_status_bmp_t bmp;

} d7a_nwl_fs_security_t;

#define D7A_FS_NWL_SECURITY_ALLOC_SIZE  sizeof(d7a_nwl_fs_security_t)
#define D7A_FS_NWL_SECURITY_SIZE        (D7A_FS_NWL_SECURITY_ALLOC_SIZE)

/// =======================================================================
/// =======================================================================
///
/// 14. NWL SECURITY KEY
///
/// =======================================================================
/// =======================================================================

// =======================================================================
// d7a_nwl_key_t
// -----------------------------------------------------------------------
/// NWL AES-128 key File structure
// =======================================================================
TYPEDEF_STRUCT_PACKED
{
    /// NWL Security key
    u8  val[16];
    /// Counter associated to the key (key id)
    u8  keyid;
    /// Keyring identifier
    u8  ringid[4];

} d7a_nwl_key_t;
#define D7A_FS_NWL_KEY_SIZE             sizeof(d7a_nwl_key_t)

/// =======================================================================
/// =======================================================================
///
/// 14. NWL SECURITY STATE REGISTER
///
/// =======================================================================
/// =======================================================================

// =======================================================================
// d7a_nwl_ssr_filter_t
// -----------------------------------------------------------------------
/// NWL SSR filter
// =======================================================================
typedef union
{
    // bit access fields
    struct {
        /// SSR filter enable
        u8 en       : 1;
        /// Consider only initial requests
        u8 req_only : 1;
        /// RFU
        u8 rfu      : 6;
    } bf;

    // byte access
    u8 byte;

} d7a_nwl_ssr_filter_t;

// =======================================================================
// d7a_nwl_cfg_t
// -----------------------------------------------------------------------
/// NWL Security Configuration structure
// =======================================================================
TYPEDEF_STRUCT_PACKED
{
    // NLS filter
    u16 nls_filter;
    // SSR filter
    d7a_nwl_ssr_filter_t ssr_filter;
    // Number of SSR entries
    u8 ssr_len;

} d7a_nwl_cfg_t;

// =======================================================================
// d7a_nwl_ssr_entry_t
// -----------------------------------------------------------------------
/// NWL Single SSR entry
// =======================================================================
TYPEDEF_STRUCT_PACKED
{
    /// Security state
    d7a_nwl_security_t sec;
    /// distant UID
    u8 uid[D7A_UID_LEN];

} d7a_nwl_ssr_entry_t;

// =======================================================================
// d7a_nwl_ssr_t
// -----------------------------------------------------------------------
/// NWL Security State Register File structure
// =======================================================================
#define D7A_FS_NWL_SSR_SIZE(n)              (sizeof(d7a_nwl_cfg_t) + (n) * sizeof(d7a_nwl_ssr_entry_t))
#define d7a_nwl_ssr_t(n) \
struct __attribute__((packed)) \
{ \
    d7a_nwl_cfg_t cfg;\
    d7a_nwl_ssr_entry_t entry[(n)];\
}

/// =======================================================================
/// =======================================================================
///
/// 16. NWL STATUS
///
/// =======================================================================
/// =======================================================================


// =======================================================================
// d7a_nwl_status_t
// -----------------------------------------------------------------------
/// NWL status File structure
// =======================================================================
TYPEDEF_STRUCT_PACKED
{
    /// Number of received valid BG frames
    u32 rx_bg_cnt;
    /// Number of received valid FG frames (D7AAdvP)
    u32 rx_fg_adv_cnt;
    /// Number of received valid FG frames (D7ANP)
    u32 rx_fg_cnt;
    /// Number of transmitted frames
    u32 tx_cnt;
    /// Number of tx failures
    u32 tx_fails;

} d7a_nwl_status_t;

#define D7A_FS_NWL_STATUS_SIZE          sizeof(d7a_nwl_status_t)

/// =======================================================================
/// =======================================================================
///
/// 17. TRL STATUS
///
/// =======================================================================
/// =======================================================================

// =======================================================================
// d7a_tp_ack_status_zip_t
// -----------------------------------------------------------------------
/// Structure of the D7ATP ACK Template (compressed ACK record)
// =======================================================================
typedef struct
{
    /// Length of the bitmap
    u8  len;
    /// First non-received ID, before which all IDs are receive.
    u8  start_id;
    /// Last received ID
    u8  end_id;
    /// Reception status bitmap
    u8  bitmap[32];

} d7a_tp_ack_status_zip_t;

// =======================================================================
// d7a_trl_status_t
// -----------------------------------------------------------------------
/// Transport Layer Status File structure
// =======================================================================
TYPEDEF_STRUCT_PACKED
{
    /// Real length of the whole record.
    u8 len;
    // Token of the recorded zip
    u8 token;
    /// At least room for one full zip
    d7a_tp_ack_status_zip_t zip;

} d7a_trl_status_t;
#define D7A_FS_TRL_STATUS_SIZE       sizeof(d7a_trl_status_t)

/// =======================================================================
/// =======================================================================
///
/// 18. SEL CONFIGURATION
///
/// =======================================================================
/// =======================================================================

// =======================================================================
// d7a_agc_ctrl_t
// -----------------------------------------------------------------------
/// AGC control structure
// =======================================================================
typedef union
{
    // bit access fields
    struct {
        /// AGC on
        /// When the field is not set, all Requesters and all Responders
        /// set their transmission power to the maximum authorized level.
        /// When the field is set, all Requesters, upon failure to receive
        /// an acknowledgement, reset their transmission power as defined
        /// by the TOGGLE field. Unicast transmissions (unicast requests
        /// and all responses) adjust the transmission power so that the
        /// expected reception RXLEV at the receiving party equals the
        /// last valid target RXLEV provided by that party, or when not
        /// provided, -80 dBm.
        u8 on       : 1;
        /// Enable target RXLEV reporting
        /// When set, the target RXLEV is provided by the transmitting
        /// party in the DLL header. The target RXLEV is computed based
        /// on local noise floor, channel class and device sensitivity
        u8 rxlev    : 1;
        /// Toggle between min and max
        /// On a first failure or when the field is not set, the transmission
        /// power is set to the maximum authorized level. When the field is
        /// set, on a subsequent failure, the transmission power is toggled
        /// between the minimum possible level and the maximum authorized
        /// levels.
        u8 toggle   : 1;
        // RFU
        u8 rfu      : 5;
    } bf;

    // byte access
    u8 byte;

} d7a_agc_ctrl_t;

// =======================================================================
// d7a_sel_filter_t
// -----------------------------------------------------------------------
/// SEL segment filter options structure
// =======================================================================
typedef union
{
    // bit access fields
    struct {
        /// Filter segments marked with retry if no response is expected
        u8 retry        : 1;
        /// Do not count TX CSMA-CA errors as ACK failures
        u8 csma_err     : 1;
        /// Do not count TX duty errors as ACK failures
        u8 duty_err     : 1;
        // RFU
        u8 rfu          : 5;
    } bf;

    // byte access
    u8 byte;

} d7a_sel_filter_t;

// =======================================================================
// d7a_sel_cfg_t
// -----------------------------------------------------------------------
/// SEL configuration file structure
// =======================================================================
TYPEDEF_STRUCT_PACKED
{
    /// AGC mode
    d7a_agc_ctrl_t agc_ctrl;
    /// AGC parameter
    u8  agc_param;
    /// retry (valid in all RESP modes)
    u8  retry;
    /// repeat (valid in RESP_NO_RPT)
    u8  repeat;
    /// maximum acknowledgement window
    u8  ack_win;
    /// segment filter options
    d7a_sel_filter_t filter;
    /// request timeout in seconds for non-persistent FIFOs (conpressed format)
    d7a_ctf_t to;

} d7a_sel_cfg_t;
#define D7A_FS_SEL_CFG_SIZE  sizeof(d7a_sel_cfg_t)

/// =======================================================================
/// =======================================================================
///
/// 19. SEL STATUS
///
/// =======================================================================
/// =======================================================================

// =======================================================================
// d7a_sel_status_t
// -----------------------------------------------------------------------
/// Structure of the SEL Protocol status
// =======================================================================
TYPEDEF_STRUCT_PACKED
{
    /// UID of the device issuing the FCP
    u8  uid[D7A_UID_LEN];
    /// Frequency offset in Hz
    s16 fof;

} d7a_sel_status_t;
#define D7A_FS_SEL_STATUS_SIZE      sizeof(d7a_sel_status_t)

/// =======================================================================
/// =======================================================================
///
/// 26. CHALLENGE
///
/// =======================================================================
/// =======================================================================
#define D7A_CHALLENGE_SIZE      32
#define D7A_FS_CHALLENGE_SIZE   sizeof(d7a_chal_t)
TYPEDEF_STRUCT_PACKED
{
    /// Challenge validity in Minutes
    u8  validity;
    /// Challenge
    u8  chal[D7A_CHALLENGE_SIZE];

} d7a_chal_t;

/// =======================================================================
/// =======================================================================
///
/// 29. TIME STAMP
///
/// =======================================================================
/// =======================================================================

// =======================================================================
// d7a_time_stamp_t
// -----------------------------------------------------------------------
/// Structure of the Time stamp status
// =======================================================================
TYPEDEF_STRUCT_PACKED
{
    /// Ping stamp (aka roundtrip time) in x100 picosec
    /// This is a RX timestamp relative to the last TX timestamp
    u32 rx;
    /// Pong stamp (aka reply time) in x100 picosec
    /// This is a TX timestamp relative to the last RX timestamp
    u32 tx;

} d7a_time_stamp_t;
#define D7A_FS_TIME_STAMP_SIZE      sizeof(d7a_time_stamp_t)

/// =======================================================================
/// =======================================================================
///
/// 32-47. ACCESS PROFILES
///
/// =======================================================================
/// =======================================================================

// =======================================================================
// d7a_subprofile_t
// -----------------------------------------------------------------------
/// Sub-profile structure
// =======================================================================
TYPEDEF_STRUCT_PACKED
{
    /// Subband select bitmap
    u8 bmp;
    /// Scheduler period
    d7a_ctf_t tsched;

} d7a_subprofile_t;

// =======================================================================
// d7a_cca_cfg_t
// -----------------------------------------------------------------------
/// Clear Channel Assessment Configuration structure
// =======================================================================
typedef union
{
    // bit access fields
    struct {
        /// Default CCA Threshold (-dBm)
        u8 val : 7;
        /// CCA not required flag
        u8 opt : 1;
    } bf;

    // byte access
    u8 byte;

} d7a_cca_cfg_t;

// =======================================================================
// d7a_subband_t
// -----------------------------------------------------------------------
/// Subband structure
// =======================================================================
TYPEDEF_STRUCT_PACKED
{
    /// Start channel index
    u16 start_idx;
    /// End channel index
    u16 end_idx;
    /// Transmission EIRP (dBm)
    s8  eirp;
    /// Clear channel assessment configuration (-dBm)
    d7a_cca_cfg_t cca;
    /// Maximum per-channel transmission duty cycle in per-mil in CTF
    d7a_ctf_t duty;

} d7a_subband_t;

// =======================================================================
// d7a_access_profile_t
// -----------------------------------------------------------------------
/// DLL Access Profile structure
// =======================================================================
TYPEDEF_STRUCT_PACKED
{
    /// Channel header
    d7a_ch_header_t header;
    /// Access control
    d7a_subprofile_t sp[4];
    /// Subbands
    d7a_subband_t sb[8];

} d7a_access_profile_t;
#define D7A_FS_AP_SIZE                  sizeof(d7a_access_profile_t)

#endif // __D7A_1X_FS_H__
