/// @copyright
/// ========================================================================={{{
/// Copyright (c) 2012-2020 WizziLab                                           /
/// All rights reserved                                                        /
///                                                                            /
/// =========================================================================}}}
/// @endcopyright

//  =======================================================================
/// @file           alp_helpers.h
/// @defgroup       ALP
/// @brief          ALP helpers functions
//  =======================================================================

#ifndef __ALP_HELPERS_H__
#define __ALP_HELPERS_H__

#include "hal_types.h"

//======================================================================
//======================================================================
//  ALP Helpers
//======================================================================
//======================================================================

// Basic _public container for ALP payload.
typedef struct {
    u32             len;
    u8              d[1];
} alp_pub_payload_t;

typedef struct {
    u8*             data;   // data
    int             type;
    union
    {
        struct {
            u32             length;
            u32             offset;
            u8              fid;
        } f_data;
        struct {
            u32             length;
            u32             offset;
            u8              fid;
        } f_prop;
        struct {
            s8              code;
            u8              id; // Action ID
        } status;
        struct {
            u8              itf;
            s8              err;
        } istatus;
        struct {
            u32             length;
            u8              type;
        } itf;
        struct {
            u8              id;
            u8              eop;
            u8              err;
        } tag;
        struct {
            u32             per;
            u8              type;
            u8              ifid;
        } urc;
    } meta;
} alp_parsed_chunk_t;

//======================================================================
// alp_size
//----------------------------------------------------------------------
/// @brief Return payload length of most ALP operations
/// @param op       : ALP Action amongst alp_opcodes_t
/// @param offset   : Associated Offset if relevant
/// @param size     : Associated Size if relevant
/// @retval payload size in bytes
//======================================================================
uint alp_size(u8 op, u32 offset, u32 length);

//======================================================================
// alp_encode_length
//----------------------------------------------------------------------
/// @brief Encodes an ALP length/offset field
/// @param p        : pointer to the payload buffer
/// @param len      : value to be encoded
/// @retval resulting payload size in bytes
//======================================================================
u8 alp_encode_length(u8* p, u32 len);

//======================================================================
// alp_encode_length
//----------------------------------------------------------------------
/// @brief Decodes an ALP length/offset field
/// @param p        : pointer to the pointer to payload buffer
/// @param actp     : pointer to ALP's Action Protocol Substitution flag
///                   Result amongst alp_actp_substitution_mode_t
/// @retval decoded value
//======================================================================
u32 alp_decode_length(u8** p, u8* actp);

//======================================================================
// alp_parse_chunk
//----------------------------------------------------------------------
/// @brief Parses an ALP response payload to a more generic
///        alp_parsed_chunk_t structure.
/// @param payload  : pointer to the pointer to payload buffer
/// @param resp     : pointer to alp_parsed_chunk_t structure
/// @retval number of parsed bytes
//======================================================================
int alp_parse_chunk(u8** payload, alp_parsed_chunk_t* resp);


#endif // __ALP_HELPERS_H__