/**
 ******************************************************************************
 * @file    LSM303C_ACC_driver.c
 * @author  MEMS Application Team
 * @version V1.1
 * @date    24-February-2016
 * @brief   LSM303C Accelerometer driver file
 ******************************************************************************
 * @attention
 *
 * <h2><center>&copy; COPYRIGHT(c) 2016 STMicroelectronics</center></h2>
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 *   1. Redistributions of source code must retain the above copyright notice,
 *      this list of conditions and the following disclaimer.
 *   2. Redistributions in binary form must reproduce the above copyright notice,
 *      this list of conditions and the following disclaimer in the documentation
 *      and/or other materials provided with the distribution.
 *   3. Neither the name of STMicroelectronics nor the names of its contributors
 *      may be used to endorse or promote products derived from this software
 *      without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 ******************************************************************************
 */

/* Includes ------------------------------------------------------------------*/
#include "LSM303C_ACC_driver.h"

/* Imported function prototypes ----------------------------------------------*/
extern uint8_t LSM303C_ACC_IO_Write(void *handle, uint8_t WriteAddr, uint8_t *pBuffer, uint16_t nBytesToWrite);
extern uint8_t LSM303C_ACC_IO_Read(void *handle, uint8_t ReadAddr, uint8_t *pBuffer, uint16_t nBytesToRead);

/* Private typedef -----------------------------------------------------------*/

/* Private define ------------------------------------------------------------*/

/* Private macro -------------------------------------------------------------*/

/* Private variables ---------------------------------------------------------*/

/* Private functions ---------------------------------------------------------*/

/*******************************************************************************
* Function Name     : LSM303C_ACC_ReadReg
* Description       : Generic Reading function. It must be fullfilled with either
*                   : I2C or SPI reading functions                  
* Input             : Register Address
* Output            : Data REad
* Return            : None
*******************************************************************************/
status_t LSM303C_ACC_ReadReg(void *handle, u8_t Reg, u8_t* Data) 
{
  
  if (LSM303C_ACC_IO_Read(handle, Reg, Data, 1))
  {
    return MEMS_ERROR;
  }
  else
  {
    return MEMS_SUCCESS;
  }
}

/*******************************************************************************
* Function Name     : LSM303C_ACC_WriteReg
* Description       : Generic Writing function. It must be fullfilled with either
*                   : I2C or SPI writing function
* Input             : Register Address, Data to be written
* Output            : None
* Return            : None
*******************************************************************************/
status_t LSM303C_ACC_WriteReg(void *handle, u8_t Reg, u8_t Data) 
{
    
  if (LSM303C_ACC_IO_Write(handle, Reg, &Data, 1))
  {
    return MEMS_ERROR;
  }
  else
  {
    return MEMS_SUCCESS;
  }
}

/*******************************************************************************
* Function Name     : SwapHighLowByte
* Description       : Swap High/low byte in multiple byte values 
*                     It works with minimum 2 byte for every dimension.
*                     Example x,y,z with 2 byte for every dimension
*
* Input             : bufferToSwap -> buffer to swap 
*                     numberOfByte -> the buffer length in byte
*                     dimension -> number of dimension 
*
* Output            : bufferToSwap -> buffer swapped 
* Return            : None
*******************************************************************************/
void LSM303C_ACC_SwapHighLowByte(u8_t *bufferToSwap, u8_t numberOfByte, u8_t dimension)
{

  u8_t numberOfByteForDimension, i, j;
  u8_t tempValue[10];
  
  numberOfByteForDimension=numberOfByte/dimension;
  
  for (i=0; i<dimension;i++ )
  {
    for (j=0; j<numberOfByteForDimension;j++ )
        tempValue[j]=bufferToSwap[j+i*numberOfByteForDimension];
    for (j=0; j<numberOfByteForDimension;j++ )
        *(bufferToSwap+i*(numberOfByteForDimension)+j)=*(tempValue+(numberOfByteForDimension-1)-j);
  } 
}

/* Exported functions ---------------------------------------------------------*/

/*******************************************************************************
* Function Name  : LSM303C_ACC_R_WHO_AM_I
* Description    : Read WHO_AM_I
* Input          : Pointer to u8_t
* Output         : Status of WHO_AM_I 
* Return         : Status [MEMS_ERROR, MEMS_SUCCESS]
*******************************************************************************/

status_t LSM303C_ACC_R_WHO_AM_I(void *handle, u8_t *value)
{
 if( !LSM303C_ACC_ReadReg(handle, LSM303C_WHO_AM_I_REG_A, (u8_t *)value) )
    return MEMS_ERROR;

  *value &= LSM303C_ACC_WHO_AM_I_MASK; //coerce   
  *value = *value >> LSM303C_ACC_WHO_AM_I_POSITION; //mask    

  return MEMS_SUCCESS;
}

/*******************************************************************************
* Function Name  : LSM303C_ACC_W_XEN
* Description    : Write XEN
* Input          : LSM303C_ACC_XEN_t
* Output         : None
* Return         : Status [MEMS_ERROR, MEMS_SUCCESS]
*******************************************************************************/
status_t  LSM303C_ACC_W_XEN(void *handle, LSM303C_ACC_XEN_t newValue)
{
  u8_t value;

  if( !LSM303C_ACC_ReadReg(handle, LSM303C_CTRL_REG1_A, &value) )
    return MEMS_ERROR;

  value &= ~LSM303C_ACC_XEN_MASK; 
  value |= newValue;
  
  if( !LSM303C_ACC_WriteReg(handle, LSM303C_CTRL_REG1_A, value) )
    return MEMS_ERROR;

  return MEMS_SUCCESS;
}

/*******************************************************************************
* Function Name  : LSM303C_ACC_R_XEN
* Description    : Read XEN
* Input          : Pointer to LSM303C_ACC_XEN_t
* Output         : Status of XEN see LSM303C_ACC_XEN_t
* Return         : Status [MEMS_ERROR, MEMS_SUCCESS]
*******************************************************************************/

status_t LSM303C_ACC_R_XEN(void *handle, LSM303C_ACC_XEN_t *value)
{
 if( !LSM303C_ACC_ReadReg(handle, LSM303C_CTRL_REG1_A, (u8_t *)value) )
    return MEMS_ERROR;

  *value &= LSM303C_ACC_XEN_MASK; //mask

  return MEMS_SUCCESS;
}
/*******************************************************************************
* Function Name  : LSM303C_ACC_W_YEN
* Description    : Write YEN
* Input          : LSM303C_ACC_YEN_t
* Output         : None
* Return         : Status [MEMS_ERROR, MEMS_SUCCESS]
*******************************************************************************/
status_t  LSM303C_ACC_W_YEN(void *handle, LSM303C_ACC_YEN_t newValue)
{
  u8_t value;

  if( !LSM303C_ACC_ReadReg(handle, LSM303C_CTRL_REG1_A, &value) )
    return MEMS_ERROR;

  value &= ~LSM303C_ACC_YEN_MASK; 
  value |= newValue;
  
  if( !LSM303C_ACC_WriteReg(handle, LSM303C_CTRL_REG1_A, value) )
    return MEMS_ERROR;

  return MEMS_SUCCESS;
}

/*******************************************************************************
* Function Name  : LSM303C_ACC_R_YEN
* Description    : Read YEN
* Input          : Pointer to LSM303C_ACC_YEN_t
* Output         : Status of YEN see LSM303C_ACC_YEN_t
* Return         : Status [MEMS_ERROR, MEMS_SUCCESS]
*******************************************************************************/

status_t LSM303C_ACC_R_YEN(void *handle, LSM303C_ACC_YEN_t *value)
{
 if( !LSM303C_ACC_ReadReg(handle, LSM303C_CTRL_REG1_A, (u8_t *)value) )
    return MEMS_ERROR;

  *value &= LSM303C_ACC_YEN_MASK; //mask

  return MEMS_SUCCESS;
}
/*******************************************************************************
* Function Name  : LSM303C_ACC_W_ZEN
* Description    : Write ZEN
* Input          : LSM303C_ACC_ZEN_t
* Output         : None
* Return         : Status [MEMS_ERROR, MEMS_SUCCESS]
*******************************************************************************/
status_t  LSM303C_ACC_W_ZEN(void *handle, LSM303C_ACC_ZEN_t newValue)
{
  u8_t value;

  if( !LSM303C_ACC_ReadReg(handle, LSM303C_CTRL_REG1_A, &value) )
    return MEMS_ERROR;

  value &= ~LSM303C_ACC_ZEN_MASK; 
  value |= newValue;
  
  if( !LSM303C_ACC_WriteReg(handle, LSM303C_CTRL_REG1_A, value) )
    return MEMS_ERROR;

  return MEMS_SUCCESS;
}

/*******************************************************************************
* Function Name  : LSM303C_ACC_R_ZEN
* Description    : Read ZEN
* Input          : Pointer to LSM303C_ACC_ZEN_t
* Output         : Status of ZEN see LSM303C_ACC_ZEN_t
* Return         : Status [MEMS_ERROR, MEMS_SUCCESS]
*******************************************************************************/

status_t LSM303C_ACC_R_ZEN(void *handle, LSM303C_ACC_ZEN_t *value)
{
 if( !LSM303C_ACC_ReadReg(handle, LSM303C_CTRL_REG1_A, (u8_t *)value) )
    return MEMS_ERROR;

  *value &= LSM303C_ACC_ZEN_MASK; //mask

  return MEMS_SUCCESS;
}

/*******************************************************************************
* Function Name  : LSM303C_ACC_W_ODR
* Description    : Write ODR
* Input          : LSM303C_ACC_ODR_t
* Output         : None
* Return         : Status [MEMS_ERROR, MEMS_SUCCESS]
*******************************************************************************/
status_t  LSM303C_ACC_W_ODR(void *handle, LSM303C_ACC_ODR_t newValue)
{
  u8_t value;

  if( !LSM303C_ACC_ReadReg(handle, LSM303C_CTRL_REG1_A, &value) )
    return MEMS_ERROR;

  value &= ~LSM303C_ACC_ODR_MASK; 
  value |= newValue;
  
  if( !LSM303C_ACC_WriteReg(handle, LSM303C_CTRL_REG1_A, value) )
    return MEMS_ERROR;

  return MEMS_SUCCESS;
}

/*******************************************************************************
* Function Name  : LSM303C_ACC_R_ODR
* Description    : Read ODR
* Input          : Pointer to LSM303C_ACC_ODR_t
* Output         : Status of ODR see LSM303C_ACC_ODR_t
* Return         : Status [MEMS_ERROR, MEMS_SUCCESS]
*******************************************************************************/

status_t LSM303C_ACC_R_ODR(void *handle, LSM303C_ACC_ODR_t *value)
{
 if( !LSM303C_ACC_ReadReg(handle, LSM303C_CTRL_REG1_A, (u8_t *)value) )
    return MEMS_ERROR;

  *value &= LSM303C_ACC_ODR_MASK; //mask

  return MEMS_SUCCESS;
}

/*******************************************************************************
* Function Name  : LSM303C_ACC_W_Data_Filter
* Description    : Write FDS
* Input          : LSM303C_ACC_FDS_t
* Output         : None
* Return         : Status [MEMS_ERROR, MEMS_SUCCESS]
*******************************************************************************/
status_t  LSM303C_ACC_W_Data_Filter(void *handle, LSM303C_ACC_FDS_t newValue)
{
  u8_t value;

  if( !LSM303C_ACC_ReadReg(handle, LSM303C_CTRL_REG2_A, &value) )
    return MEMS_ERROR;

  value &= ~LSM303C_ACC_FDS_MASK; 
  value |= newValue;
  
  if( !LSM303C_ACC_WriteReg(handle, LSM303C_CTRL_REG2_A, value) )
    return MEMS_ERROR;

  return MEMS_SUCCESS;
}

/*******************************************************************************
* Function Name  : LSM303C_ACC_R_Data_Filter
* Description    : Read FDS
* Input          : Pointer to LSM303C_ACC_FDS_t
* Output         : Status of FDS see LSM303C_ACC_FDS_t
* Return         : Status [MEMS_ERROR, MEMS_SUCCESS]
*******************************************************************************/

status_t LSM303C_ACC_R_Data_Filter(void *handle, LSM303C_ACC_FDS_t *value)
{
 if( !LSM303C_ACC_ReadReg(handle, LSM303C_CTRL_REG2_A, (u8_t *)value) )
    return MEMS_ERROR;

  *value &= LSM303C_ACC_FDS_MASK; //mask

  return MEMS_SUCCESS;
}

/*******************************************************************************
* Function Name  : LSM303C_ACC_W_hpf_cutoff_freq
* Description    : Write HPCF
* Input          : LSM303C_ACC_DFC_t
* Output         : None
* Return         : Status [MEMS_ERROR, MEMS_SUCCESS]
*******************************************************************************/
status_t  LSM303C_ACC_W_hpf_cutoff_freq(void *handle, LSM303C_ACC_DFC_t newValue)
{
  u8_t value;

  if( !LSM303C_ACC_ReadReg(handle, LSM303C_CTRL_REG2_A, &value) )
    return MEMS_ERROR;

  value &= ~LSM303C_ACC_DFC_MASK; 
  value |= newValue;
  
  if( !LSM303C_ACC_WriteReg(handle, LSM303C_CTRL_REG2_A, value) )
    return MEMS_ERROR;

  return MEMS_SUCCESS;
}

/*******************************************************************************
* Function Name  : LSM303C_ACC_R_hpf_cutoff_freq
* Description    : Read HPCF
* Input          : Pointer to LSM303C_ACC_DFC_t
* Output         : Status of HPCF see LSM303C_ACC_DFC_t
* Return         : Status [MEMS_ERROR, MEMS_SUCCESS]
*******************************************************************************/

status_t LSM303C_ACC_R_hpf_cutoff_freq(void *handle, LSM303C_ACC_DFC_t *value)
{
 if( !LSM303C_ACC_ReadReg(handle, LSM303C_CTRL_REG2_A, (u8_t *)value) )
    return MEMS_ERROR;

  *value &= LSM303C_ACC_DFC_MASK; //mask

  return MEMS_SUCCESS;
}
/*******************************************************************************
* Function Name  : LSM303C_ACC_W_hpf_mode
* Description    : Write HPM
* Input          : LSM303C_ACC_HPM_t
* Output         : None
* Return         : Status [MEMS_ERROR, MEMS_SUCCESS]
*******************************************************************************/
status_t  LSM303C_ACC_W_hpf_mode(void *handle, LSM303C_ACC_HPM_t newValue)
{
  u8_t value;

  if( !LSM303C_ACC_ReadReg(handle, LSM303C_CTRL_REG2_A, &value) )
    return MEMS_ERROR;

  value &= ~LSM303C_ACC_HPM_MASK; 
  value |= newValue;
  
  if( !LSM303C_ACC_WriteReg(handle, LSM303C_CTRL_REG2_A, value) )
    return MEMS_ERROR;

  return MEMS_SUCCESS;
}

/*******************************************************************************
* Function Name  : LSM303C_ACC_R_hpf_mode
* Description    : Read HPM
* Input          : Pointer to LSM303C_ACC_HPM_t
* Output         : Status of HPM see LSM303C_ACC_HPM_t
* Return         : Status [MEMS_ERROR, MEMS_SUCCESS]
*******************************************************************************/

status_t LSM303C_ACC_R_hpf_mode(void *handle, LSM303C_ACC_HPM_t *value)
{
 if( !LSM303C_ACC_ReadReg(handle, LSM303C_CTRL_REG2_A, (u8_t *)value) )
    return MEMS_ERROR;

  *value &= LSM303C_ACC_HPM_MASK; //mask

  return MEMS_SUCCESS;
}

/*******************************************************************************
* Function Name  : LSM303C_ACC_W_HiRes
* Description    : Write HR
* Input          : LSM303C_ACC_HR_t
* Output         : None
* Return         : Status [MEMS_ERROR, MEMS_SUCCESS]
*******************************************************************************/
status_t  LSM303C_ACC_W_HiRes(void *handle, LSM303C_ACC_HR_t newValue)
{
  u8_t value;

  if( !LSM303C_ACC_ReadReg(handle, LSM303C_CTRL_REG1_A, &value) )
    return MEMS_ERROR;

  value &= ~LSM303C_ACC_HR_MASK; 
  value |= newValue;
  
  if( !LSM303C_ACC_WriteReg(handle, LSM303C_CTRL_REG1_A, value) )
    return MEMS_ERROR;

  return MEMS_SUCCESS;
}

/*******************************************************************************
* Function Name  : LSM303C_ACC_R_HiRes
* Description    : Read HR
* Input          : Pointer to LSM303C_ACC_HR_t
* Output         : Status of HR see LSM303C_ACC_HR_t
* Return         : Status [MEMS_ERROR, MEMS_SUCCESS]
*******************************************************************************/

status_t LSM303C_ACC_R_HiRes(void *handle, LSM303C_ACC_HR_t *value)
{
 if( !LSM303C_ACC_ReadReg(handle, LSM303C_CTRL_REG1_A, (u8_t *)value) )
    return MEMS_ERROR;

  *value &= LSM303C_ACC_HR_MASK; //mask

  return MEMS_SUCCESS;
}

/*******************************************************************************
* Function Name  : LSM303C_ACC_W_FullScale
* Description    : Write FS
* Input          : LSM303C_ACC_FS_t
* Output         : None
* Return         : Status [MEMS_ERROR, MEMS_SUCCESS]
*******************************************************************************/
status_t  LSM303C_ACC_W_FullScale(void *handle, LSM303C_ACC_FS_t newValue)
{
  u8_t value;

  if( !LSM303C_ACC_ReadReg(handle, LSM303C_CTRL_REG4_A, &value) )
    return MEMS_ERROR;

  value &= ~LSM303C_ACC_FS_MASK; 
  value |= newValue;
  
  if( !LSM303C_ACC_WriteReg(handle, LSM303C_CTRL_REG4_A, value) )
    return MEMS_ERROR;

  return MEMS_SUCCESS;
}

/*******************************************************************************
* Function Name  : LSM303C_ACC_R_FullScale
* Description    : Read FS
* Input          : Pointer to LSM303C_ACC_FS_t
* Output         : Status of FS see LSM303C_ACC_FS_t
* Return         : Status [MEMS_ERROR, MEMS_SUCCESS]
*******************************************************************************/

status_t LSM303C_ACC_R_FullScale(void *handle, LSM303C_ACC_FS_t *value)
{
 if( !LSM303C_ACC_ReadReg(handle, LSM303C_CTRL_REG4_A, (u8_t *)value) )
    return MEMS_ERROR;

  *value &= LSM303C_ACC_FS_MASK; //mask

  return MEMS_SUCCESS;
}

/*******************************************************************************
* Function Name  : LSM303C_ACC_W_BlockDataUpdate
* Description    : Write BDU
* Input          : LSM303C_ACC_BDU_t
* Output         : None
* Return         : Status [MEMS_ERROR, MEMS_SUCCESS]
*******************************************************************************/
status_t  LSM303C_ACC_W_BlockDataUpdate(void *handle, LSM303C_ACC_BDU_t newValue)
{
  u8_t value;

  if( !LSM303C_ACC_ReadReg(handle, LSM303C_CTRL_REG1_A, &value) )
    return MEMS_ERROR;

  value &= ~LSM303C_ACC_BDU_MASK; 
  value |= newValue;
  
  if( !LSM303C_ACC_WriteReg(handle, LSM303C_CTRL_REG1_A, value) )
    return MEMS_ERROR;

  return MEMS_SUCCESS;
}

/*******************************************************************************
* Function Name  : LSM303C_ACC_R_BlockDataUpdate
* Description    : Read BDU
* Input          : Pointer to LSM303C_ACC_BDU_t
* Output         : Status of BDU see LSM303C_ACC_BDU_t
* Return         : Status [MEMS_ERROR, MEMS_SUCCESS]
*******************************************************************************/

status_t LSM303C_ACC_R_BlockDataUpdate(void *handle, LSM303C_ACC_BDU_t *value)
{
 if( !LSM303C_ACC_ReadReg(handle, LSM303C_CTRL_REG1_A, (u8_t *)value) )
    return MEMS_ERROR;

  *value &= LSM303C_ACC_BDU_MASK; //mask

  return MEMS_SUCCESS;
}

/*******************************************************************************
* Function Name  : LSM303C_ACC_R_XDataAvail
* Description    : Read XDA
* Input          : Pointer to LSM303C_ACC_XDA_t
* Output         : Status of XDA see LSM303C_ACC_XDA_t
* Return         : Status [MEMS_ERROR, MEMS_SUCCESS]
*******************************************************************************/

status_t LSM303C_ACC_R_XDataAvail(void *handle, LSM303C_ACC_XDA_t *value)
{
 if( !LSM303C_ACC_ReadReg(handle, LSM303C_STATUS_REG_A, (u8_t *)value) )
    return MEMS_ERROR;

  *value &= LSM303C_ACC_XDA_MASK; //mask

  return MEMS_SUCCESS;
}
/*******************************************************************************
* Function Name  : LSM303C_ACC_R_YDataAvail
* Description    : Read YDA
* Input          : Pointer to LSM303C_ACC_YDA_t
* Output         : Status of YDA see LSM303C_ACC_YDA_t
* Return         : Status [MEMS_ERROR, MEMS_SUCCESS]
*******************************************************************************/

status_t LSM303C_ACC_R_YDataAvail(void *handle, LSM303C_ACC_YDA_t *value)
{
 if( !LSM303C_ACC_ReadReg(handle, LSM303C_STATUS_REG_A, (u8_t *)value) )
    return MEMS_ERROR;

  *value &= LSM303C_ACC_YDA_MASK; //mask

  return MEMS_SUCCESS;
}
/*******************************************************************************
* Function Name  : LSM303C_ACC_R_ZDataAvail
* Description    : Read ZDA
* Input          : Pointer to LSM303C_ACC_ZDA_t
* Output         : Status of ZDA see LSM303C_ACC_ZDA_t
* Return         : Status [MEMS_ERROR, MEMS_SUCCESS]
*******************************************************************************/

status_t LSM303C_ACC_R_ZDataAvail(void *handle, LSM303C_ACC_ZDA_t *value)
{
 if( !LSM303C_ACC_ReadReg(handle, LSM303C_STATUS_REG_A, (u8_t *)value) )
    return MEMS_ERROR;

  *value &= LSM303C_ACC_ZDA_MASK; //mask

  return MEMS_SUCCESS;
}
/*******************************************************************************
* Function Name  : LSM303C_ACC_R_XYZDataAvail
* Description    : Read ZYXDA
* Input          : Pointer to LSM303C_ACC_ZYXDA_t
* Output         : Status of ZYXDA see LSM303C_ACC_ZYXDA_t
* Return         : Status [MEMS_ERROR, MEMS_SUCCESS]
*******************************************************************************/

status_t LSM303C_ACC_R_XYZDataAvail(void *handle, LSM303C_ACC_ZYXDA_t *value)
{
 if( !LSM303C_ACC_ReadReg(handle, LSM303C_STATUS_REG_A, (u8_t *)value) )
    return MEMS_ERROR;

  *value &= LSM303C_ACC_ZYXDA_MASK; //mask

  return MEMS_SUCCESS;
}
/*******************************************************************************
* Function Name  : LSM303C_ACC_R_XDataOverrun
* Description    : Read XOR
* Input          : Pointer to LSM303C_ACC_XOR_t
* Output         : Status of XOR see LSM303C_ACC_XOR_t
* Return         : Status [MEMS_ERROR, MEMS_SUCCESS]
*******************************************************************************/

status_t LSM303C_ACC_R_XDataOverrun(void *handle, LSM303C_ACC_XOR_t *value)
{
 if( !LSM303C_ACC_ReadReg(handle, LSM303C_STATUS_REG_A, (u8_t *)value) )
    return MEMS_ERROR;

  *value &= LSM303C_ACC_XOR_MASK; //mask

  return MEMS_SUCCESS;
}
/*******************************************************************************
* Function Name  : LSM303C_ACC_R_YDataOverrun
* Description    : Read YOR
* Input          : Pointer to LSM303C_ACC_YOR_t
* Output         : Status of YOR see LSM303C_ACC_YOR_t
* Return         : Status [MEMS_ERROR, MEMS_SUCCESS]
*******************************************************************************/

status_t LSM303C_ACC_R_YDataOverrun(void *handle, LSM303C_ACC_YOR_t *value)
{
 if( !LSM303C_ACC_ReadReg(handle, LSM303C_STATUS_REG_A, (u8_t *)value) )
    return MEMS_ERROR;

  *value &= LSM303C_ACC_YOR_MASK; //mask

  return MEMS_SUCCESS;
}
/*******************************************************************************
* Function Name  : LSM303C_ACC_R_ZDataOverrun
* Description    : Read ZOR
* Input          : Pointer to LSM303C_ACC_ZOR_t
* Output         : Status of ZOR see LSM303C_ACC_ZOR_t
* Return         : Status [MEMS_ERROR, MEMS_SUCCESS]
*******************************************************************************/

status_t LSM303C_ACC_R_ZDataOverrun(void *handle, LSM303C_ACC_ZOR_t *value)
{
 if( !LSM303C_ACC_ReadReg(handle, LSM303C_STATUS_REG_A, (u8_t *)value) )
    return MEMS_ERROR;

  *value &= LSM303C_ACC_ZOR_MASK; //mask

  return MEMS_SUCCESS;
}
/*******************************************************************************
* Function Name  : LSM303C_ACC_R_XYZDataOverrun
* Description    : Read ZYXOR
* Input          : Pointer to LSM303C_ACC_ZYXOR_t
* Output         : Status of ZYXOR see LSM303C_ACC_ZYXOR_t
* Return         : Status [MEMS_ERROR, MEMS_SUCCESS]
*******************************************************************************/

status_t LSM303C_ACC_R_XYZDataOverrun(void *handle, LSM303C_ACC_ZYXOR_t *value)
{
 if( !LSM303C_ACC_ReadReg(handle, LSM303C_STATUS_REG_A, (u8_t *)value) )
    return MEMS_ERROR;

  *value &= LSM303C_ACC_ZYXOR_MASK; //mask

  return MEMS_SUCCESS;
}

/*******************************************************************************
* Function Name  : status_t LSM303C_ACC_Get_Raw_Acceleration(u8_t *buff)
* Description    : Read Acceleration output register
* Input          : pointer to [u8_t]
* Output         : Acceleration buffer u8_t
* Return         : Status [MEMS_ERROR, MEMS_SUCCESS]
*******************************************************************************/
status_t LSM303C_ACC_Get_Raw_Acceleration(void *handle, u8_t *buff) 
{
  u8_t i, j, k;
  u8_t numberOfByteForDimension;
  
  numberOfByteForDimension=6/3;

  k=0;
  for (i=0; i<3;i++ ) 
  {
    for (j=0; j<numberOfByteForDimension;j++ )
    {   
        if( !LSM303C_ACC_ReadReg(handle, LSM303C_OUT_X_L_A+k, &buff[k]))
          return MEMS_ERROR;
        k++;    
    }
  }

  return MEMS_SUCCESS; 
}

/*
 * Following is the table of sensitivity values for each case.
 * Values are espressed in ug/digit.
 */
const long long LSM303C_ACC_Sensitivity_List[3] = {
       980, /* FS @2g */
       1950,    /* FS @4g */
       3900,    /* FS @8g */
};

/*
 * Values returned are espressed in mg.
 */
status_t LSM303C_ACC_Get_Acceleration(void *handle, int *buff)
{
  Type3Axis16bit_U raw_data_tmp;
  u8_t fs_mode = 0, shift = 0;
  LSM303C_ACC_FS_t fs;
 
  /* Determine the Full Scale the acc is set */
  if(!LSM303C_ACC_R_FullScale(handle, &fs)) {
    return MEMS_ERROR;
  }

  switch (fs) {
  case LSM303C_ACC_FS_2G:
    fs_mode = 0;
    break;

  case LSM303C_ACC_FS_4G:
    fs_mode = 1;
    break;

  case LSM303C_ACC_FS_8G:
    fs_mode = 2;
    break;
  }

  /* Read out raw accelerometer samples */
  if(!LSM303C_ACC_Get_Raw_Acceleration(handle, raw_data_tmp.u8bit)) {
    return MEMS_ERROR;
  }

  /* Apply proper shift and sensitivity */
  buff[0] = ((raw_data_tmp.i16bit[0] >> shift) * LSM303C_ACC_Sensitivity_List[fs_mode] + 500) / 1000;
  buff[1] = ((raw_data_tmp.i16bit[1] >> shift) * LSM303C_ACC_Sensitivity_List[fs_mode] + 500) / 1000;
  buff[2] = ((raw_data_tmp.i16bit[2] >> shift) * LSM303C_ACC_Sensitivity_List[fs_mode] + 500) / 1000;

  return MEMS_SUCCESS;
}
