/**
 *******************************************************************************
 * @file    LSM303C_MAG_driver.c
 * @author  MEMS Application Team
 * @version V1.1
 * @date    25-February-2016
 * @brief   LSM303C Magnetometer driver file
 *******************************************************************************
 * @attention
 *
 * <h2><center>&copy; COPYRIGHT(c) 2016 STMicroelectronics</center></h2>
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 *   1. Redistributions of source code must retain the above copyright notice,
 *      this list of conditions and the following disclaimer.
 *   2. Redistributions in binary form must reproduce the above copyright notice,
 *      this list of conditions and the following disclaimer in the documentation
 *      and/or other materials provided with the distribution.
 *   3. Neither the name of STMicroelectronics nor the names of its contributors
 *      may be used to endorse or promote products derived from this software
 *      without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 ******************************************************************************
 */

/* Includes ------------------------------------------------------------------*/
#include "LSM303C_MAG_driver.h"

/* Imported function prototypes ----------------------------------------------*/
extern uint8_t LSM303C_MAG_IO_Write(void *handle, uint8_t WriteAddr, uint8_t *pBuffer, uint16_t nBytesToWrite);
extern uint8_t LSM303C_MAG_IO_Read(void *handle, uint8_t ReadAddr, uint8_t *pBuffer, uint16_t nBytesToRead);

/* Private typedef -----------------------------------------------------------*/

/* Private define ------------------------------------------------------------*/

/* Private macro -------------------------------------------------------------*/

/* Private variables ---------------------------------------------------------*/

/* Private functions ---------------------------------------------------------*/

/*******************************************************************************
* Function Name     : LSM303C_MAG_ReadReg
* Description       : Generic Reading function. It must be fullfilled with either
*                   : I2C or SPI reading functions                  
* Input             : Register Address
* Output            : Data REad
* Return            : None
*******************************************************************************/
status_t LSM303C_MAG_ReadReg( void *handle, u8_t Reg, u8_t* Data ) 
{
  if (LSM303C_MAG_IO_Read(handle, Reg, Data, 1))
  {
    return MEMS_ERROR;
  }
  else
  {
    return MEMS_SUCCESS;
  }
}

/*******************************************************************************
* Function Name     : LSM303C_MAG_WriteReg
* Description       : Generic Writing function. It must be fullfilled with either
*                   : I2C or SPI writing function
* Input             : Register Address, Data to be written
* Output            : None
* Return            : None
*******************************************************************************/
status_t LSM303C_MAG_WriteReg( void *handle, u8_t Reg, u8_t Data ) 
{
    
  if (LSM303C_MAG_IO_Write(handle, Reg, &Data, 1))
  {
    return MEMS_ERROR;
  }
  else
  {
    return MEMS_SUCCESS;
  }
}

/*******************************************************************************
* Function Name     : SwapHighLowByte
* Description       : Swap High/low byte in multiple byte values 
*                     It works with minimum 2 byte for every dimension.
*                     Example x,y,z with 2 byte for every dimension
*
* Input             : bufferToSwap -> buffer to swap 
*                     numberOfByte -> the buffer length in byte
*                     dimension -> number of dimension 
*
* Output            : bufferToSwap -> buffer swapped 
* Return            : None
*******************************************************************************/
void LSM303C_MAG_SwapHighLowByte(u8_t *bufferToSwap, u8_t numberOfByte, u8_t dimension)
{

  u8_t numberOfByteForDimension, i, j;
  u8_t tempValue[10];
  
  numberOfByteForDimension=numberOfByte/dimension;
  
  for (i=0; i<dimension;i++ )
  {
    for (j=0; j<numberOfByteForDimension;j++ )
        tempValue[j]=bufferToSwap[j+i*numberOfByteForDimension];
    for (j=0; j<numberOfByteForDimension;j++ )
        *(bufferToSwap+i*(numberOfByteForDimension)+j)=*(tempValue+(numberOfByteForDimension-1)-j);
  } 
}

/* Exported functions ---------------------------------------------------------*/

/*******************************************************************************
* Function Name  : LSM303C_MAG_R_WHO_AM_I
* Description    : Read WHO_AM_I
* Input          : Pointer to u8_t
* Output         : Status of WHO_AM_I 
* Return         : Status [MEMS_ERROR, MEMS_SUCCESS]
*******************************************************************************/

status_t LSM303C_MAG_R_WHO_AM_I(void *handle, u8_t *value)
{
 if( !LSM303C_MAG_ReadReg(handle, LSM303C_WHO_AM_I_REG_M, (u8_t *)value) )
    return MEMS_ERROR;

  *value &= LSM303C_MAG_WHO_AM_I_MASK; //coerce   
  *value = *value >> LSM303C_MAG_WHO_AM_I_POSITION; //mask    

  return MEMS_SUCCESS;
}

/*******************************************************************************
* Function Name  : LSM303C_MAG_W_MD
* Description    : Write MD
* Input          : LSM303C_MAG_MD_t
* Output         : None
* Return         : Status [MEMS_ERROR, MEMS_SUCCESS]
*******************************************************************************/
status_t  LSM303C_MAG_W_MD(void *handle, LSM303C_MAG_MD_t newValue)
{
  u8_t value;

  if( !LSM303C_MAG_ReadReg(handle, LSM303C_CTRL_REG3_M, &value) )
    return MEMS_ERROR;

  value &= ~LSM303C_MAG_MD_MASK; 
  value |= newValue;
  
  if( !LSM303C_MAG_WriteReg(handle, LSM303C_CTRL_REG3_M, value) )
    return MEMS_ERROR;

  return MEMS_SUCCESS;
}

/*******************************************************************************
* Function Name  : LSM303C_MAG_R_MD
* Description    : Read MD
* Input          : Pointer to LSM303C_MAG_MD_t
* Output         : Status of MD see LSM303C_MAG_MD_t
* Return         : Status [MEMS_ERROR, MEMS_SUCCESS]
*******************************************************************************/

status_t LSM303C_MAG_R_MD(void *handle, LSM303C_MAG_MD_t *value)
{
 if( !LSM303C_MAG_ReadReg(handle, LSM303C_CTRL_REG3_M, (u8_t *)value) )
    return MEMS_ERROR;

  *value &= LSM303C_MAG_MD_MASK; //mask

  return MEMS_SUCCESS;
}
/*******************************************************************************
* Function Name  : LSM303C_MAG_W_ODR
* Description    : Write ODR
* Input          : LSM303C_MAG_ODR_t
* Output         : None
* Return         : Status [MEMS_ERROR, MEMS_SUCCESS]
*******************************************************************************/
status_t  LSM303C_MAG_W_ODR(void *handle, LSM303C_MAG_ODR_t newValue)
{
  u8_t value;

  if( !LSM303C_MAG_ReadReg(handle, LSM303C_CTRL_REG1_M, &value) )
    return MEMS_ERROR;

  value &= ~LSM303C_MAG_ODR_MASK; 
  value |= newValue;
  
  if( !LSM303C_MAG_WriteReg(handle, LSM303C_CTRL_REG1_M, value) )
    return MEMS_ERROR;

  return MEMS_SUCCESS;
}

/*******************************************************************************
* Function Name  : LSM303C_MAG_R_ODR
* Description    : Read ODR
* Input          : Pointer to LSM303C_MAG_ODR_t
* Output         : Status of ODR see LSM303C_MAG_ODR_t
* Return         : Status [MEMS_ERROR, MEMS_SUCCESS]
*******************************************************************************/

status_t LSM303C_MAG_R_ODR(void *handle, LSM303C_MAG_ODR_t *value)
{
 if( !LSM303C_MAG_ReadReg(handle, LSM303C_CTRL_REG1_M, (u8_t *)value) )
    return MEMS_ERROR;

  *value &= LSM303C_MAG_ODR_MASK; //mask

  return MEMS_SUCCESS;
}
/*******************************************************************************
* Function Name  : LSM303C_MAG_W_LP
* Description    : Write LP
* Input          : LSM303C_MAG_LP_t
* Output         : None
* Return         : Status [MEMS_ERROR, MEMS_SUCCESS]
*******************************************************************************/
status_t  LSM303C_MAG_W_LP(void *handle, LSM303C_MAG_LP_t newValue)
{
  u8_t value;

  if( !LSM303C_MAG_ReadReg(handle, LSM303C_CTRL_REG3_M, &value) )
    return MEMS_ERROR;

  value &= ~LSM303C_MAG_LP_MASK; 
  value |= newValue;
  
  if( !LSM303C_MAG_WriteReg(handle, LSM303C_CTRL_REG3_M, value) )
    return MEMS_ERROR;

  return MEMS_SUCCESS;
}

/*******************************************************************************
* Function Name  : LSM303C_MAG_R_LP
* Description    : Read LP
* Input          : Pointer to LSM303C_MAG_LP_t
* Output         : Status of LP see LSM303C_MAG_LP_t
* Return         : Status [MEMS_ERROR, MEMS_SUCCESS]
*******************************************************************************/

status_t LSM303C_MAG_R_LP(void *handle, LSM303C_MAG_LP_t *value)
{
 if( !LSM303C_MAG_ReadReg(handle, LSM303C_CTRL_REG3_M, (u8_t *)value) )
    return MEMS_ERROR;

  *value &= LSM303C_MAG_LP_MASK; //mask

  return MEMS_SUCCESS;
}
/*******************************************************************************
* Function Name  : LSM303C_MAG_W_SOFT_RST
* Description    : Write SOFT_RST
* Input          : LSM303C_MAG_SOFT_RST_t
* Output         : None
* Return         : Status [MEMS_ERROR, MEMS_SUCCESS]
*******************************************************************************/
status_t  LSM303C_MAG_W_SOFT_RST(void *handle, LSM303C_MAG_SOFT_RST_t newValue)
{
  u8_t value;

  if( !LSM303C_MAG_ReadReg(handle, LSM303C_CTRL_REG2_M, &value) )
    return MEMS_ERROR;

  value &= ~LSM303C_MAG_SOFT_RST_MASK; 
  value |= newValue;
  
  if( !LSM303C_MAG_WriteReg(handle, LSM303C_CTRL_REG2_M, value) )
    return MEMS_ERROR;

  return MEMS_SUCCESS;
}

/*******************************************************************************
* Function Name  : LSM303C_MAG_R_SOFT_RST
* Description    : Read SOFT_RST
* Input          : Pointer to LSM303C_MAG_SOFT_RST_t
* Output         : Status of SOFT_RST see LSM303C_MAG_SOFT_RST_t
* Return         : Status [MEMS_ERROR, MEMS_SUCCESS]
*******************************************************************************/

status_t LSM303C_MAG_R_SOFT_RST(void *handle, LSM303C_MAG_SOFT_RST_t *value)
{
 if( !LSM303C_MAG_ReadReg(handle, LSM303C_CTRL_REG2_M, (u8_t *)value) )
    return MEMS_ERROR;

  *value &= LSM303C_MAG_SOFT_RST_MASK; //mask

  return MEMS_SUCCESS;
}

/*******************************************************************************
* Function Name  : LSM303C_MAG_W_ST
* Description    : Write ST
* Input          : LSM303C_MAG_ST_t
* Output         : None
* Return         : Status [MEMS_ERROR, MEMS_SUCCESS]
*******************************************************************************/
status_t  LSM303C_MAG_W_ST(void *handle, LSM303C_MAG_ST_t newValue)
{
  u8_t value;

  if( !LSM303C_MAG_ReadReg(handle, LSM303C_CTRL_REG1_M, &value) )
    return MEMS_ERROR;

  value &= ~LSM303C_MAG_ST_MASK; 
  value |= newValue;
  
  if( !LSM303C_MAG_WriteReg(handle, LSM303C_CTRL_REG1_M, value) )
    return MEMS_ERROR;

  return MEMS_SUCCESS;
}

/*******************************************************************************
* Function Name  : LSM303C_MAG_R_ST
* Description    : Read ST
* Input          : Pointer to LSM303C_MAG_ST_t
* Output         : Status of ST see LSM303C_MAG_ST_t
* Return         : Status [MEMS_ERROR, MEMS_SUCCESS]
*******************************************************************************/

status_t LSM303C_MAG_R_ST(void *handle, LSM303C_MAG_ST_t *value)
{
 if( !LSM303C_MAG_ReadReg(handle, LSM303C_CTRL_REG1_M, (u8_t *)value) )
    return MEMS_ERROR;

  *value &= LSM303C_MAG_ST_MASK; //mask

  return MEMS_SUCCESS;
}
/*******************************************************************************
* Function Name  : LSM303C_MAG_W_BLE
* Description    : Write BLE
* Input          : LSM303C_MAG_BLE_t
* Output         : None
* Return         : Status [MEMS_ERROR, MEMS_SUCCESS]
*******************************************************************************/
status_t  LSM303C_MAG_W_BLE(void *handle, LSM303C_MAG_BLE_t newValue)
{
  u8_t value;

  if( !LSM303C_MAG_ReadReg(handle, LSM303C_CTRL_REG4_M, &value) )
    return MEMS_ERROR;

  value &= ~LSM303C_MAG_BLE_MASK; 
  value |= newValue;
  
  if( !LSM303C_MAG_WriteReg(handle, LSM303C_CTRL_REG4_M, value) )
    return MEMS_ERROR;

  return MEMS_SUCCESS;
}

/*******************************************************************************
* Function Name  : LSM303C_MAG_R_BLE
* Description    : Read BLE
* Input          : Pointer to LSM303C_MAG_BLE_t
* Output         : Status of BLE see LSM303C_MAG_BLE_t
* Return         : Status [MEMS_ERROR, MEMS_SUCCESS]
*******************************************************************************/

status_t LSM303C_MAG_R_BLE(void *handle, LSM303C_MAG_BLE_t *value)
{
 if( !LSM303C_MAG_ReadReg(handle, LSM303C_CTRL_REG4_M, (u8_t *)value) )
    return MEMS_ERROR;

  *value &= LSM303C_MAG_BLE_MASK; //mask

  return MEMS_SUCCESS;
}
/*******************************************************************************
* Function Name  : LSM303C_MAG_W_BDU
* Description    : Write BDU
* Input          : LSM303C_MAG_BDU_t
* Output         : None
* Return         : Status [MEMS_ERROR, MEMS_SUCCESS]
*******************************************************************************/
status_t  LSM303C_MAG_W_BDU(void *handle, LSM303C_MAG_BDU_t newValue)
{
  u8_t value;

  if( !LSM303C_MAG_ReadReg(handle, LSM303C_CTRL_REG5_M, &value) )
    return MEMS_ERROR;

  value &= ~LSM303C_MAG_BDU_MASK; 
  value |= newValue;
  
  if( !LSM303C_MAG_WriteReg(handle, LSM303C_CTRL_REG5_M, value) )
    return MEMS_ERROR;

  return MEMS_SUCCESS;
}

/*******************************************************************************
* Function Name  : LSM303C_MAG_R_BDU
* Description    : Read BDU
* Input          : Pointer to LSM303C_MAG_BDU_t
* Output         : Status of BDU see LSM303C_MAG_BDU_t
* Return         : Status [MEMS_ERROR, MEMS_SUCCESS]
*******************************************************************************/

status_t LSM303C_MAG_R_BDU(void *handle, LSM303C_MAG_BDU_t *value)
{
 if( !LSM303C_MAG_ReadReg(handle, LSM303C_CTRL_REG5_M, (u8_t *)value) )
    return MEMS_ERROR;

  *value &= LSM303C_MAG_BDU_MASK; //mask

  return MEMS_SUCCESS;
}

/*******************************************************************************
* Function Name  : LSM303C_MAG_W_I2C_DIS
* Description    : Write I2C_DIS
* Input          : LSM303C_MAG_I2C_DIS_t
* Output         : None
* Return         : Status [MEMS_ERROR, MEMS_SUCCESS]
*******************************************************************************/
status_t  LSM303C_MAG_W_I2C_DIS(void *handle, LSM303C_MAG_I2C_DIS_t newValue)
{
  u8_t value;

  if( !LSM303C_MAG_ReadReg(handle, LSM303C_CTRL_REG3_M, &value) )
    return MEMS_ERROR;

  value &= ~LSM303C_MAG_I2C_DIS_MASK; 
  value |= newValue;
  
  if( !LSM303C_MAG_WriteReg(handle, LSM303C_CTRL_REG3_M, value) )
    return MEMS_ERROR;

  return MEMS_SUCCESS;
}

/*******************************************************************************
* Function Name  : LSM303C_MAG_R_I2C_DIS
* Description    : Read I2C_DIS
* Input          : Pointer to LSM303C_MAG_I2C_DIS_t
* Output         : Status of I2C_DIS see LSM303C_MAG_I2C_DIS_t
* Return         : Status [MEMS_ERROR, MEMS_SUCCESS]
*******************************************************************************/

status_t LSM303C_MAG_R_I2C_DIS(void *handle, LSM303C_MAG_I2C_DIS_t *value)
{
 if( !LSM303C_MAG_ReadReg(handle, LSM303C_CTRL_REG3_M, (u8_t *)value) )
    return MEMS_ERROR;

  *value &= LSM303C_MAG_I2C_DIS_MASK; //mask

  return MEMS_SUCCESS;
}

/*******************************************************************************
* Function Name  : LSM303C_MAG_R_XDA
* Description    : Read XDA
* Input          : Pointer to LSM303C_MAG_XDA_t
* Output         : Status of XDA see LSM303C_MAG_XDA_t
* Return         : Status [MEMS_ERROR, MEMS_SUCCESS]
*******************************************************************************/

status_t LSM303C_MAG_R_XDA(void *handle, LSM303C_MAG_XDA_t *value)
{
 if( !LSM303C_MAG_ReadReg(handle, LSM303C_STATUS_REG_M, (u8_t *)value) )
    return MEMS_ERROR;

  *value &= LSM303C_MAG_XDA_MASK; //mask

  return MEMS_SUCCESS;
}
/*******************************************************************************
* Function Name  : LSM303C_MAG_R_YDA
* Description    : Read YDA
* Input          : Pointer to LSM303C_MAG_YDA_t
* Output         : Status of YDA see LSM303C_MAG_YDA_t
* Return         : Status [MEMS_ERROR, MEMS_SUCCESS]
*******************************************************************************/

status_t LSM303C_MAG_R_YDA(void *handle, LSM303C_MAG_YDA_t *value)
{
 if( !LSM303C_MAG_ReadReg(handle, LSM303C_STATUS_REG_M, (u8_t *)value) )
    return MEMS_ERROR;

  *value &= LSM303C_MAG_YDA_MASK; //mask

  return MEMS_SUCCESS;
}
/*******************************************************************************
* Function Name  : LSM303C_MAG_R_ZDA
* Description    : Read ZDA
* Input          : Pointer to LSM303C_MAG_ZDA_t
* Output         : Status of ZDA see LSM303C_MAG_ZDA_t
* Return         : Status [MEMS_ERROR, MEMS_SUCCESS]
*******************************************************************************/

status_t LSM303C_MAG_R_ZDA(void *handle, LSM303C_MAG_ZDA_t *value)
{
 if( !LSM303C_MAG_ReadReg(handle, LSM303C_STATUS_REG_M, (u8_t *)value) )
    return MEMS_ERROR;

  *value &= LSM303C_MAG_ZDA_MASK; //mask

  return MEMS_SUCCESS;
}
/*******************************************************************************
* Function Name  : LSM303C_MAG_R_ZYXDA
* Description    : Read ZYXDA
* Input          : Pointer to LSM303C_MAG_ZYXDA_t
* Output         : Status of ZYXDA see LSM303C_MAG_ZYXDA_t
* Return         : Status [MEMS_ERROR, MEMS_SUCCESS]
*******************************************************************************/

status_t LSM303C_MAG_R_ZYXDA(void *handle, LSM303C_MAG_ZYXDA_t *value)
{
 if( !LSM303C_MAG_ReadReg(handle, LSM303C_STATUS_REG_M, (u8_t *)value) )
    return MEMS_ERROR;

  *value &= LSM303C_MAG_ZYXDA_MASK; //mask

  return MEMS_SUCCESS;
}
/*******************************************************************************
* Function Name  : LSM303C_MAG_R_XOR
* Description    : Read XOR
* Input          : Pointer to LSM303C_MAG_XOR_t
* Output         : Status of XOR see LSM303C_MAG_XOR_t
* Return         : Status [MEMS_ERROR, MEMS_SUCCESS]
*******************************************************************************/

status_t LSM303C_MAG_R_XOR(void *handle, LSM303C_MAG_XOR_t *value)
{
 if( !LSM303C_MAG_ReadReg(handle, LSM303C_STATUS_REG_M, (u8_t *)value) )
    return MEMS_ERROR;

  *value &= LSM303C_MAG_XOR_MASK; //mask

  return MEMS_SUCCESS;
}
/*******************************************************************************
* Function Name  : LSM303C_MAG_R_YOR
* Description    : Read YOR
* Input          : Pointer to LSM303C_MAG_YOR_t
* Output         : Status of YOR see LSM303C_MAG_YOR_t
* Return         : Status [MEMS_ERROR, MEMS_SUCCESS]
*******************************************************************************/

status_t LSM303C_MAG_R_YOR(void *handle, LSM303C_MAG_YOR_t *value)
{
 if( !LSM303C_MAG_ReadReg(handle, LSM303C_STATUS_REG_M, (u8_t *)value) )
    return MEMS_ERROR;

  *value &= LSM303C_MAG_YOR_MASK; //mask

  return MEMS_SUCCESS;
}
/*******************************************************************************
* Function Name  : LSM303C_MAG_R_ZOR
* Description    : Read ZOR
* Input          : Pointer to LSM303C_MAG_ZOR_t
* Output         : Status of ZOR see LSM303C_MAG_ZOR_t
* Return         : Status [MEMS_ERROR, MEMS_SUCCESS]
*******************************************************************************/

status_t LSM303C_MAG_R_ZOR(void *handle, LSM303C_MAG_ZOR_t *value)
{
 if( !LSM303C_MAG_ReadReg(handle, LSM303C_STATUS_REG_M, (u8_t *)value) )
    return MEMS_ERROR;

  *value &= LSM303C_MAG_ZOR_MASK; //mask

  return MEMS_SUCCESS;
}
/*******************************************************************************
* Function Name  : LSM303C_MAG_R_ZYXOR
* Description    : Read ZYXOR
* Input          : Pointer to LSM303C_MAG_ZYXOR_t
* Output         : Status of ZYXOR see LSM303C_MAG_ZYXOR_t
* Return         : Status [MEMS_ERROR, MEMS_SUCCESS]
*******************************************************************************/

status_t LSM303C_MAG_R_ZYXOR(void *handle, LSM303C_MAG_ZYXOR_t *value)
{
 if( !LSM303C_MAG_ReadReg(handle, LSM303C_STATUS_REG_M, (u8_t *)value) )
    return MEMS_ERROR;

  *value &= LSM303C_MAG_ZYXOR_MASK; //mask

  return MEMS_SUCCESS;
}
/*******************************************************************************
* Function Name  : status_t LSM303C_MAG_Get_Raw_Magnetic(u8_t *buff)
* Description    : Read Magnetic output register
* Input          : pointer to [u8_t]
* Output         : Magnetic buffer u8_t
* Return         : Status [MEMS_ERROR, MEMS_SUCCESS]
*******************************************************************************/
status_t LSM303C_MAG_Get_Raw_Magnetic(void *handle, u8_t *buff) 
{
  u8_t i, j, k;
  u8_t numberOfByteForDimension;
  
  numberOfByteForDimension=6/3;

  k=0;
  for (i=0; i<3;i++ ) 
  {
    for (j=0; j<numberOfByteForDimension;j++ )
    {   
        if( !LSM303C_MAG_ReadReg(handle, LSM303C_OUT_X_L_M+k, &buff[k]))
          return MEMS_ERROR;
        k++;    
    }
  }

  return MEMS_SUCCESS; 
}

#define LSM303C_MAG_SENSITIVITY   15/10

status_t LSM303C_MAG_Get_Magnetic(void *handle, int *buff)
{
  Type3Axis16bit_U raw_data_tmp;

  /* Read out raw magnetometer samples */
  if(!LSM303C_MAG_Get_Raw_Magnetic(handle, raw_data_tmp.u8bit)) {
    return MEMS_ERROR;
  }

  /* Applysensitivity */
  buff[0] = raw_data_tmp.i16bit[0] * LSM303C_MAG_SENSITIVITY;
  buff[1] = raw_data_tmp.i16bit[1] * LSM303C_MAG_SENSITIVITY;
  buff[2] = raw_data_tmp.i16bit[2] * LSM303C_MAG_SENSITIVITY;

  return MEMS_SUCCESS;
}
