#ifndef DBG_H
#define DBG_H

#include "mbed.h"
#include "WizziCom.h"

// Enable the debug prints.
#define __PRINT_ENABLED__

// Reboot the device on fatal error, else trap the processor.
#define __REBOOT_ON_ASSERT__

// Flush the debug queue if full to avoid losing traces (can impact performances)
//#define __FLUSH_IF_FULL__

// Force flush until an EOL character ('\n')
//#define __FORCE_LINE_INTEGRITY__

// Force flush, do not bufferise trace
//#define __FORCE_FLUSH__

#ifdef __PRINT_ENABLED__
    #define DBG_OPEN(...)   dbg_open(__VA_ARGS__)
    #define FLUSH()         dbg_flush()

// Enable or disable trace levels
    #define _USER_PRINTS_
    #define _PRINT_ERRORS_
    #define _PRINT_WARNINGS_
    #define _PRINT_INFOS_
    //#define _PRINT_DEBUG_
    //#define _PRINT_FUNCTIONS_ // Print the function name where FPRINT is called
    //#define _TRACE_MALLOC_
    
#else
    #define DBG_OPEN(led)   dbg_set_led(led)
    #define FLUSH();
#endif


#define RAM_START_ADDRESS   (0x20000000)
#define RAM_END_ADDRESS     (0x40000000)

#define ASSERT(...)         dbg_assert(__VA_ARGS__)


#ifdef _USER_PRINTS_
    #define PRINT(...)      dbg_print(__VA_ARGS__)
    #define PRINT_DATA(...) dbg_print_data(__VA_ARGS__)
#else
    #define PRINT(...);
    #define PRINT_DATA(...);
#endif


#ifdef _PRINT_ERRORS_
    #define EPRINT(format,...)     dbg_print("ERROR: "format, __VA_ARGS__)
#else
    #define EPRINT(...);
#endif


#ifdef _PRINT_WARNINGS_
    #define WARNING(w,format,...)  do { if(!(w)) { dbg_print("WARNING: "format, __VA_ARGS__); } } while(0)
#else
    #define WARNING(...);
#endif


#ifdef _PRINT_FUNCTIONS_
    #define FPRINT(format,...)    dbg_print("%s"format,__FUNCTION__,##__VA_ARGS__)
#else
    #define FPRINT(...);
#endif


#ifdef _PRINT_INFOS_
    #define IPRINT(...)         dbg_print("INFO: "__VA_ARGS__)
#else
    #define IPRINT(...);
#endif


#ifdef _PRINT_DEBUG_
    #define DPRINT(...)         dbg_print(__VA_ARGS__)
    #define DPRINT_DATA(...)    dbg_print_data(__VA_ARGS__)
#else
    #define DPRINT(...);
    #define DPRINT_DATA(...);
#endif


#ifdef _TRACE_MALLOC_
    #define MALLOC(s)       dbg_malloc((s),__FUNCTION__,__LINE__);FLUSH();
    #define REALLOC(p,s)    dbg_realloc((p),(s),__FUNCTION__,__LINE__);FLUSH();
    #define FREE(p)         dbg_free((p),__FUNCTION__,__LINE__);FLUSH();
#else
    #define MALLOC(s)       malloc(s)
    #define REALLOC(p,s)    realloc(p,s)
    #define FREE(p)         free(p)
#endif

void dbg_open(PinName led = NC, PinName tx = USBTX, PinName rx = USBRX);
void dbg_close( void );

void dbg_set_led(PinName led = NC, PinName dummy1 = NC, PinName dummy2 = NC);

/**
    Asserts and trap processor.

    @param bool                 Assert condition
    @param const char*          Assert message format
    @param ...                  Assert message parameters
    @return void
*/
void dbg_assert(bool test, const char* format, ...);

/**
    Prints a message to the debug port.

    @param const char*          Message format
    @param ...                  Message parameters
    @return void
*/
void dbg_print(const char* format, ...);

/**
    Flushes the pending debug messages.

    @param void
    @return void
*/
void dbg_flush( void );

/**
    Prints malloc parameters.

    @param size_t               Size to allocate
    @param const char*          Current function name
    @param uint32_t             Current file line
    @return void*               Pointer to allocated memory
*/
void* dbg_malloc(size_t size, const char* f, uint32_t l);

/**
    Prints realloc parameters.

    @param void*                Pointer to reallocate
    @param size_t               New size to allocate
    @param const char*          Current function name
    @param uint32_t             Current file line
    @return void*               Pointer to allocated memory
*/
void* dbg_realloc(void* p, size_t size, const char* f, uint32_t l);

/**
    Prints free parameters.

    @param void*                Pointer to memory to free
    @param const char*          Current function name
    @param uint32_t             Current file line
    @return void
*/
void dbg_free(void* p, const char* f, uint32_t l);

void dbg_print_data(const char* before, const char* format, uint8_t* data, uint8_t length, const char* after);

void dbg_set_rx_callback(void (*rx_isr)(char c));

#endif // DBG_H