#include "WizziComModem.h"

//======================================================================
// MODEM commands over COM port
//======================================================================
typedef enum
{    //============================================
    // COMMANDS
    //============================================
    // Soft-reset Modem Device
    // ARGS: none
    WM_CMD_RST                      = 0x01,

    // Start D7A communication stack
    // ARGS: none TODO: pass number of fifos ?
    WM_CMD_D7A_STACK_START          = 0x02,

    // Stop D7A communication stack
    // ARGS: none
    WM_CMD_D7A_STACK_STOP           = 0x03,

    // Register a File for notification
    // ARGS: register_file_param_t
    WM_CMD_REGISTER_FILE            = 0x04,

    // Notify a (previously registered) file
    // ARGS: notify_file_param_t
    WM_CMD_NOTIFY_FILE              = 0x05,

    // Command that will be instantly answered with WM_OK
    // ARGS: none
    WM_CMD_READY                    = 0x06,

    //============================================
    // RESPONSES
    //============================================
    // Send by the Modem Device after a command has been executed/taken
    // into account, and the Modem is ready to handle a new command.
    // ARGS: none
    WM_OK                           = 0x80,

    // Send by the Modem Device after a (re)boot.
    // ARGS: boot_status_t
    WM_BOOT                         = 0x81,

    // Send by the Modem Device in case of failing command
    // ARGS: wm_error_code_t (u8)
    WM_ERROR                        = 0x82,

    // Send by the Modem Device upon the end of a File notification.
    // This response is generated depending on ALP's Retry-Policy ('respond' field set)
    // ARGS: wm_notif_result_t
    WM_NOTIF_DONE                   = 0x83,
} wm_cmd_t;


WizziComModem::WizziComModem(WizziCom* com, PinName reset_pin) :
_com(com),
_reset_pin(reset_pin),
_modem_ready(0)
{
    _com->attach(this, &WizziComModem::_modem_command, WizziComPacketCmd);
}

WizziComModem::~WizziComModem()
{}

void WizziComModem::_modem_command(WizziCom* com, WizziComPacket_t* pkt)
{
    uint8_t cmd = pkt->data[0];
    
    if (cmd == WM_OK)
    {
        PRINT("Modem ready\r\n");
        _modem_ready.release();
    }
    else if (cmd == WM_BOOT)
    {
        boot_status_t* bs = (boot_status_t*)&(pkt->data[1]);
        if (_boot_callback)
        {
            _boot_callback.call(*bs);
        }
        //WARNING(false, "Modem booted CAUSE:%d NB_BOOT:%d\r\n", bs->bf.cause, bs->bf.nb_boot);
    }
    else if (cmd == WM_ERROR)
    {
        if (_error_callback)
        {
            _error_callback.call((int8_t)pkt->data[1]);
        }
    }
    else if (cmd == WM_NOTIF_DONE)
    {
        PRINT("Notif done fid %d err %d\r\n", pkt->data[1], pkt->data[2]);
    }
    else
    {
        EPRINT("MODEM Unknown cmd %d\r\n", cmd);
    }
    
    FREE(pkt);
}

void WizziComModem::start_dash7(void)
{
    uint8_t cmd[] = { WM_CMD_D7A_STACK_START };
    _com->send(WizziComPacketCmd, sizeof(cmd), cmd);
    _modem_ready.wait();
}

void WizziComModem::stop_dash7(void)
{
    uint8_t cmd[] = { WM_CMD_D7A_STACK_STOP };
    _com->send(WizziComPacketCmd, sizeof(cmd), cmd);
    _modem_ready.wait();
}

void WizziComModem::software_reset(void)
{
    uint8_t cmd[] = { WM_CMD_RST };
    _com->send(WizziComPacketCmd, sizeof(cmd), cmd);
    _modem_ready.wait();
}

void WizziComModem::hardware_reset(void)
{
    DigitalOut reset_low(_reset_pin, 0);
    Thread::wait(100);
    DigitalIn reset_release(_reset_pin);
    _modem_ready.wait();
}

void WizziComModem::attach_boot(WizziComModemBootCallback function)
{
    _boot_callback = callback(function);
}

void WizziComModem::attach_error(WizziComModemErrorCallback function)
{
    _error_callback = callback(function);
}