/// @copyright
/// ========================================================================={{{
/// Copyright (c) 2013-2021 WizziLab                                           /
/// All rights reserved                                                        /
///                                                                            /
/// IMPORTANT: This Software may not be modified, copied or distributed unless /
/// embedded on a WizziLab product. Other than for the foregoing purpose, this /
/// Software and/or its documentation may not be used, reproduced, copied,     /
/// prepared derivative works of, modified, performed, distributed, displayed  /
/// or sold for any purpose. For the sole purpose of embedding this Software   /
/// on a WizziLab product, copy, modification and distribution of this         /
/// Software is granted provided that the following conditions are respected:  /
///                                                                            /
/// *  Redistributions of source code must retain the above copyright notice,  /
///    this list of conditions and the following disclaimer                    /
///                                                                            /
/// *  Redistributions in binary form must reproduce the above copyright       /
///    notice, this list of conditions and the following disclaimer in the     /
///    documentation and/or other materials provided with the distribution.    /
///                                                                            /
/// *  The name of WizziLab can not be used to endorse or promote products     /
///    derived from this software without specific prior written permission.   /
///                                                                            /
/// THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS        /
/// "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED  /
/// TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR /
/// PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR          /
/// CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,      /
/// EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,        /
/// PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA,            /
/// OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY     /
/// OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING    /
/// NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS         /
/// SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.               /
/// WIZZILAB HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT, UPDATES,       /
/// ENHANCEMENTS OR MODIFICATIONS.                                             /
///                                                                            /
/// Should you have any questions regarding your right to use this Software,   /
/// contact WizziLab at www.wizzilab.com.                                      /
///                                                                            /
/// =========================================================================}}}
/// @endcopyright

//  =======================================================================
/// @file           kal_buf.c
/// @brief          Buffers header
//  =======================================================================

#ifndef __KAL_BUF_CIRC_H__
#define __KAL_BUF_CIRC_H__

#include "mbed.h"

// define to create a static Circular buffer buffer
#define kal_buf_circ_static_buffer_t(_name, _nb_elements, _element_size) uint8_t _name[(_nb_elements+1)*_element_size]

typedef struct {
    uint8_t* buffer;
    volatile uint32_t head;
    volatile uint32_t tail;
    uint32_t element_size;
    uint32_t max_length;
    uint8_t is_static;
} kal_buf_circ_static_handle_t;

typedef void* kal_buf_circ_handle_t;

//--------------------
/// @brief Dynamically creates a Circular buffer
/// @param uint32_t                          Maximum number of elements in the buffer
/// @param uint32_t                          Size of each element
/// @retval kal_buf_circ_handle_t            Circular buffer handle (NULL if create failed)
//--------------------
kal_buf_circ_handle_t kal_buf_circ_create(uint32_t nb_elements, uint32_t element_size);

//--------------------
/// @brief Statically creates a Circular buffer
/// @param kal_buf_circ_StaticHandle_t*      Pointer to the static handle
/// @param uint8_t*                          Static buffer created with kal_buf_circ_StaticBuffer_t macro
/// @param uint32_t                          Maximum number of elements in the buffer
/// @param uint32_t                          Size of each element
/// @retval int                              0 if function is successful, -1 otherwise
//--------------------
int kal_buf_circ_create_static(kal_buf_circ_static_handle_t* static_handle, uint8_t* buffer, uint32_t nb_elements, uint32_t element_size);

//--------------------
/// @brief Deletes a Circular buffer (Static or Dynamic)
/// @param kal_buf_circ_handle_t             Handle to the Circular buffer
/// @retval int                              0 if function is successful, -1 otherwise
//--------------------
int kal_buf_circ_delete(kal_buf_circ_handle_t handle);

//--------------------
/// @brief Empty the Circular buffer
/// @param kal_buf_circ_handle_t             Handle to the Circular buffer
/// @retval int                              0 if function is successful, -1 otherwise
//--------------------
int kal_buf_circ_reset(kal_buf_circ_handle_t handle);

//--------------------
/// @brief Check if buffer is full
/// @param kal_buf_circ_handle_t             Handle to the Circular buffer
/// @retval int                              TRUE if buffer is full, FALSE otherwise
//--------------------
int kal_buf_circ_full(kal_buf_circ_handle_t handle);

//--------------------
/// @brief Check if buffer is empty
/// @param kal_buf_circ_handle_t             Handle to the Circular buffer
/// @retval int                              TRUE if buffer is empty, FALSE otherwise
//--------------------
int kal_buf_circ_empty(kal_buf_circ_handle_t handle);

//--------------------
/// @brief Returns the number of elements currently in the buffer
/// @param kal_buf_circ_handle_t             Handle to the Circular buffer
/// @retval uint32_t                         Number of elements in the buffer
//--------------------
uint32_t kal_buf_circ_size(kal_buf_circ_handle_t handle);

//--------------------
/// @brief Returns the number of free elements in the buffer
/// @param kal_buf_circ_handle_t             Handle to the Circular buffer
/// @retval uint32_t                         Number of elements in the buffer
//--------------------
uint32_t kal_buf_circ_space(kal_buf_circ_handle_t handle);

//--------------------
/// @brief Adds an element to the buffer
/// @param kal_buf_circ_handle_t             Handle to the Circular buffer
/// @param uint8_t*                          Pointer to the element to push in the buffer
/// @retval int                              0 if function is successful, -1 otherwise (buffer full)
//--------------------
int kal_buf_circ_push(kal_buf_circ_handle_t handle, uint8_t* p);

//--------------------
/// @brief Gets an element from the buffer
/// @param kal_buf_circ_handle_t             Handle to the Circular buffer
/// @param uint8_t*                          Pointer to the element to get from the buffer, can be NULL to discard data
/// @retval int                              0 if function is successful, -1 otherwise (buffer empty)
//--------------------
int kal_buf_circ_pop(kal_buf_circ_handle_t handle, uint8_t* p);

//--------------------
/// @brief Gets an element from the buffer without removing it from the buffer
/// @param kal_buf_circ_handle_t             Handle to the Circular buffer
/// @param uint8_t*                          Pointer to the element to get from the buffer
/// @retval int                              0 if function is successful, -1 otherwise (buffer empty)
//--------------------
int kal_buf_circ_peek(kal_buf_circ_handle_t handle, uint8_t* p);

//--------------------
/// @brief Write several elements to the buffer
/// @param kal_buf_circ_handle_t             Handle to the Circular buffer
/// @param uint8_t*                          Pointer to the element array to write to the buffer
/// @param uint32_t                          Number of elements to write
/// @retval int                              0 if function is successful, -1 otherwise
//--------------------
int kal_buf_circ_put(kal_buf_circ_handle_t handle, uint8_t* p, uint32_t nb_elements);

//--------------------
/// @brief Gets several elements from the buffer
/// @param kal_buf_circ_handle_t             Handle to the Circular buffer
/// @param uint8_t*                          Pointer to the element array to get from the buffer, can be NULL to discard data
/// @param uint32_t                          Number of elements to get
/// @retval int                              0 if function is successful, -1 otherwise
//--------------------
int kal_buf_circ_get(kal_buf_circ_handle_t handle, uint8_t* p, uint32_t nb_elements);

//--------------------
/// @brief Gets several elements from the buffer without removing them from the buffer
/// @param kal_buf_circ_handle_t             Handle to the Circular buffer
/// @param uint8_t*                          Pointer to the element array to get from the buffer
/// @param uint32_t                          Number of elements to get
/// @retval int                              0 if function is successful, -1 otherwise
//--------------------
int kal_buf_circ_fetch(kal_buf_circ_handle_t handle, uint8_t* p, uint32_t nb_elements);

//--------------------
/// @brief Erase the last element written to the buffer
/// @param kal_buf_circ_handle_t             Handle to the Circular buffer
/// @retval int                              0 if function is successful, -1 otherwise (buffer empty)
//--------------------
int kal_buf_circ_erase(kal_buf_circ_handle_t handle);

#endif // __KAL_BUF_CIRC_H__