// @autor: jeremie@wizzilab.com
// @date: 2017-05-02

#include "modem_ref_helper.h"
#include "modem_callbacks.h"
#include "files.h"
#include "sensor.h"

#define MIN_REPORT_PERIOD   (10) // Seconds

Semaphore modem_ready(0);
sensor_config_t g_light_config;
Queue<void, 8> g_file_modified;

// Callback for id User
void my_main_callback(uint8_t terminal, int8_t err, uint8_t id)
{
    UNUSED(id);
    
    if (ALP_ERR_NONE > err)
    {
        modem_print_error(ALP_ITF_TYPE_D7A, err);
    }
    
    if (terminal)
    {
        modem_ready.release();
    }
}

static bool report_ok(uint32_t last_report_time)
{
    // Do not send a report if it's been less than MIN_REPORT_PERIOD since the last report
    if ((last_report_time/1000) < MIN_REPORT_PERIOD)
    {
        PRINT("Report Skipped, next in %ds min\n", MIN_REPORT_PERIOD - (last_report_time/1000));
        return false;
    }
    
    return true;
}

// Check parameters to see if data should be send
static bool report_needed(sensor_config_t* config, int32_t value, int32_t last_value, uint32_t last_report_time, uint8_t id)
{
    switch (config->report_type)
    {
        case REPORT_ALWAYS:
            // Send a report at each measure
            PRINT("Report[%d] always\r\n", id);
            return report_ok(last_report_time);
        case REPORT_ON_DIFFERENCE:
            // Send a report when the difference between the last reported measure and the current mesure is greater than max_diff
            if (abs(last_value - value) >= config->max_diff && config->max_diff)
            {
                PRINT("Report[%d] on difference (last:%d new:%d max_diff:%d)\r\n", id, last_value, value, config->max_diff);
                return report_ok(last_report_time);
            }
            break;
        case REPORT_ON_THRESHOLD:
            // Send a report when crossing a threshold
            if (   (value >= config->threshold_high && last_value < config->threshold_high)
                || (value <= config->threshold_low  && last_value > config->threshold_low)
                || (value < config->threshold_high  && last_value >= config->threshold_high)
                || (value > config->threshold_low   && last_value <= config->threshold_low))
            {
                PRINT("Report[%d] on threshold (last:%d new:%d th:%d tl:%d)\r\n", id, last_value, value, config->threshold_high, config->threshold_low);
                return report_ok(last_report_time);
            }
            break;
        default:
            break;
    }
    
    // Send a report if it's been more than max_period since the last report
    if (((last_report_time/1000) >= config->max_period) && config->max_period)
    {
        PRINT("Report[%d] on period (max_period:%d time:%d)\r\n", id, config->max_period, last_report_time);
        return report_ok(last_report_time);
    }

    return false;
}

void thread_sensor_light()
{
    light_value_t light_level;
    light_value_t light_level_old = 0;
    
    // To force a first report
    uint32_t last_report_time = 0xFFFFFFFF;
    uint8_t id = modem_get_id(my_main_callback);
    
    FPRINT("(id:0x%08x)\r\n", osThreadGetId());
    
    // Get the sensor configuration
    ram_fs_read(FID_SENSOR_CONFIG, 0, SIZE_SENSOR_CONFIG, (uint8_t*)&g_light_config);
    
    while (true)
    {
        light_level = sensor_get_light();
        
        PRINT("Light %d\r\n", light_level);
                
        if (report_needed(&g_light_config, light_level, light_level_old, last_report_time, id))
        {
            PRINT("Light report %d\r\n", light_level);
        
            // Send notification
            modem_write_file(FID_SENSOR_LIGHT, &light_level, 0, SIZE_SENSOR_LIGHT, id);
            modem_ready.acquire();
        
            // Update 
            light_level_old = light_level;
            last_report_time = 0;
        }
        
        // Update last report time
        last_report_time += g_light_config.read_period;
        
        ThisThread::sleep_for(g_light_config.read_period);
    }
}

void thread_file_modified()
{
    uint8_t fid;
    osEvent evt;
    
    while (true)
    {
        evt = g_file_modified.get();
        fid = (evt.status == osEventMessage)? (uint8_t)(uint32_t)evt.value.p : NULL;
        
        switch (fid)
        {
            case FID_SENSOR_CONFIG:
                // Update sensor configuration
                ram_fs_read(FID_SENSOR_CONFIG, 0, SIZE_SENSOR_CONFIG, (uint8_t*)&g_light_config);
                PRINT("Sensor configuration updated\r\n");
                break;
            default:
            break;
        }
    }
}

// Misc
// ============================================================{{{

void my_get_alp_file_props(uint8_t fid, alp_file_header_t* hdr)
{
    memcpy(hdr, ram_fs_get_header(fid), sizeof(alp_file_header_t));
}

modem_callbacks_t callbacks = {
    .read       = my_read,
    .write      = my_write,
    .read_fprop = my_read_fprop,
    .flush      = my_flush,
    .remove     = my_delete,
    .udata      = my_udata,
    .lqual      = my_lqual,
    .ldown      = my_ldown,
    .reset      = my_reset,
    .boot       = my_boot,
    .busy       = my_busy,
};

/*** Main function ------------------------------------------------------------- ***/
int main()
{
    // Start & initialize
#ifdef DEBUG_LED
    DBG_OPEN(DEBUG_LED);
#else
    DBG_OPEN(NC);
#endif
    PRINT("\n"
          "-----------------------------------------\n"
          "---------- Demo send and forget ---------\n"
          "-----------------------------------------\n");
    
    modem_helper_open(&callbacks);
    
    uint8_t id = modem_get_id(my_main_callback);
    
    DPRINT("Register Files\n");
        
    // Create report file on modem.
    modem_update_file(FID_SENSOR_LIGHT, (alp_file_header_t*)&h_sensor_light, NULL);

    // Allow remote access.
    modem_update_file(FID_SENSOR_CONFIG, (alp_file_header_t*)&h_sensor_config, (uint8_t*)&f_sensor_config);

    // Configure URC: LQUAL on report file notification every 10 reports
    PRINT("Setup URCs\n");
    modem_enable_urc(ALP_URC_TYPE_LQUAL, IFID_REPORT, 10, true, id);
    modem_ready.acquire();
    
    PRINT("Start D7A Stack\n");
    modem_activate_itf(ALP_ITF_TYPE_D7A, 24, 0, ALP_D7A_ISTAT_RESP | ALP_D7A_ISTAT_UNS | ALP_D7A_ISTAT_EOP, true, id);
    modem_ready.acquire();
    
    PRINT("Notify Modem Version\n");
    modem_notify_file(D7A_FID_FIRMWARE_VERSION, 0, SIZE_HOST_REV, id);
    modem_ready.acquire();
    
    PRINT("Notify FW Version\n");
    uint8_t default_root_key[16] = DEFAULT_ROOT_KEY;
    modem_notify_host_rev(&f_rev, &h_rev, default_root_key);
    
    modem_free_id(id);
    
    // Start file modified thread
    Thread th_file_modified(osPriorityNormal, 1024, NULL);
    osStatus status = th_file_modified.start(thread_file_modified);
    ASSERT(status == osOK, "Failed to start thread_file_modified (err: %d)\r\n", status);
    
    // Start light measure thread
    Thread th_sensor_light(osPriorityNormal, 1024, NULL);
    status = th_sensor_light.start(thread_sensor_light);
    ASSERT(status == osOK, "Failed to start thread_sensor_light (err: %d)\r\n", status);

#ifdef DEBUG_LED
    DigitalOut my_led(DEBUG_LED);
#endif
    
    // Set main task to lowest priority
    osThreadSetPriority(osThreadGetId(), osPriorityLow);
    while(true)
    {
        ThisThread::sleep_for(500);
#ifdef DEBUG_LED
        my_led = !my_led;
#endif
    }
}