// @autor: jeremie@wizzilab.com
// @date: 2017-05-02

#include "modem_d7a.h"
#include "modem_callbacks.h"
#include "files.h"
#include "crc.h"
#include "CriusOLED.h"

#define RESTORE_TIME    30
#define KEEP_ALIVE      3600
#define MIN_REPORT_PERIOD   (10) // Seconds
#define CHUNK_SIZE              128

Queue<touch_t, 8> g_file_modified;

// This describe the upload interface
// Do not modify uncommented parameters
alp_itf_d7a_cfg_t report_itf = {
    .type                           = ALP_ITF_TYPE_D7A,
    .cfg.to.byte                    = D7A_CTF_ENCODE(0),
    .cfg.te.byte                    = D7A_CTF_ENCODE(0),
    .cfg.qos.bf.resp                = D7A_RESP_PREFERRED,
    .cfg.qos.bf.retry               = ALP_RPOL_ONESHOT,
    .cfg.addressee.ctrl.bf.nls      = D7A_NLS_AES_CCM_64, // Security level
    .cfg.addressee.ctrl.bf.idf      = D7A_ID_NBID,
    .cfg.addressee.xcl.bf           = {.s = 2, .m = 0x1}, // Gateway access class
    .cfg.addressee.id[0]            = D7A_CTF_ENCODE(4),
};


uint8_t id;
uint32_t g_restore_time = RESTORE_TIME + 1;

void display_logo(void)
{
    clear_display();
    sendImage();
}


void thread_reset_display()
{
    FPRINT("(id:0x%08x)\r\n", osThreadGetId());
    
    uint32_t time = 0;
    uint8_t status = 0;
    
    while (true)
    {
        if (g_restore_time++ == RESTORE_TIME)
        {
            display_logo();
        }
       
        if ((time++ % KEEP_ALIVE) == 0)
        {
            // Keep alive: Report status
            modem_notify_file(FID_STATUS_FILE, 0, SIZE_STATUS_FILE);
        }
        
        Thread::wait(1000);
    }
}

void thread_file_modified()
{
    touch_t* touch;
    
    FPRINT("(id:0x%08x)\r\n", osThreadGetId());
    
    uint8_t fid;
    osEvent evt;
    
    int err;
    alp_payload_t* alp;
    alp_payload_t* rsp;
    
    PRINT("Register Files\n");
    ram_fs_new(FID_STRING_FILE, (uint8_t*)&h_string_file, (uint8_t*)&f_string_file);
    ram_fs_new(FID_STATUS_FILE, (uint8_t*)&h_status_file, (uint8_t*)&f_status_file);
    
    modem_declare_file(FID_STRING_FILE, (alp_file_header_t*)&h_string_file);
    modem_declare_file(FID_STATUS_FILE, (alp_file_header_t*)&h_status_file);
    
    PRINT("Enable D7A interface\n");
    modem_d7a_enable_itf();
    
    // Host revision file is in the modem. Update it.
    PRINT("Update host revision\n");
    modem_write_file(FID_HOST_REV, (void*)&f_rev, 0, sizeof(revision_t));
    
    
    // Retrieve modem revision
    PRINT("Send revision\n");
    
    revision_t rev;
    
    modem_read_file(FID_WM_REV, &rev, 0, sizeof(revision_t));
    
    // Send both to the server
    // Build payload
    alp = NULL;
    alp = alp_payload_rsp_f_data(alp, FID_WM_REV, &rev, 0, sizeof(revision_t));
    alp = alp_payload_rsp_f_data(alp, FID_HOST_REV, (void*)&f_rev, 0, sizeof(revision_t));
    // Send
    modem_remote_raw_alp((void*)&report_itf, alp, NULL, 10000);
    
    
    while (true)
    {
        evt = g_file_modified.get();
        
        touch = (evt.status == osEventMessage)? (touch_t*)evt.value.p : NULL;
        ASSERT(touch != NULL, "NULL touch pointer!\n");
        
        switch (touch->fid)
        {
            case FID_STRING_FILE:
                // Print string
                uint8_t str[SIZE_STRING_FILE];
                ram_fs_read(FID_STRING_FILE, (uint8_t*)&str, 0, SIZE_STRING_FILE);
                PRINT("STRING: ");
                PRINT((char*)str);
                PRINT("\r\n");
                
                clear_display();
                sendStrXY((char*)str,0,0);
                
                g_restore_time = 0;
                break;
            default:
                PRINT("Unknown file %d\n", touch->fid);
                break;
        }
        FREE(touch);
    }
}

modem_ref_callbacks_t callbacks = {
    .read       = my_read,
    .write      = my_write,
    .read_fprop = my_read_fprop,
    .flush      = my_flush,
    .remove     = my_delete,
    .udata      = my_udata,
    .lqual      = my_lqual,
    .ldown      = my_ldown,
    .reset      = my_reset,
    .boot       = my_boot,
    .busy       = my_busy,
};

/*** Main function ------------------------------------------------------------- ***/
int main()
{
    // Start & initialize
#ifdef DEBUG_LED
    DBG_OPEN(DEBUG_LED);
#else
    DBG_OPEN(NC);
#endif
    PRINT("\n"
          "-----------------------------------------\n"
          "-------- Demo Receive file data ---------\n"
          "-----------------------------------------\n");
          
    FPRINT("(id:0x%08x)\r\n", osThreadGetId());
    
    modem_open(&callbacks);
    
    
    // Start file modified thread
    Thread th_file_modified(osPriorityNormal, 1024, NULL);
    osStatus status = th_file_modified.start(thread_file_modified);
    ASSERT(status == osOK, "Failed to start thread_file_modified (err: %d)\r\n", status);
    
    // Start reset display thread
    Thread th_reset_display(osPriorityLow, 1024, NULL);
    status = th_reset_display.start(thread_reset_display);
    ASSERT(status == osOK, "Failed to start thread_reset_display (err: %d)\r\n", status);

    // Initialize I2C and OLED Display
    i2c.frequency(400000);
    init_OLED();
    displayOn();
    reset_display();
    
    display_logo();

#ifdef DEBUG_LED
    DigitalOut my_led(DEBUG_LED);
#endif
    
    // Set main task to lowest priority
    osThreadSetPriority(osThreadGetId(), osPriorityLow);
    while(true)
    {
        ThisThread::sleep_for(500);
#ifdef DEBUG_LED
        my_led = !my_led;
#endif
    }
}