// @autor: jeremie@wizzilab.com
// @date: 2017-05-02

#include "DebouncedInterrupt.h"
#include "modem_d7a.h"
#include "modem_callbacks.h"

Semaphore button_user(0);
Queue<void, 8> g_udata;

typedef struct {
    Thread* thread;
    uint32_t count;
    d7a_addressee_t addressee;
} ping_t;

#define FID_PING_PONG           128
#define PING_DELAY              1000
#define PING_COUNTER_SIZE       sizeof(uint32_t)
#define MAX_PING_NB             MAX_USER_NB-2

// Special access classes for tests: no duty cycle limit, continuous scan
// { .bf.s = 12, .bf.m = 1 }; --> High Rate
// { .bf.s = 13, .bf.m = 1 }; --> Normal Rate
// { .bf.s = 14, .bf.m = 1 }; --> Slow Rate

// Use Gateway access profile and its 2 subprofiles at the same time
// Subprofile 0: 3 channels
// Subprofile 1: 4 channels
d7a_xcl_t ping_pong_xcl = { .bf.s = 2, .bf.m = 0x2 };

// This describe the interface used for communication
// Do not modify uncommented parameters
alp_d7a_itf_t my_itf = {
    .type                           = ALP_ITF_TYPE_D7A,
    .cfg.to                         = 0,
    .cfg.te                         = 0,
    .cfg.qos.bf.resp                = D7A_RESP_NO, // Communication protocol
    .cfg.qos.bf.retry               = ALP_RPOL_ONESHOT, // Retry policy 
    .cfg.addressee.ctrl.bf.nls      = D7A_NLS_AES_CCM_64, // Security level
    .cfg.addressee.xcl              = ping_pong_xcl, // Used Access Class
    // One of the followings:
    .cfg.addressee.ctrl.bf.idf      = D7A_ID_NOID, // in Broadcast (same as D7A_ID_NBID with .cfg.addressee.id[0] = D7A_CTF_VAL(2,2) (32))
 
    //.cfg.addressee.ctrl.bf.idf      = D7A_ID_UID, // in Unicast
    //.cfg.addressee.id               = { 0x00, 0x1B, 0xC5, 0x0C, 0x70, 0x00, 0x00, 0x00 }, // Destination UID

    //.cfg.addressee.ctrl.bf.idf      = D7A_ID_NBID, // in Broadcast
    //.cfg.addressee.id[0]            = D7A_CTF_ENCODE(4), // Estimation of expected responses (1 to 32)
};

// Interrupt Service Routine on button press.
void button_push_isr( void )
{
    button_user.release();
}

#ifdef DEBUG_LED
DigitalOut my_led(DEBUG_LED);
#endif


void my_udata(alp_payload_t* alp)
{   
    g_udata.put((void*)alp);
}

void udata_thread()
{
    alp_payload_t* alp;
    alp_payload_t* alp_op;
    alp_parsed_chunk_t r;
    u8* p;
    d7a_sp_res_t istat;
    osEvent evt;
            
    while (true)
    {
        evt = g_udata.get();
        alp = (evt.status == osEventMessage)? (alp_payload_t*)evt.value.p : NULL;

        //alp_payload_print(alp);
        
        alp_op = alp_payload_get(alp, ALP_OPCODE_RSP_F_DATA);
                    
        if (alp_op)
        {
            p = alp_op->d;
            
            alp_parse_chunk(&p, &r);
            
            if(FID_PING_PONG == r.meta.f_data.fid && 0 == r.meta.f_data.offset && PING_COUNTER_SIZE == r.meta.f_data.length)
            {
                uint32_t count;
    
                // Get data
                memcpy(&count, r.data, r.meta.f_data.length);
                
                PRINT("Got  PING %d", count);
                
                // Get metadata
                alp_op = alp_payload_get(alp, ALP_OPCODE_RSP_ISTATUS);
    
                if (alp_op)
                {
                    p = alp_op->d;
                
                    alp_parse_chunk(&p, &r);
                    memcpy(&istat, r.data, r.meta.itf.length);
                
                    PRINT_DATA(" from ", "%02X", istat.addressee.id, 8, "");
                    PRINT(" (SNR:%d dB RXLEV:%d dBm LB:%d dB)\n", istat.snr, -istat.rxlev, istat.lb);
                }
                else
                {
                    PRINT(" No istat!\n");
                    alp_payload_free(alp);
                    continue;
                }
                
    #ifdef DEBUG_LED
                my_led = 1;
    #endif
            
                ThisThread::sleep_for(PING_DELAY);
                
    #ifdef DEBUG_LED
                my_led = 0;
    #endif
                
                alp_d7a_itf_t resp_itf = {
                    .type                           = ALP_ITF_TYPE_D7A,
                    .cfg.to                         = D7A_CTF(0),
                    .cfg.te                         = D7A_CTF(0),
                    .cfg.qos.bf.resp                = D7A_RESP_NO,
                    .cfg.qos.bf.retry               = ALP_RPOL_ONESHOT,
                    .cfg.addressee                  = istat.addressee,
                };
                
                count++;
                
                // Send ping
                PRINT("Send PING %d", count);
                PRINT_DATA(" to   ", "%02X", resp_itf.cfg.addressee.id, 8, "\n");
                
                // Build payload
                alp_op = NULL;
                alp_op = alp_payload_rsp_f_data(alp_op, FID_PING_PONG, &count, 0, PING_COUNTER_SIZE);
                // Send
                modem_remote_raw_alp((void*)&resp_itf, alp_op, NULL);
            }
        }
        
        alp_payload_free(alp);
    }
}

void button_user_thread()
{
    alp_payload_t* alp;
    uint32_t ping = 0;
        
    FPRINT("(id:0x%08x)\r\n", osThreadGetId());
        
    while (true)
    {
        // Wait for button press
        PRINT("PRESS BUTTON TO INITIATE PING...\n");
        button_user.acquire();

        // Initiate ping
        PRINT("Initiate PING\n");
        
        // Build payload
        alp = NULL;
        alp = alp_payload_rsp_f_data(alp, FID_PING_PONG, &ping, 0, PING_COUNTER_SIZE);
        // Send
        modem_remote_raw_alp((void*)&my_itf, alp, NULL);
    }
}

modem_ref_callbacks_t callbacks = {
    .read       = my_read,
    .write      = my_write,
    .read_fprop = my_read_fprop,
    .flush      = my_flush,
    .remove     = my_delete,
    .udata      = my_udata,
    .reset      = my_reset,
    .boot       = my_boot,
};

/*** Main function ------------------------------------------------------------- ***/
int main()
{
    // Start & initialize
    #ifdef DEBUG_LED
    DBG_OPEN(DEBUG_LED);
#else
    DBG_OPEN(NC);
#endif
    PRINT("\n"
          "-----------------------------------------\n"
          "------------ Demo Ping Pong -------------\n"
          "-----------------------------------------\n");
          
    FPRINT("(id:0x%08x)\r\n", osThreadGetId());
    
    modem_open(&callbacks);
    
    // Put modem to listen to this access class
    modem_write_file(D7A_FID_DLL_CFG, &ping_pong_xcl, offsetof(d7a_dll_cfg_t, xcl), sizeof(d7a_xcl_t));
    
    PRINT("Start D7A Stack\n");
    modem_d7a_enable_itf();
    
    Thread udata_th(osPriorityNormal, 1024, NULL);
    osStatus status = udata_th.start(udata_thread);
    ASSERT(status == osOK, "Failed to start udata thread (err: %d)\r\n", status);
    
#ifdef DEBUG_BUTTON
    DebouncedInterrupt user_interrupt(DEBUG_BUTTON);
    user_interrupt.attach(button_push_isr, IRQ_FALL, 500, true);
    
    Thread but_th(osPriorityNormal, 1024, NULL);
    status = but_th.start(button_user_thread);
    ASSERT(status == osOK, "Failed to start but thread (err: %d)\r\n", status);
#endif
    
    // Set main task to lowest priority
    osThreadSetPriority(osThreadGetId(), osPriorityLow);
    while(true)
    {
        ThisThread::sleep_for(500);
    }
}