// @autor: jeremie@wizzilab.com
// @date: 2017-05-02

#include "DebouncedInterrupt.h"
#include "modem_d7a.h"
#include "modem_callbacks.h"

#if 1
    // Use D7 LoRa
    #define XCL_DL d7lora_dl_xcl
    #define XCL_UL d7lora_ul_xcl
#else
    // Use D7 FSK
    #define XCL_DL d7fsk_dl_xcl
    #define XCL_UL d7fsk_ul_xcl
#endif

#if 0
    // Disable Duty Cycle
    #define DUTY 255
#else
    // Use default Duty Cycle
    #define DUTY D7A_CTF_ENCODE(31)
#endif

Semaphore button_user(0);

d7a_xcl_t d7fsk_dl_xcl = { .bf = { .m = 0x1, .s = 0 } };
d7a_xcl_t d7fsk_ul_xcl = { .bf = { .m = 0x1, .s = 2 } };

d7a_xcl_t d7lora_dl_xcl = { .bf = { .m = 0x1, .s = 8 } };
d7a_xcl_t d7lora_ul_xcl = { .bf = { .m = 0x1, .s = 9 } };

alp_d7a_itf_t my_itf = {
    .type                           = ALP_ITF_TYPE_D7A,
    .cfg.to.byte                    = D7A_CTF_ENCODE(0),
    .cfg.te.byte                    = D7A_CTF_ENCODE(0),
    .cfg.qos.bf.resp                = D7A_RESP_ALL,
    .cfg.qos.bf.retry               = ALP_RPOL_ONESHOT,
    .cfg.addressee.ctrl.bf.nls      = D7A_NLS_AES_CCM_64,
    .cfg.addressee.ctrl.bf.idf      = D7A_ID_NBID,
    .cfg.addressee.xcl              = XCL_UL,
    .cfg.addressee.id[0]            = D7A_CTF_ENCODE(4),
};

// Interrupt Service Routine on button press.
void button_push_isr( void )
{
    button_user.release();
}

void button_user_thread()
{
    FPRINT("(id:0x%08x)\r\n", osThreadGetId());

    osEvent evt;
    fw_version_t fw_ver;
    uint8_t nb = 0;
    alp_payload_t* alp;
    alp_payload_t* alp_rsp;
    d7a_sp_res_t istat;
    int err;
        
    uint8_t scan_xcl[] = { XCL_DL.byte, XCL_UL.byte };
    
    modem_d7a_enable_itf();
    
    while (true)
    {
        // Wait for button press
        PRINT("Press button to scan...\r\n");
        button_user.acquire();
        
        for (uint8_t i = 0; i < sizeof(scan_xcl); i++)
        {
            nb = 0;
            my_itf.cfg.addressee.xcl.byte = scan_xcl[i];
            
            PRINT("Scanning XCL 0x%02X...\n", my_itf.cfg.addressee.xcl.byte);
            
            alp = NULL;
            alp = alp_payload_f_rd_data(alp, D7A_FID_FIRMWARE_VERSION, 12, sizeof(fw_version_t), false);
            
            err = modem_remote_raw_alp((void*)&my_itf, alp, &alp_rsp, (uint32_t)60000);

            if (err < ALP_ERR_NONE)
            {
                PRINT("Timeout.\n");
            }
            else
            {
                err = alp_payload_get_err(alp_rsp);
                PRINT("Err %d: ", err);
                modem_print_error(my_itf.type, err);
            }
            
            do {
                nb++;
                
                alp = alp_payload_extract(&alp_rsp, ALP_OPCODE_RSP_ISTATUS);

                if (alp)
                {
                    alp_parsed_chunk_t r;
                    u8* p = alp->d;
                
                    alp_parse_chunk(&p, &r);
                    memcpy(&istat, r.data, r.meta.itf.length);
                    
                    PRINT("%d: XCL:%02X ", nb, istat.addressee.xcl.byte);
                    PRINT_DATA("UID:", "%02X", istat.addressee.id, 8, " ");
                    PRINT("snr:%d rxlev:%d lb:%d ", istat.snr, istat.rxlev, istat.lb);
                    
                    alp_payload_free(alp);
                }
                else
                {
                    break;
                }
                
                alp = alp_payload_extract(&alp_rsp, ALP_OPCODE_RSP_F_DATA);
                
                if (alp)
                {
                    alp_parsed_chunk_t r;
                    u8* p = alp->d;
                    
                    alp_parse_chunk(&p, &r);
                    memcpy(&fw_ver, r.data, r.meta.f_data.length);
                    
                    PRINT("v%d.%d.%d\n", fw_ver.major, fw_ver.minor, fw_ver.patch);
                    
                    alp_payload_free(alp);
                }
                else
                {
                    break;
                }
                
                FLUSH();
            } while (1);
            
            alp_payload_free(alp_rsp);
        }
        
        PRINT("Done.\n");
    }
}

modem_ref_callbacks_t callbacks = {
    .read       = my_read,
    .write      = my_write,
    .read_fprop = my_read_fprop,
    .flush      = my_flush,
    .remove     = my_delete,
    .udata      = my_udata,
    .lqual      = my_lqual,
    .ldown      = my_ldown,
    .reset      = my_reset,
    .boot       = my_boot
};

/*** Main function ------------------------------------------------------------- ***/
int main()
{
    // Start & initialize
#ifdef DEBUG_LED
    DBG_OPEN(DEBUG_LED);
#else
    DBG_OPEN(NC);
#endif
    PRINT("\n"
          "-----------------------------------------\n"
          "------------- Demo D7A LoRa -------------\n"
          "-----------------------------------------\n");
          
    modem_open(&callbacks);
    
    // Put modem to listen to downlink access class
    modem_write_file(D7A_FID_DLL_CFG, &XCL_DL, offsetof(d7a_dll_cfg_t, xcl), sizeof(d7a_xcl_t));
    
    // Configure interface to use uplink access class
    modem_write_file(IFID_REPORT, &XCL_UL, offsetof(alp_d7a_itf_t, cfg.addressee.xcl), sizeof(d7a_xcl_t));

#if 0
    // Configure duty
    d7a_ctf_t duty = { .byte = DUTY };
    for (int i = 0; i < 8; i++)
    {
        modem_write_file(D7A_FID_ACCESS_PROFILE_0 + XCL_DL.bf.s, &duty, offsetof(d7a_access_profile_t, sb[0].duty) + (i * sizeof(d7a_subband_t)), sizeof(d7a_ctf_t));
    }
    
    for (int i = 0; i < 8; i++)
    {
        modem_write_file(D7A_FID_ACCESS_PROFILE_0 + XCL_UL.bf.s, &duty, offsetof(d7a_access_profile_t, sb[0].duty) + (i * sizeof(d7a_subband_t)), sizeof(d7a_ctf_t));
    }
#endif

#ifdef DEBUG_BUTTON
    DebouncedInterrupt user_interrupt(DEBUG_BUTTON);
    user_interrupt.attach(button_push_isr, IRQ_FALL, 500, true);
    
    Thread but_th(osPriorityNormal, 2048, NULL);
    osStatus status = but_th.start(button_user_thread);
    ASSERT(status == osOK, "Failed to start but thread (err: %d)\r\n", status);
#else
    #error You need a button to use this APP as is
#endif

#ifdef DEBUG_LED
    DigitalOut my_led(DEBUG_LED);
#endif
    
    // Set main task to lowest priority
    osThreadSetPriority(osThreadGetId(), osPriorityLow);
    while(true)
    {
        ThisThread::sleep_for(500);
#ifdef DEBUG_LED
        my_led = !my_led;
#endif
    }
}