// @autor: jeremie@wizzilab.com
// @date: 2017-12-14

#include "modem_d7a.h"
#include "modem_callbacks.h"
#include "files.h"
#include "crc.h"
#include "cup_app.h"

#define CHUNK_SIZE (128)

Queue<touch_t, 8> g_file_modified;

// CRC calculated on the incoming data stream
extern uint32_t stream_crc;
extern int32_t last_end;

modem_ref_callbacks_t callbacks = {
    .read       = my_read,
    .write      = my_write,
    .read_fprop = my_read_fprop,
    .flush      = my_flush,
    .remove     = my_delete,
    .udata      = my_udata,
    .lqual      = my_lqual,
    .ldown      = my_ldown,
    .reset      = my_reset,
    .boot       = my_boot,
    .busy       = my_busy
};

void thread_file_modified()
{
    touch_t* touch;
    osEvent evt;
    Timer tim;
    uint32_t total = 0;
    uint32_t next_chunk = 0;
    uint32_t chunks=0;
    
    PRINT("Ready\n");
    
    while (true)
    {
        evt = g_file_modified.get();
        touch = (evt.status == osEventMessage)? (touch_t*)evt.value.p : NULL;
        ASSERT(touch != NULL, "NULL touch pointer!\n");
        
        switch (touch->fid)
        {
            case FID_APP_CUP_CFG_BCAST:
            {
                cup_cfg_bcast_header_t cup_cfg;
                
                ram_fs_read(FID_APP_CUP_CFG_BCAST, (uint8_t*)&cup_cfg, 0, sizeof(cup_cfg_bcast_header_t));
                        
                if (CUP_CMD_UPGRADE_FILE_START == cup_cfg.cmd)
                {
                    if (cup_cfg_bcast_init())
                    {
                        PRINT("Enter upload mode. Watchdog %ds\n", cup_cfg.to);
                        
                        tim.stop();
                        tim.reset();
                        total       = 0;
                        next_chunk  = 0;
                        stream_crc  = 0;
                        chunks      = 0;
                        
                        // Write to the modem CUP config file (as root)
                        // to enter upload mode
                        cup_cfg_t cup_start = {
                            .cmd = CUP_CMD_UPGRADE_UPLOAD,
                            .arch_nb = cup_cfg.to,
                            };
                        modem_write_file(FID_CUP_CFG, (uint8_t*)&cup_start, 0, 4);
                    }
                    else
                    {
                        PRINT("Failed bcast init\n");
                    }
                }
                else
                {
                    PRINT("CUP command 0x%04X\n", cup_cfg.cmd);
                }
               
                break;
            }
            case FID_APP_CUP_CFG:
            {
                cup_cfg_t cup_cfg = { 0 };
                uint8_t* data = (uint8_t*)((uint8_t*)&cup_cfg + touch->offset);
                
                ram_fs_read(touch->fid, data, touch->offset, touch->length);
                
                if (CUP_CMD_UPGRADE_FILE_END == cup_cfg.cmd)
                {
                    PRINT("Done.\n");

                    // Actual check is done here since we could have received some of the missed data in a previous session.
                    
                    /******************************/
                    /* Compute CRC on saved data. */
                    /******************************/

                    // If you have the full binary data,
                    // You can use this function to compute CRC
                    //binary_crc = crc32((char*)binary_data, expected_size);
                    
                    // Else just do a loop and a fast CRC on data chunks like in my_write
                }
                else
                {
                    PRINT("CUP command 0x%04X\n", cup_cfg.cmd);
                }
                
                break;
            }
            case FID_APP_CUP_CODE:
            {
                cup_cfg_bcast_update(touch->offset, touch->length);

                // Chunk data should be saved in my_write callback
                
                // Notify modem code file to reset upload watchdog
                // else the device will exit upload mode after the watchdog timeout
                modem_notify_file(FID_CUP_CODE, 0, 1);
                
                break;
            }
            default:
                PRINT("TOUCH FID %d OFF %d LEN %d\n", touch->fid, touch->offset, touch->length);
                break;
        }
        
        FREE(touch);
    }
}

/*** Main function ------------------------------------------------------------- ***/
int main() {
    // Start & initialize
#ifdef DEBUG_LED
    DBG_OPEN(DEBUG_LED);
#else
    DBG_OPEN(NC);
#endif
    PRINT("\n"
          "-----------------------------------------\n"
          "---------------- Demo CUP ---------------\n"
          "-----------------------------------------\n");
              
    modem_open(&callbacks);
    
    PRINT("--------------- APP infos ---------------\r\n");
    PRINT(" - Manufacturer ID:  %08X\r\n", f_rev.manufacturer_id);
    PRINT(" - Device ID:        %08X\r\n", f_rev.device_id);
    PRINT(" - Hardware version: %08X\r\n", f_rev.hw_version);
    PRINT(" - Firmware version: v%d.%d.%d [%02X]\r\n", f_rev.fw_version.major, f_rev.fw_version.minor, f_rev.fw_version.patch, f_rev.fw_version.id);
    PRINT(" - CUP max size:     %d\r\n", f_rev.cup_max_size);
    PRINT("-----------------------------------------\r\n");
    
    PRINT("Register Files\n");
    ram_fs_new(FID_APP_CUP_CFG, (uint8_t*)&h_cup_cfg, (uint8_t*)&f_cup_cfg);
    ram_fs_new(FID_APP_CUP_CFG_BCAST, (uint8_t*)&h_cup_cfg_bcast, (uint8_t*)&f_cup_cfg_bcast);
    modem_declare_file(FID_APP_CUP_CFG, (alp_file_header_t*)&h_cup_cfg);
    modem_declare_file(FID_APP_CUP_CFG_BCAST, (alp_file_header_t*)&h_cup_cfg_bcast);
    
    // Declare the cup code file
    // It needs a special handling of its data in my_write callback since we want to keep them.
    // (can't be done with the current RAM file system)
    modem_declare_file(FID_APP_CUP_CODE, (alp_file_header_t*)&h_cup_code);
    
    PRINT("Enable D7A interface\n");
    modem_d7a_enable_itf();
    
    // Host revision file is in the modem. Update it.
    PRINT("Update host revision\n");
    modem_write_file(FID_HOST_REV, &f_rev, 0, sizeof(revision_t));
    
    // Retrieve modem revision
    PRINT("Send revision\n");
    revision_t rev;
    modem_read_file(FID_WM_REV, &rev, 0, sizeof(revision_t));
    
    // Send both to the server
    // Build payload
    alp_payload_t* alp = NULL;
    alp = alp_payload_rsp_f_data(alp, FID_WM_REV, &rev, 0, sizeof(revision_t));
    alp = alp_payload_rsp_f_data(alp, FID_HOST_REV, &f_rev, 0, sizeof(revision_t));
    
    // Send
    alp_itf_d7a_cfg_t report_itf;
    modem_read_file(IFID_REPORT, &report_itf, 0, sizeof(alp_itf_d7a_cfg_t));
    modem_remote_raw_alp((void*)&report_itf, alp, NULL, 10000);
        
    // Start file modified thread
    Thread th_file_modified(osPriorityNormal, 1024, NULL);
    osStatus status = th_file_modified.start(thread_file_modified);
    ASSERT(status == osOK, "Failed to start thread_file_modified (err: %d)\r\n", status);

#ifdef DEBUG_LED
    DigitalOut my_led(DEBUG_LED);
#endif
    
    // Set main task to lowest priority
    osThreadSetPriority(osThreadGetId(), osPriorityLow);
    while(true)
    {
        ThisThread::sleep_for(500);
#ifdef DEBUG_LED
        my_led = !my_led;
#endif
    }
}