// @autor: jeremie@wizzilab.com
// @date: 2017-05-02

#include "DebouncedInterrupt.h"
#include "modem_d7a.h"
#include "modem_callbacks.h"

static Semaphore button_user(0);

modem_ref_callbacks_t callbacks = {
    .read       = my_read,
    .write      = my_write,
    .read_fprop = my_read_fprop,
    .flush      = my_flush,
    .remove     = my_delete,
    .udata      = my_udata,
    .reset      = my_reset,
    .boot       = my_boot,
};

alp_d7a_itf_t my_itf = {
    .type                           = ALP_ITF_TYPE_D7A,
    .cfg.to                         = D7A_CTF(0),
    .cfg.te                         = D7A_CTF(0),
    .cfg.qos.bf.resp                = D7A_RESP_ALL,
    .cfg.qos.bf.retry               = ALP_RPOL_ONESHOT,
    .cfg.addressee.ctrl.bf.nls      = D7A_NLS_AES_CCM_64,
    .cfg.addressee.ctrl.bf.idf      = D7A_ID_NBID,
    .cfg.addressee.xcl.bf           = {.s = 2, .m = 0x1},// XXX D7A_XCL_GW,
    .cfg.addressee.id[0]            = D7A_CTF_ENCODE(8),
};

// Interrupt Service Routine on button press.
void button_push_isr( void )
{
    button_user.release();
}

void button_user_thread()
{
    FPRINT("(id:0x%08x)\r\n", osThreadGetId());

    osEvent evt;
    fw_version_t fw_ver;
    uint8_t nb = 0;
    alp_payload_t* alp;
    alp_payload_t* alp_rsp;
    d7a_sp_res_t istat;
    int err;
        
    uint8_t scan_xcl[] = { 0x01, 0x21 };
        
    modem_d7a_enable_itf();
    
    while (true)
    {
        // Wait for button press
        PRINT("Press button to scan...\r\n");
        button_user.acquire();
        
        for (uint8_t i = 0; i < sizeof(scan_xcl); i++)
        {
            nb = 0;
            my_itf.cfg.addressee.xcl.byte = scan_xcl[i];
            
            PRINT("Scanning XCL 0x%02X...\n", my_itf.cfg.addressee.xcl.byte);
            
            alp = NULL;
            alp = alp_payload_f_rd_data(alp, D7A_FID_FIRMWARE_VERSION, 12, sizeof(fw_version_t), false);
            
            err = modem_remote_raw_alp((void*)&my_itf, alp, &alp_rsp);

            if (err < ALP_ERR_NONE)
            {
                PRINT("Timeout.\n");
            }
            else
            {
                err = alp_payload_get_err(alp_rsp);
                modem_print_error(my_itf.type, err);
            }
            
            do {
                nb++;
                
                if (alp_payload_extract_data(&alp_rsp, ALP_OPCODE_RSP_ISTATUS, &istat))
                {
                    PRINT("%2d: XCL:%02X ", nb, istat.addressee.xcl.byte);
                    PRINT_DATA("UID:", "%02X", istat.addressee.id, 8, " ");
                    PRINT("snr:%2d rxlev:%2d lb:%3d ", istat.snr, istat.rxlev, istat.lb);
                }
                else
                {
                    break;
                }
                                
                if (alp_payload_extract_data(&alp_rsp, ALP_OPCODE_RSP_F_DATA, &fw_ver))
                {
                    PRINT("v%d.%d.%d\n", fw_ver.major, fw_ver.minor, fw_ver.patch);
                }
                else
                {
                    break;
                }
                
                FLUSH();
            } while (1);
            
            alp_payload_free(alp_rsp);
        }
        
        PRINT("Done.\n");
    }
}

/*** Main function ------------------------------------------------------------- ***/
int main()
{
    // Start & initialize
#ifdef DEBUG_LED
    DBG_OPEN(DEBUG_LED);
#else
    DBG_OPEN(NC);
#endif
    PRINT("\n"
          "-----------------------------------------\n"
          "--------- Demo ActiveRFIDReader ---------\n"
          "-----------------------------------------\n");
          
    FPRINT("(id:0x%08x)\r\n", osThreadGetId());
    
    modem_open(&callbacks);

#ifdef DEBUG_BUTTON
    DebouncedInterrupt user_interrupt(DEBUG_BUTTON);
    user_interrupt.attach(button_push_isr, IRQ_FALL, 500, true);
    
    Thread but_th(osPriorityNormal, 2048, NULL);
    osStatus status = but_th.start(button_user_thread);
    ASSERT(status == osOK, "Failed to start but thread (err: %d)\r\n", status);
#else
    #error You need a button to use this APP as is
#endif

#ifdef DEBUG_LED
    DigitalOut my_led(DEBUG_LED);
#endif
    
    // Set main task to lowest priority
    osThreadSetPriority(osThreadGetId(), osPriorityLow);
    while(true)
    {
        ThisThread::sleep_for(500);
#ifdef DEBUG_LED
        my_led = !my_led;
#endif
    }
}