#include "mbed.h"
#include "WizziDebug.h"

#define LocalCircularBuffer CircularBuffer<T,BufferSize,CounterType>

template<typename T, uint32_t BufferSize, typename CounterType = uint32_t>
class CBuffer : private LocalCircularBuffer
{
public:
    
    CBuffer() : _buffered_data_size(0) {}
    
    ~CBuffer() {}

    CounterType get_size(void)
    {
        return BufferSize;
    }
    
    CounterType available_data(void)
    {
        return _buffered_data_size;
    }
    
    CounterType remaining_space(void)
    {
        return BufferSize - _buffered_data_size;
    }
    
    bool full(void)
    {
        return this->LocalCircularBuffer::full();
    }
    
    bool empty(void)
    {
        return this->LocalCircularBuffer::empty();
    }
    
    void reset(void)
    {
        _buffered_data_size = 0;
        this->LocalCircularBuffer::reset();
    }
    
    T pop(void)
    {
        T data;
        
        ASSERT(this->LocalCircularBuffer::pop(data), "CBuffer: No data in buffer to pop\r\n");
        _buffered_data_size--;
        
        return data;
    }
    
    void push(T data)
    {
        ASSERT(!this->LocalCircularBuffer::full(), "CBuffer: Buffer is full, can't push\r\n");
        
        this->LocalCircularBuffer::push(data);
        _buffered_data_size++;
    }
    
    void get(T* buf, CounterType size)
    {
        ASSERT(size, "CBuffer: Can't copy length 0\r\n");
        ASSERT(size <= _buffered_data_size, "CBuffer: Size too long (%d) for available data (%d)\r\n", size, _buffered_data_size);
        
        CounterType len = 0;
        
        while (len < size)
        {
            if (buf != NULL)
            {
                this->LocalCircularBuffer::pop(buf[len]);
            }
            else
            {
                // Pop data in dummy buffer
                T dummy;
                this->LocalCircularBuffer::pop(dummy);
            }
            len++;
        }
    
        _buffered_data_size -= size;
    }
    
    void add(T* buf, CounterType size)
    {
        ASSERT(size, "CBuffer: Can't add 0 bytes\r\n");
        ASSERT(size <= this->remaining_space(), "CBuffer: Size too long (%d) for available space (%d)\r\n", size, BufferSize - _buffered_data_size);
        ASSERT(buf != NULL, "CBuffer: Can't copy from NULL buffer\r\n");
        
        CounterType len = 0;
        
        while (len < size)
        {
            this->LocalCircularBuffer::push(buf[len++]);
        }
        
        _buffered_data_size += size;
    }
    
private:
    volatile CounterType _buffered_data_size;
};