#define I2C_SDA p3
#define I2C_SCL p4

#include "mbed.h"
#include "BLE.h"
#include "battery.h"
#include "WeatherService.h"
#include "BatteryService.h"
#include "BMP180.h"
#include "SHT21_ncleee.h"


float temperature1;
float temperature2;
float pressure;
float humidity;

BLE ble;

DigitalOut okLED(LED1);
DigitalOut errLED(LED2);
DigitalOut instrumentsPower(p30);

BMP180 bmp180;
SHT21 sht21(&i2c);


const static char DEVICE_NAME[] = "Weather Station";

static const uint16_t serviceList[] = {
    GattService::UUID_ENVIRONMENTAL_SERVICE, 
    GattService::UUID_BATTERY_SERVICE
};

static volatile bool triggerSensorPolling = false;

void disconnectionCallback(const Gap::DisconnectionCallbackParams_t *params)
{
    /* Restart Advertising on disconnection*/
    ble.gap().startAdvertising();
}

void blink(void)
{
//    statusLED = !statusLED;
    triggerSensorPolling = true;
}

void updateFromBMP180() {
    uint8_t c = bmp180.readByte(BMP180_ADDRESS, BMP180_WHO_AM_I);   
    
    printf("BMP-180 is 0x%x\n\r", c);
    printf("BMP-180 should be 0x55\n");
    
    if(c == 0x55) {
        printf("BMP-180 online\n");
       
        printf("Calibrating BMP-180...");
        bmp180.BMP180Calibration();
        printf("done\n");
    }
    else 
    {
        printf("BMP-180 unreachable\n");
        return;
    }    
    
    temperature1 = (float)bmp180.BMP180GetTemperature()/10.0f;
    pressure = (float)bmp180.BMP180GetPressure();
}

void updateFromSHT21() {
    temperature2 = sht21.readTemp();
    humidity = sht21.readHumidity();
}

int main(void)
{
    printf("Start\n");
    
    okLED = 1;
    errLED = 1;

    Ticker ticker;
    ticker.attach(blink, 5);

    ble.init();
    ble.gap().onDisconnection(disconnectionCallback);

    /* Setup weather service. */
    WeatherService weatherService(ble);
    BatteryService batteryService(ble, 0);
    Battery battery(BATTERY_PIN);

    /* setup advertising */
    ble.gap().accumulateAdvertisingPayload(GapAdvertisingData::BREDR_NOT_SUPPORTED | GapAdvertisingData::LE_GENERAL_DISCOVERABLE);
    ble.gap().accumulateAdvertisingPayload(GapAdvertisingData::COMPLETE_LIST_16BIT_SERVICE_IDS, (uint8_t *)serviceList, sizeof(serviceList));
    ble.gap().accumulateAdvertisingPayload(GapAdvertisingData::UNKNOWN);
    ble.gap().accumulateAdvertisingPayload(GapAdvertisingData::COMPLETE_LOCAL_NAME, (uint8_t *)DEVICE_NAME, sizeof(DEVICE_NAME));
    ble.gap().setAdvertisingType(GapAdvertisingParams::ADV_CONNECTABLE_UNDIRECTED);
    ble.gap().setAdvertisingInterval(1000); /* 1000ms */
    ble.gap().startAdvertising();

    while (true) {
        if (triggerSensorPolling && ble.getGapState().connected) {
            okLED = 1;
            triggerSensorPolling = false;

            instrumentsPower = 1;
            wait(1);
            updateFromBMP180();
            updateFromSHT21();
            instrumentsPower = 0;
            
            float temperature = (temperature1 + temperature2) / 2;
            
            printf("Temp1: %.1f ºC \n", temperature1);
            printf("Temp2: %.1f ºC \n", temperature2);
            printf("Temp Avg.: %.1f ºC \n", temperature);
            printf("Pressure: %.3f Pa \n", pressure);
            printf("Humidity: %.1f%% \n", humidity);
            printf("---\n");

            
            weatherService.updateTemperature(temperature);
            weatherService.updatePressure(pressure);
            weatherService.updateHumidity(humidity);
                        
            
            batteryService.updateBatteryLevel(battery.read());
        } else {
            ble.waitForEvent();
        }
    }
}
