#include "salinity.h"
#include "general_control.h"
#include "TextLCD.h"
#include "motors.h"

AnalogIn sal(p20);  
extern Serial pc;
extern float cbV; extern int bC; extern int pw_csv; extern int sw_csv;
extern int pwTC; extern int pwTV; extern int swTC; extern int swTV;
extern bool rfl_flag, sal_suspension;
extern TextLCD lcd;


//Salinity calculation
float calc_sal(float voltage) {
    return 15.1747217178 * voltage - 2.89491343498;
}

float calc_sal2(float voltage) {
    return 16.3 * voltage;
}

float calc_finalsal(float s1, float s2) {
    return 14.259 + 6.795*s1 -5.148*s2 -1.3;
}

float calc_salinity(){
    float s1, s2, sfinal, sensor_volt = 1.6 * 3.3 * sal.read();;
    s1=calc_sal(sensor_volt);
    s2=calc_sal2(sensor_volt);
    sfinal=calc_finalsal(s1,s2);
    return sfinal;
}

void print_water_limits() {
    pc.printf("MAX volume of water that can be added: %.3f ml\n", bC-cbV);
    pc.printf("MAX volume of water that can be extracted: %.3f ml\n", cbV);
}

void syringe_capacity_control(char motor, int target) {
    if (motor=='a' && pw_csv<target) {                                          //Pure water syringe does not currently have target ml of water 
        while (!rfl_flag) {wait(0.1);}                                                    //Refill needed - Wait untill rfl_flag is raised
        rfl_flag = false;
        move_motors(motor, "left", SDC - pw_csv);                               //Refill the syringe with 25ml with water
    }
    if (motor=='b' && sw_csv<target) {                                          //Pure water syringe does not currently have target ml of water 
        while (!rfl_flag) {wait(0.1);}                                                    //Refill needed - Wait untill rfl_flag is raised
        rfl_flag = false;
        move_motors(motor, "left", SDC - sw_csv);                               //Refill the syringe with water
    }
}

void valve_warning_messages(int vol, bool pure, int pos, bool refill) {
    if(refill)  pc.printf("Refill of the %s water syringe is needed\n", pure ? "pure":"salted");
    else pc.printf("Going to add %d ml of %s water to the beaker\n", SDC, pure ? "pure":"salted");
    pc.printf("Make sure the %s water valve is at designated position %d!!\n", pure ? "pure":"salted", pos);    
    pc.printf("Operation has been suspended. Press switch 3 to resume\n");
}

void add_water(int vol, bool pure) {
    int i;
    char motor = pure ? 'a':'b';                                                //select the proper motor
    syringe_capacity_control(motor, SDC);                                       //make sure that the syringe has enough water 
    if (vol>SDC) {                                                              //if more than SDC ml are needed, refill will be required
        for (i=0; i<vol/SDC; i++) {                                             //vol div SDC refills needed
            valve_warning_messages(SDC, pure, 1, false);                        //print valve warning messages
            pc.printf("rfl_flag in add_water = %s\n", rfl_flag ? "true":"false");
            while (!rfl_flag) {wait(0.1);}                                      //Wait until switch 3 has been pressed         
            pc.printf("Operation has been resumed\n");   
            rfl_flag = false;  
            move_motors(motor, "right", SDC);                                   //add SDC ml of water to the beaker
            cbV += SDC;                                                         //Update current beaker volume
            pc.printf("New beaker volume: %.3f\n",cbV);  
            print_water_limits();                                                //Print the new water limits
            valve_warning_messages(SDC, pure, 1, true);                         //print valve warning messages
            while (!rfl_flag) {wait(0.1);}                                      //Wait untill switch 3 has been pressed
            pc.printf("Operation has been resumed\n");
            rfl_flag = false;                                                   //Reset flag
            move_motors(motor, "left", SDC);                                    //Refill the syringe with 25ml with water
            capacity_control();                                                 //Capacity control                           
        }
        move_motors(motor, "right", vol % SDC);                                 //Add the remaining vol modulo SDC ml with water - no syringe capacity control needed because already full
        cbV += vol % SDC;                                                       //Update current beaker volume
        print_water_limits();                                                         //Print the water limits
        capacity_control();                                                     //Capacity control
    }
    else {
        move_motors(motor, "right", vol);                                       //Add the needed water (no refill needed)
        cbV += vol;                                                             //Update current beaker volume
        print_water_limits();                                                         //Print the water limits
        capacity_control();                                                     //Capacity control
    }
}

void print_salinity(float sal) {
    pc.printf("Salinity = %.3f\n", sal);
    lcd.cls();
    lcd.printf("Sal: %.3f\n", sal);
}

void print_system_status() {
    pc.printf("bC = %d\n", bC);
    pc.printf("cbV = %.3f\n", cbV);                                                   
    pc.printf("pwTC = %d\n", pwTC);
    pc.printf("pwTV = %d\n", pwTV);
    pc.printf("swTC = %d\n", swTC);
    pc.printf("swTV = %d\n", swTV);
}

bool sal_sensor_out_of_water_check(float sl) {
    if(sl<SSOOW) {
        pc.printf("Salinity extremely low. Highly likely that the sensor is out of the water\n");
        pc.printf("Please check the salinity sensor position\n");
        sal_suspension = true;                                                  //Suspend heater
        pc.printf("For safety reasons, system operation has been suspended. If everything is OK, resume operation by pressing switch 4\n");
        while(sal_suspension) {wait(0.1);}                                               //Wait until switch 4 is pressed
        return false;
    }
    return true;
}

bool sal_connection_check(float sl) {
    if(SCLLL<sl && sl<SCLUL) {
        pc.printf("Salinity really low! Highly likely that the salinity sensor connection is lost\n");
        pc.printf("Please check the salinity sensor connection\n");
        sal_suspension = true;                                                  //Suspend heater
        pc.printf("For safety reasons, system operation has been suspended. If everything is OK, resume operation by pressing switch 4\n");
        while(sal_suspension) {wait(0.1);}                                               //Wait until switch 4 is pressed
        return false;
    }
    return true;
}

/*bool sal_malfunction_check(float sl) {
  ////
  return true;
}*/

bool salinity_control() {
    bool sal_state = true;
    float pwv = 0, swv = 0, salt;                                               //pwv=Pure Water Volume (ml), swv = Salted Water Volume (ml), salt (gr)
    float cbs = calc_salinity();                                                //Current salinity
    print_salinity(cbs);
    sal_state = sal_connection_check(cbs);                                      //connection check
    //sal_state = sal_malfunction_check(cbs);                                   //Malfunction check
    sal_state = sal_sensor_out_of_water_check(cbs);                             //Sensor out of water check
    cbs = calc_salinity();                                                      //Everything ok, take the value of salinity now
    
    if (cbs>=SSUL) {                                                            //pw case
        sal_state = false;
        pwv = cbs*cbV/DS - cbV;                                                 //how many ml?
        pc.printf("%.3f ml of pure water need to be added to the beaker\n", pwv);
        pc.printf("rfl_flag right before adding water = %s\n", rfl_flag ? "true":"false");
        add_water(pwv,true);                                                    //add pure water
    }
    else if (cbs<=SSLL) {                                                       //sw case
        sal_state = false;
        salt = cbV*cbs/1000;        
        swv = cbV-50*salt/4.8;                                                  ///how many ml? (Applicable ONLY for DS=20)
        pc.printf("%.3f ml of salted water need to be added to the beaker\n", swv);
        add_water(swv,false);                                                   //add salted water
    }
    pc.printf("\n");
    return sal_state;
}