#include "temp.h"
#include "TextLCD.h"
#include <cmath>

AnalogIn temp(p19);
DigitalOut htr(p27);
extern Serial pc;
extern TextLCD lcd;
extern bool htr_suspension, rfl_flag;
extern float air_temp;

float calc_temp() {
    float rt, t, voltage = 3.3 * temp.read();;
    rt = R1*voltage/(3.3-voltage);
    t = 1 / (K0 + K1 * log(rt) + K2 * pow(log(rt),3.0f));
    return t - 273.15;
}

float take_air_temp() {
    pc.printf("Going to take air temperature. Please make sure that the sensor is in the air and connected to the controller\n");
    pc.printf("If you are ready, please press switch 3\n");
    //pc.printf("rfl_flag = %s\n", rfl_flag ? "true":"false");
    while (!rfl_flag) {wait(0.1);}        //Wait until switch 4 has been pressed
    rfl_flag = false;
    return calc_temp();
}

void print_temp(float temp) {
    pc.printf("Temperature = %.3f\n", temp);
    lcd.cls();
    lcd.printf("Temp: %.3f\n", temp);
}

bool sensor_out_of_water_check(float cT) {
    if (air_temp-ATB<cT && cT<air_temp+ATB) {                                   //sensor out of water
        pc.printf("Temperature similar to air temperature. Highly likely that the sensor is out of the water\n");
        pc.printf("Please check the temperature sensor position\n");
        htr_suspension = true;                                                  //Suspend heater
        pc.printf("For safety reasons, heater operation has been suspended. If everything is OK, resume operation by pressing switch 4\n");
        return false;
    }
    return true;
}

bool connection_check(float cT) {
    if (cT<LC) {                                                                //Connection lost
        pc.printf("Temperature extremely freezingly low! Highly likely that the temp sensor connection is lost\n");
        pc.printf("Please check the temperature sensor connection\n");
        htr_suspension = true;                                                  //Suspend heater
        pc.printf("For safety reasons, heater operation has been suspended. If everything is OK, resume operation by pressing switch 4\n");
        return false;
    }
    return true;
}

bool malfunction_check(float cT) {
    if (cT<TMLL || cT>TMUL) {                                                   //Bizzare temps
        pc.printf("Temperature sensor is giving bizzare temps\n");
        pc.printf("Temperature sensor malfunction is higly likely! Please check the sensor\n");
        htr_suspension = true;                                                  //Suspend heater
        pc.printf("For safety reasons, heater operation has been suspended. If everything is OK, resume operation by pressing switch 4\n");
        return false;
    }
    return true;
}

void heater_control(float cT) {
    if (cT<=TSLL) {
        if(!htr_suspension) {
            htr=1;  pc.printf("Heater activated\n");
            pc.printf("Expecting temperature to rise\n");
        }
        else { pc.printf("Heater operation is suspended! If you want to resume operation press switch 4\n");}
    }
    else if(cT>=TSUL) {
        htr=0;
        pc.printf("Heater deactivated\n");
        pc.printf("Expecting temperature to fall\n");
    }
}

bool temp_control(){
    bool temp_state = true;                                                     //Assume normal temperature
    float cT = calc_temp();                                                     //Calculate temperature
    print_temp(cT);                                                             //Print temp to lcd and pc
    temp_state = connection_check(cT);                                          //connection check
    temp_state = malfunction_check(cT);                                         //Malfunction check
    temp_state = sensor_out_of_water_check(cT);                                 //Sensor out of water check
    heater_control(cT);                                                         //Heater control
    wait(3);                                                                    //Introduce a 3" delay in order for the changes to have an effect
    cT = calc_temp();                                                           //Calculate temperature again    
    if (cT<TALL) {
        temp_state = false;
        pc.printf("Temperature really low! Heater must be activated!\n");
        if(htr==1) pc.printf("Heater already activated so beaker is exposed to freezing conditions");
    }
    if (cT>TAUL) {
        temp_state = false;
        pc.printf("Temperature really high!\n");
        if(htr==0) pc.printf("Heater already deactivated so beaker is exposed to an external heating source or heater is powered directly from the mains!\nPlease check heater connection!");
    }
    pc.printf("\n");
    return temp_state;
}