/* WizFi250 implementation of NetworkInterfaceAPI
 * Copyright (c) 2015 ARM Limited
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
 

#include "WizFi250Interface.h"

// Various timeouts for different WizFi250 operations
#define WizFi250_CONNECT_TIMEOUT 15000
#define WizFi250_SEND_TIMEOUT    500
#define WizFi250_RECV_TIMEOUT    0
#define WizFi250_MISC_TIMEOUT    500

#define WizFi250_DELAY_MS       300


// WizFi250Interface implementation
WizFi250Interface::WizFi250Interface(PinName tx, PinName rx, PinName cts, PinName rts, PinName reset, PinName alarm,  int baud)
    : _wizfi250(tx, rx, cts, rts, reset, alarm, baud)
{
    memset(_ids, 0, sizeof(_ids));
}

int WizFi250Interface::connect(
    const char *ssid,
    const char *pass,
    nsapi_security_t security)
{
    if (!_wizfi250.startup()) 
    {
        return NSAPI_ERROR_DEVICE_ERROR;
    }

    _wizfi250.setSsid(ssid);
    _wizfi250.setSec(WizFi250::SEC_AUTO, pass);
    _wizfi250.setAddress("");
    
    if( _wizfi250.join(WizFi250::WM_STATION) == -1)
    {
        return NSAPI_ERROR_NO_CONNECTION;
    }

    return 0;
}

int WizFi250Interface::disconnect()
{
    if ( _wizfi250.cmdWLEAVE() == -1 )  return NSAPI_ERROR_DEVICE_ERROR;

    return 0;
}

const char *WizFi250Interface::get_ip_address()
{
    return _wizfi250.getIPAddress();
}

const char *WizFi250Interface::get_mac_address()
{
    return _wizfi250.getMACAddress();
}

struct wizfi250_socket {
    int id;
    nsapi_protocol_t proto;
    bool connected;
};

int WizFi250Interface::socket_open(void **handle, nsapi_protocol_t proto)
{
    // Look for an unused socket

    /*
    int id = -1;
 
    for (int i = 0; i < WIZFI250_SOCKET_COUNT; i++) {
        if (_ids[i] == false) {
            id = i;
            _ids[i] = true;
            break;
        }
    }
 
    if (id == -1) {
        return NSAPI_ERROR_NO_SOCKET;
    }
    */
    
    struct wizfi250_socket *socket = new struct wizfi250_socket;
    if (!socket) {
        return NSAPI_ERROR_NO_SOCKET;
    }
    
    socket->id = -1;
    socket->proto = proto;
    socket->connected = false;
    *handle = socket;
    return 0;
}

int WizFi250Interface::socket_close(void *handle)
{
    struct wizfi250_socket *socket = (struct wizfi250_socket *)handle;
    int err = 0;
 
    if(socket->id == -1){
        err = NSAPI_ERROR_NO_SOCKET;
    }
    else if (_wizfi250.close(socket->id) == -1) {
        err = NSAPI_ERROR_DEVICE_ERROR;
    }

    _ids[socket->id] = false;
    wait_ms(WizFi250_DELAY_MS);
    delete socket;
    return err;
}

int WizFi250Interface::socket_bind(void *handle, const SocketAddress &address)
{
    return NSAPI_ERROR_UNSUPPORTED;
}

int WizFi250Interface::socket_listen(void *handle, int backlog)
{
    return NSAPI_ERROR_UNSUPPORTED;
}

int WizFi250Interface::socket_connect(void *handle, const SocketAddress &addr)
{
    int cid=-1;
    struct wizfi250_socket *socket = (struct wizfi250_socket *)handle;

    WizFi250::Protocol proto = (socket->proto == NSAPI_UDP) ? WizFi250::PROTO_UDP : WizFi250::PROTO_TCP;
    if((cid = _wizfi250.open(proto, addr.get_ip_address(), addr.get_port())) == -1 )
    {
        return NSAPI_ERROR_DEVICE_ERROR;
    }

    if(cid >= WIZFI250_SOCKET_COUNT)
    {
        return NSAPI_ERROR_NO_SOCKET;
    }
    
    _ids[cid] = true;
    socket->id = cid;
    socket->connected = true;
    wait_ms(WizFi250_DELAY_MS);
    return 0;
}
    
int WizFi250Interface::socket_accept(void **handle, void *server)
{
    return NSAPI_ERROR_UNSUPPORTED;
}

int WizFi250Interface::socket_send(void *handle, const void *data, unsigned size)
{
    struct wizfi250_socket *socket = (struct wizfi250_socket *)handle;
 
    if ( _wizfi250.send(socket->id, (const char*)data, size) == -1 ) {
        return NSAPI_ERROR_DEVICE_ERROR;
    }
 
    return size;
}

int WizFi250Interface::socket_recv(void *handle, void *data, unsigned size)
{
    struct wizfi250_socket *socket = (struct wizfi250_socket *)handle;
 
    int32_t recv = _wizfi250.recv(socket->id, (char*)data, size);
    if (recv < 0) {
        return NSAPI_ERROR_WOULD_BLOCK;
    }
 
    return recv;
}

int WizFi250Interface::socket_sendto(void *handle, const SocketAddress &addr, const void *data, unsigned size)
{
    struct wizfi250_socket *socket = (struct wizfi250_socket *)handle;
    if (!socket->connected) {
        int err = socket_connect(socket, addr);
        if (err < 0) {
            return err;
        }
    }
    
    return socket_send(socket, data, size);
}

int WizFi250Interface::socket_recvfrom(void *handle, SocketAddress *addr, void *data, unsigned size)
{
    struct wizfi250_socket *socket = (struct wizfi250_socket *)handle;    
    return socket_recv(socket, data, size);
}

void WizFi250Interface::socket_attach(void *handle, void (*callback)(void *), void *data)
{
}

