#include "mbed.h"
#include "NCP5623BMUTBG.h"
#include "CAM_M8.h"
#include "SDFileSystem.h"
#include "Adafruit_ADS1015.h"
#include "BME280.h"
#include "BMP280.h"

/////////////////////////////////////////////
//Define core buses and pin states.
/////////////////////////////////////////////
I2C                 i2c(PB_9, PB_8);//(D14, D15); SDA,SCL
Serial              pc(USBTX, USBRX);
DigitalOut          pumps(PA_9, 0);//(D8, 0);
DigitalOut          pbKill(PC_12, 1); // Digital input pin that conncect to the LTC2950 battery charger used to shutdown the UPAS 
DigitalIn           nINT(PA_15); //Connected but currently unused is a digital ouput pin from LTC2950 battery charger. http://cds.linear.com/docs/en/datasheet/295012fd.pdf
NCP5623BMUTBG       RGB_LED(PB_9, PB_8);    //(D14, D15);

/////////////////////////////////////////////
//RN4677 BT/BLE Module
/////////////////////////////////////////////
Serial              ble(PB_10, PB_11);
DigitalOut          bleRTS(PB_14, 0);
DigitalOut          bleCTS(PB_13, 0);
DigitalOut          BT_IRST(PC_8, 0);
DigitalOut          BT_SW(PA_12, 0);

/////////////////////////////////////////////
//Analog to Digital Converter
/////////////////////////////////////////////
DigitalIn           ADS_ALRT(PA_10); //Connected but currently unused. (ADS1115) http://www.ti.com/lit/ds/symlink/ads1115.pdf
Adafruit_ADS1115    ads_sun(&i2c, ADS1015_ADDRESS_VDD); //Adress pin connected to 3.3V

/////////////////////////////////////////////
//Battery, Charger, & Supply Monitoring
/////////////////////////////////////////////
DigitalIn           LTCALT(PB_2); //High for normal operation. Low when a threshold is exceeded for Voltage, gas gauge, or temp
DigitalIn           bcs1(PC_9); //Batt charging if High. (BQ24100)[U23]
DigitalIn           bcs2(PA_8); //Charge complete if High. (BQ24100)[U23]  
DigitalIn           bc_npg(PB_1); //Power to the charge controller. (BQ24100)[U23] 
DigitalIn           SW3flt(PC_4); //When EN = 0 pin is HIGH, When EN = 1, LOW can be current limit, thermal limit, or UVLO.

/////////////////////////////////////////////
//Sensirion SDP3X(s)
/////////////////////////////////////////////
DigitalIn           SDP3Xflt(PC_0);
DigitalIn           SDP3XAltflt(PB_7);

/////////////////////////////////////////////
//Accelerometer and Magnometer
/////////////////////////////////////////////
DigitalOut          iNemoEnable(PA_1, 0);
DigitalIn           iNemoReady(PB_0);
DigitalIn           iNemoInt1(PC_5);
DigitalIn           iNemoInt2(PC_6);
DigitalIn           iNemoInt3(PC_7);

/////////////////////////////////////////////
//Temperature and Pressure
/////////////////////////////////////////////
BMP280              bmp(PB_9, PB_8, 0xEE);
BME280              bme(PB_9, PB_8, 0xEC);  //(D14, D15);

/////////////////////////////////////////////
//GPS
/////////////////////////////////////////////
DigitalOut          gpsEN(PB_15, 0);
CAM_M8              gps(PB_9, PB_8,(0x84));   
uint8_t gpsBTState = 1;
uint8_t meState = 0;
DigitalIn           gpsENFault(PB_12); //When EN = 0 pin is HIGH, When EN = 1, LOW can be current limit, thermal limit, or UVLO.

/////////////////////////////////////////////
//SD Card
/////////////////////////////////////////////
DigitalIn           sdCD(PA_11);
DigitalOut          sdClk(PB_3,0 );
DigitalIn           sdMISO(PB_4);
DigitalOut          sdMOSI(PB_5, 0);
DigitalOut          sdCS(PB_6, 1);

DigitalIn           pbIso(PA_0);
DigitalOut          hFault1(PA_7, 0);
DigitalOut          hFault2(PA_6, 0);
DigitalOut          hFault3(PA_5, 0);
DigitalOut          hFault4(PA_4, 0);

SDFileSystem        sd(PB_5, PB_4, PB_3, PB_6, "sd");//(D4, D5, D3, D10, "sd"); // (MOSI, MISO, SCK, SEL)
char VoltFile[] = "/sd/ROAD_00.txt";

/////////////////////////////////////////////
//Callbacks
/////////////////////////////////////////////
Ticker          loggVolts;  //This is the AOD logging callback object

volatile bool g_loggVolts_flag = false;

static void loggVolts_ISR( void ) {
    g_loggVolts_flag = true;
}

void logVolts(void);

/////////////////////////////////////////////
//Globals
/////////////////////////////////////////////
double  gpslatitude = 0.0;
double  gpslongitude = 0.0;
float   gpsaltitude = 0.0;
uint8_t gpsquality = 0;
uint8_t gpssatellites = 0;
long    gpsTime;
long    gpsDate;
int     gps_year;
int     gps_month;
int     gps_day;
int     gps_hour;
int     gps_minute;
double  gps_second;
double  v870;     //Voltage read from 870nm photodiode
double  v_raw870; //Raw analog output from 870nm photodiode
double  v680;     //Voltage read from 680nm photodiode
double  v_raw680; //Raw analog output from 680nm photodiode
double  v520;     //Voltage read from 520nm photodiode
double  v_raw520; //Raw analog output from 520nm photodiode
double  v440;     //Voltage read from 440nm photodiode
double  v_raw440; //Raw analog output from 440nm photodiode
float   bme_press;
float   bme_temp;
float   bmp_press;
//////////////////////////////////////////////////////////////
//Main Function
//////////////////////////////////////////////////////////////
int main()
{
    pc.baud(115200);
    loggVolts.attach(&loggVolts_ISR, 1.0);
    RGB_LED.set_led(1, 1, 1);
    gpsEN = 1;
    
    FILE *fp = fopen(VoltFile, "r");//try reading the log file
    
    for(int i = 0; i < 100; i++)
    {
        VoltFile[9] = i/10 + '0';
        VoltFile[10] = i%10 + '0';
        fp = fopen(VoltFile, "r");//try reading the log file
        if(fp == NULL)
        {
            fp = fopen(VoltFile, "w"); // fp is local variable
            fclose(fp);
            fp = fopen(VoltFile, "a");
            fprintf(fp, "GPS Date,GPS Time,Latitude,Longitude,Altitude,Temperature (C),Pressure (mbar),v440,v520,v680,v870\r\n");
            fclose(fp);
            free(fp);
            break;
        }
    }
    
    while(1){
        
        if(g_loggVolts_flag){
            g_loggVolts_flag = false;
            logVolts();
        }
        
    } //end while
} //end main

void logVolts(void)
{
    RGB_LED.set_led(0, 1, 1);
    //Get needed location data
    gps.read_gps();
    gpsquality =  gps.quality;
    if(gpsquality!=0)
    {
        gpslatitude = gps.lat;
        gpslongitude = gps.lon;
        gpsaltitude = gps.altitude;
    }
    gpsTime = (long)gps.utc;
    gpsDate = (long)gps.date;
                        
    gps_minute = (int)(floor((float)(gpsTime - (uint8_t)(floor((float)gpsTime/10000))*10000)/100));    // 0-59
    gps_second = (double)(floor(((float)gpsTime - 100*(floor((float)gpsTime/100)))));//0; //(uint8_t)(floor(((float)gpsTime - 100*(floor((float)gpsTime/100)))));    // 0-59
    gps_hour =  (int)(floor((float)gpsTime/10000));  // 0-23
    gps_day = (int)(floor((float)gpsDate/10000));   // 1-31
    gps_month = (uint8_t)(floor((float)(gpsDate - gps_day*10000)/100));     // 0-11
    gps_year = 1900 + (uint8_t)(100+floor(((float)gpsDate - 100*(floor((float)gpsDate/100))))); // year since 1900 (116 = 2016)//100+16
    
    RGB_LED.set_led(0, 1, 0);
    
    //Get temperature and pressure from the BME
    bme_temp = bme.getTemperature();
    bme_press = bme.getPressure(bme_temp);
    bmp_press = bmp.getPressure(bme_temp);
    
    // Take the average of 10 440 voltage measurements
    double v_440_temp;
    double v_440_sum = 0;
    for(int i = 0; i < 10; i++)
    {
        v_440_temp = (double)ads_sun.readADC_SingleEnded(A0_GAIN_TWO); //Channel A0 | 2x gain | +/-2.048V | 1 bit = 1mV | 0.0625mV
        v_440_sum += v_440_temp;
    }    
    v_raw440 = v_440_sum/10.0; 
    v440 = (v_raw440*0.0625)/(1000); //Converts to a voltage
    
    // Take the average of 10 520 voltage measurements                    
    double v_520_temp;
    double v_520_sum = 0;
    for(int i = 0; i < 10; i++)
    {
        v_520_temp = (double)ads_sun.readADC_SingleEnded(A1_GAIN_TWO); //Channel A0 | 2x gain | +/-2.048V | 1 bit = 1mV | 0.0625mV
        v_520_sum += v_520_temp;
    }    
    v_raw520 = v_520_sum/10.0; 
    v520 = (v_raw520*0.0625)/(1000); //Converts to a voltage
    
    // Take the average of 10 680 voltage measurements                    
    double v_680_temp;
    double v_680_sum = 0;
    for(int i = 0; i < 10; i++)
    {
        v_680_temp = (double)ads_sun.readADC_SingleEnded(A2_GAIN_TWO); //Channel A0 | 2x gain | +/-2.048V | 1 bit = 1mV | 0.0625mV
        v_680_sum += v_680_temp;
    }    
    v_raw680 = v_680_sum/10.0; 
    v680 = (v_raw680*0.0625)/(1000); //Converts to a voltage
    
    // Take the average of 10 870 voltage measurements                      
    double v_870_temp;
    double v_870_sum = 0;
    for(int i = 0; i < 10; i++)
    {
        v_870_temp = (double)ads_sun.readADC_SingleEnded(A3_GAIN_TWO); //Channel A0 | 2x gain | +/-2.048V | 1 bit = 1mV | 0.0625mV
        v_870_sum += v_870_temp;
    }    
    v_raw870 = v_870_sum/10.0; 
    v870 = (v_raw870*0.0625)/(1000); //Converts to a voltage
       
    // Write data to SD card
    FILE *fpAOD = fopen(VoltFile, "a");
    fprintf(fpAOD,"%06d,%06d,%f,%f,%f,%f,%f,%f,%f,%f,%f\r\n",gpsDate,gpsTime,gpslatitude,gpslongitude,gpsaltitude,bme_temp,bme_press,v440,v520,v680,v870);
    fclose(fpAOD);
    free(fpAOD);
    pc.printf("%06d,%06d,%f,%f,%f,%f,%f,%f,%f,%f,%f\r\n",gpsDate,gpsTime,gpslatitude,gpslongitude,gpsaltitude,bme_temp,bme_press,v440,v520,v680,v870);   
}


