/*
Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
THE SOFTWARE.
*/

#include "mbed.h"
#include "rga_func.h"
#include "Images/BinaryImage_RZ_A1H.h"

#define ZOOM_SRC_CENTER_X                   (IMAGE_WIDTH_ZOOM_FUNC / 2)
#define ZOOM_SRC_CENTER_Y                   (IMAGE_HEIGHT_ZOOM_FUNC / 2)

Canvas2D_ContextClass canvas2d;

static animation_timing_function_t* accelerator;

void Set_RGAObject(frame_buffer_t* frmbuf_info) {
    errnum_t err;
    Canvas2D_ContextConfigClass config;

    config.frame_buffer = frmbuf_info;
    canvas2d = R_RGA_New_Canvas2D_ContextClass(config);
    err = R_OSPL_GetErrNum();
    if (err != 0) {
        printf("Line %d, error %d\n", __LINE__, err);
        while (1);
    }
}

void RGA_Func_DrawFullScreen(frame_buffer_t* frmbuf_info, const graphics_image_t* image) {

    /* Draw a image */
    if(image)
    	canvas2d.drawImage(image, 0, 0);
    else
    {
    	/* Clear */
    	canvas2d.clearRect(0, 0, frmbuf_info->width, frmbuf_info->height);
    	//canvas2d.fillStyle = "#000000";
    	//canvas2d.fillRect(0, 0, frmbuf_info->width, frmbuf_info->height);
    }

    /* Complete drawing */
    R_GRAPHICS_Finish(canvas2d.c_LanguageContext);
}

void RGA_Func_DrawPNG(frame_buffer_t* frmbuf_info, const graphics_image_t* png_img, int x, int y) {
    /* Clear */
    //canvas2d.clearRect(0, 0, frmbuf_info->width, frmbuf_info->height);

    /* Draw a image */
	if (frmbuf_info->pixel_format == PIXEL_FORMAT_RGB565)
		canvas2d.drawImage(png_img, x, y);  //png_ARGB4444 assumed
	//else
	//    canvas2d.drawImage(png_img, x, y);

    /* Complete drawing */
    R_GRAPHICS_Finish(canvas2d.c_LanguageContext);
}

void RGA_Func_DrawCamScreen(uint8_t* cam_buffer) {
    /* Clear */
	//if(cam_buffer)
    //canvas2d.clearRect(0, 0, frmbuf_info->width, frmbuf_info->height);

    /* Draw a image */
	if(cam_buffer)
		canvas2d.drawImage((graphics_image_t*)cam_buffer, 0, 0);

    /* Complete drawing */
    R_GRAPHICS_Finish(canvas2d.c_LanguageContext);
}

#if (LCD_VDC5_CH0_PANEL == LCD_CH0_PANEL_VKLCD70RT)

void RGA_Func_DrawRectangle(frame_buffer_t* frmbuf_info, draw_rectangle_pos_t * pos, int pos_num) {
    int cnt;

    /* Clear */
    canvas2d.clearRect(0, 0, frmbuf_info->width, frmbuf_info->height);

    /* Draw description */
    canvas2d.drawImage(decs_swipe_jpg_File, 294, 530);

    /* Draw rectangle */
    for (cnt = 0; cnt < pos_num; cnt++) {
        canvas2d.fillStyle = pos->style;
        canvas2d.fillRect(pos->x, pos->y, pos->w, pos->h);
        pos++;
    }

    /* Draw return button */
    canvas2d.drawImage(RetBtn_jpg_File, 792, 6);

    /* Complete drawing */
    R_GRAPHICS_Finish(canvas2d.c_LanguageContext);
}

void RGA_Func_DrawImage(frame_buffer_t* frmbuf_info, int x, int y) {
    int draw_pos_x = x - (CRYSTAL_DIAMETER / 2);
    int draw_pos_y = y - (CRYSTAL_DIAMETER / 2);

    /* Clear */
    canvas2d.clearRect(0, 0, frmbuf_info->width, frmbuf_info->height);

    /* Draw description */
    canvas2d.drawImage(decs_board_jpg_File, 170, 55);
    canvas2d.drawImage(decs_swipe_jpg_File, 294, 530);

    /* Draw the first image */
    canvas2d.drawImage(Board_jpg_File, 106, 140);

    /* Draw the second image */
    if ((frmbuf_info->pixel_format == PIXEL_FORMAT_ARGB8888) || (frmbuf_info->pixel_format == PIXEL_FORMAT_XRGB8888)) {
        canvas2d.drawImage(Crystal_png_ARGB8888, draw_pos_x, draw_pos_y);
    } else if (frmbuf_info->pixel_format != PIXEL_FORMAT_YUV422) {
        canvas2d.drawImage(Crystal_png_ARGB4444, draw_pos_x, draw_pos_y);
    } else {
        canvas2d.drawImage(Crystal_png_File, draw_pos_x, draw_pos_y);
    }

    /* Draw return button */
    canvas2d.drawImage(RetBtn_jpg_File, 792, 6);

    /* Complete drawing */
    R_GRAPHICS_Finish(canvas2d.c_LanguageContext);
}

void RGA_Func_Dissolve(frame_buffer_t* frmbuf_info, float32_t global_alpha, interact_t mode) {
	/* Dissolve = Fade Out + Fade In */
	/* Clear */
	canvas2d.clearRect(0, 0, frmbuf_info->width, frmbuf_info->height);

	/* Draw description */
	canvas2d.drawImage(decs_board_jpg_File, 170, 55);
	if(mode == AUTO_INTERACT)
	{
	  	canvas2d.drawImage(ManualBtn_jpg_File, 848, 230);
	    canvas2d.drawImage(decs_wait_jpg_File, 405, 530);
	}
	else
	{
	  	canvas2d.drawImage(AutoBtn_jpg_File, 848, 230);
	  	canvas2d.drawImage(decs_swipe_side_jpg_File, 180, 530);
	}

	/* fade Out */
	canvas2d.globalAlpha = 1.0f - global_alpha;
	canvas2d.drawImage(Board_jpg_File, 106, 140);

	/* fade In */
	canvas2d.globalAlpha = global_alpha;
	canvas2d.fillStyle = "#FFFFFF";
	canvas2d.fillRect(106, 140, IMG_DRAW_WIDTH, IMG_DRAW_HEIGHT);

	canvas2d.globalAlpha = 1.0f;

	/* Draw return button */
	canvas2d.drawImage(RetBtn_jpg_File, 792, 6);

	/* Complete drawing */
	R_GRAPHICS_Finish(canvas2d.c_LanguageContext);

}

void RGA_Func_Diss(frame_buffer_t* frmbuf_info, float32_t global_alpha, interact_t mode, graphics_image_t* image) {
    /* Dissolve = Fade Out + Fade In */
	static int prv_mode = mode;
	static int crsr = 0;

	if(!crsr)
		if(prv_mode != mode)
		{
			prv_mode = mode;
			crsr = 2;
		}

	if(crsr)
	{
		canvas2d.clearRect(0, 530, frmbuf_info->width, 44);
		if(mode == AUTO_INTERACT)
		{
			canvas2d.drawImage(ManualBtn_jpg_File, 848, 230);
			canvas2d.drawImage(decs_wait_jpg_File, 405, 530);
		}
		else
		{
			canvas2d.drawImage(AutoBtn_jpg_File, 848, 230);
			canvas2d.drawImage(decs_swipe_side_jpg_File, 180, 530);
		}

		crsr--;
	}

    /* fade Out */
    canvas2d.globalAlpha = 1.0f - global_alpha;
    canvas2d.drawImage(image, 106, 140);

    /* fade In */
    canvas2d.globalAlpha = global_alpha;
    canvas2d.fillStyle = "#FFFFFF";
    canvas2d.fillRect(106, 140, IMG_DRAW_WIDTH, IMG_DRAW_HEIGHT);

    canvas2d.globalAlpha = 1.0f;

    /* Complete drawing */
    R_GRAPHICS_Finish(canvas2d.c_LanguageContext);
}

void RGA_Func_Scroll(frame_buffer_t* frmbuf_info, int src_width_pos,  interact_t mode) {
	/* Clear */
	canvas2d.clearRect(0, 0, frmbuf_info->width, frmbuf_info->height);

	/* Draw description */
	canvas2d.drawImage(decs_board_jpg_File, 170, 55);
	if(mode == AUTO_INTERACT)
	{
		canvas2d.drawImage(ManualBtn_jpg_File, 848, 230);
		canvas2d.drawImage(decs_wait_jpg_File, 405, 530);
	}
	else
	{
		canvas2d.drawImage(AutoBtn_jpg_File, 848, 230);
		canvas2d.drawImage(decs_swipe_side_jpg_File, 180, 530);
	}

	/* Scroll from left to right */
	canvas2d.drawImage(Board_wide_jpg_File,
	                     /* src */       src_width_pos, 0, IMG_DRAW_WIDTH, IMG_DRAW_HEIGHT,
	                     /* dst */       106, 140, IMG_DRAW_WIDTH, IMG_DRAW_HEIGHT);

	/* Draw return button */
	canvas2d.drawImage(RetBtn_jpg_File, 792, 6);

	/* Complete drawing */
	R_GRAPHICS_Finish(canvas2d.c_LanguageContext);
}

void RGA_Func_Scr(frame_buffer_t* frmbuf_info, int src_width_pos, interact_t mode, graphics_image_t* image) {
	static int prv_mode = mode;
	static int crsr = 0;

	if(!crsr)
		if(prv_mode != mode)
		{
			prv_mode = mode;
			crsr = 2;
		}

	if(crsr)
	{
		canvas2d.clearRect(0, 530, frmbuf_info->width, 44);
		if(mode == AUTO_INTERACT)
		{
			canvas2d.drawImage(ManualBtn_jpg_File, 848, 230);
			canvas2d.drawImage(decs_wait_jpg_File, 405, 530);
		}
		else
		{
			canvas2d.drawImage(AutoBtn_jpg_File, 848, 230);
			canvas2d.drawImage(decs_swipe_side_jpg_File, 180, 530);
		}
		crsr--;
	}

	/* Scroll from left to right */
    canvas2d.drawImage(image,
                        /* src */       src_width_pos, 0, IMG_DRAW_WIDTH, IMG_DRAW_HEIGHT,
                        /* dst */       106, 140, IMG_DRAW_WIDTH, IMG_DRAW_HEIGHT);

    /* Complete drawing */
    R_GRAPHICS_Finish(canvas2d.c_LanguageContext);
}

void RGA_Func_Zoom(frame_buffer_t* frmbuf_info, int src_height_pos, interact_t mode) {
	/* Clear */
	canvas2d.clearRect(0, 0, frmbuf_info->width, frmbuf_info->height);

	/* Draw description */
	canvas2d.drawImage(decs_board_jpg_File, 170, 55);
	if(mode == AUTO_INTERACT)
	{
		canvas2d.drawImage(ManualBtn_jpg_File, 848, 230);
		canvas2d.drawImage(decs_wait_jpg_File, 405, 530);
	}
	else
	{
		canvas2d.drawImage(AutoBtn_jpg_File, 848, 230);
		canvas2d.drawImage(decs_swipe_side_jpg_File, 180, 530);
	}

	/* Zoom out */
	canvas2d.drawImage(Board_jpg_File,
	                   /* src X */     ZOOM_SRC_CENTER_X - (src_height_pos * IMG_DRAW_WIDTH / IMG_DRAW_HEIGHT),
	                   /* src Y */     ZOOM_SRC_CENTER_Y - src_height_pos,
	                   /* src W */     src_height_pos * 2 * IMG_DRAW_WIDTH / IMG_DRAW_HEIGHT,
	                   /* src H */     src_height_pos * 2,
	                   /* dst   */     106, 140, IMG_DRAW_WIDTH, IMG_DRAW_HEIGHT);

	/* Draw return button */
	canvas2d.drawImage(RetBtn_jpg_File, 792, 6);

	/* Complete drawing */
	R_GRAPHICS_Finish(canvas2d.c_LanguageContext);
}


void RGA_Func_Zo(frame_buffer_t* frmbuf_info, int src_height_pos, interact_t mode, graphics_image_t* image) {
	static int prv_mode = mode;
	static int crsr = 0;

	if(!crsr)
	if(prv_mode != mode)
	{
		prv_mode = mode;
		crsr = 2;
	}

	if(crsr)
	{
		canvas2d.clearRect(0, 530, frmbuf_info->width, 44);
		if(mode == AUTO_INTERACT)
		{
			canvas2d.drawImage(ManualBtn_jpg_File, 848, 230);
			canvas2d.drawImage(decs_wait_jpg_File, 405, 530);
		}
		else
		{
			canvas2d.drawImage(AutoBtn_jpg_File, 848, 230);
			canvas2d.drawImage(decs_swipe_side_jpg_File, 180, 530);
		}
		crsr--;
	}

    /* Zoom out */
    canvas2d.drawImage(image,
                        /* src X */     ZOOM_SRC_CENTER_X - (src_height_pos * IMG_DRAW_WIDTH / IMG_DRAW_HEIGHT),
                        /* src Y */     ZOOM_SRC_CENTER_Y - src_height_pos,
                        /* src W */     src_height_pos * 2 * IMG_DRAW_WIDTH / IMG_DRAW_HEIGHT,
                        /* src H */     src_height_pos * 2,
                        /* dst   */     106, 140, IMG_DRAW_WIDTH, IMG_DRAW_HEIGHT);

    /* Complete drawing */
    R_GRAPHICS_Finish(canvas2d.c_LanguageContext);
}

void RGA_Func_Rotation(frame_buffer_t* frmbuf_info, graphics_matrix_float_t image_angle, interact_t mode) {
	/* Rotate the image to the right */
	/* Clear */
	canvas2d.clearRect(0, 0, frmbuf_info->width, frmbuf_info->height);

	/* Draw description */
	canvas2d.drawImage(decs_board_jpg_File, 650, 143);
	if(mode == AUTO_INTERACT)
	{
		canvas2d.drawImage(ManualBtn_jpg_File, 848, 230);
	    canvas2d.drawImage(decs_wait_jpg_File, 704, 530);
	}
	else
	{
		canvas2d.drawImage(AutoBtn_jpg_File, 848, 230);
	   	canvas2d.drawImage(decs_swipe_side_jpg_File,   0, 0, 238, 42, 704, 397, 238, 42);
	   	canvas2d.drawImage(decs_swipe_side_jpg_File, 241, 0, 198, 42, 714, 463, 198, 42);
	   	canvas2d.drawImage(decs_swipe_side_jpg_File, 439, 0, 234, 42, 693, 530, 234, 42);
	}

	/* Move to drawing position */
	canvas2d.translate((100 + (IMG_DRAW_WIDTH / 2)), (140 + (IMG_DRAW_HEIGHT / 2)));

	/* Rotate */
	canvas2d.rotate(image_angle * (3.14159 / 180));

	/* Move to center to rotate */
	canvas2d.translate((-IMG_DRAW_WIDTH / 2), (-IMG_DRAW_HEIGHT / 2));
	canvas2d.drawImage(Board_jpg_File, 0, 0);
	canvas2d.setTransform(1, 0, 0, 1, 0, 0);

	/* Draw return button */
	canvas2d.drawImage(RetBtn_jpg_File, 792, 6);

	/* Complete drawing */
	R_GRAPHICS_Finish(canvas2d.c_LanguageContext);
}

void RGA_Func_Rotor(frame_buffer_t* frmbuf_info, graphics_matrix_float_t image_angle, interact_t mode, graphics_image_t* image) {
	/* Rotate the image to the right */
	static int prv_mode = mode;
	static int crsr = 0;

	if(!crsr)
		if(prv_mode != mode)
		{
			prv_mode = mode;
			crsr = 2;
		}

	if(crsr)
	{
		if(mode == AUTO_INTERACT)
		{
			canvas2d.drawImage(ManualBtn_jpg_File, 848, 230);
			canvas2d.clearRect(704, 397, 341, 198);
			canvas2d.drawImage(decs_wait_jpg_File, 704, 530);
		}
		else
		{
			canvas2d.drawImage(AutoBtn_jpg_File, 848, 230);
			canvas2d.drawImage(decs_swipe_side_jpg_File,   0, 0, 238, 42, 704, 397, 238, 42);
			canvas2d.drawImage(decs_swipe_side_jpg_File, 241, 0, 198, 42, 714, 463, 198, 42);
			canvas2d.drawImage(decs_swipe_side_jpg_File, 439, 0, 234, 42, 693, 530, 234, 42);
		}
		crsr--;
	}

    /* Clear [6ms] */
    canvas2d.clearRect(0, 0, 90+(IMG_DRAW_WIDTH / 2)+(frmbuf_info->height / 2), frmbuf_info->height);

    /* Move to drawing position [1ms]*/
    canvas2d.translate((100 + (IMG_DRAW_WIDTH / 2)), (140 + (IMG_DRAW_HEIGHT / 2)));

    /* Rotate [2ms]*/
    canvas2d.rotate(image_angle * (3.14159 / 180));

    /* Move to center to rotate [0,5 + 120 + 1ms]*/
    canvas2d.translate((-IMG_DRAW_WIDTH / 2), (-IMG_DRAW_HEIGHT / 2));
    canvas2d.drawImage(image, 0, 0);
    canvas2d.setTransform(1, 0, 0, 1, 0, 0);

    /* Complete drawing [4ms]*/
    R_GRAPHICS_Finish(canvas2d.c_LanguageContext);
}

void RGA_Func_Accelerate(frame_buffer_t* frmbuf_info, int animation_timing, float32_t relative_pos) {
    /* Do accelerated motion from right to left */
    int position;

    /* Clear */
    canvas2d.clearRect(0, 0, frmbuf_info->width, frmbuf_info->height);

    /* Draw description */
    canvas2d.drawImage(decs_board_jpg_File, 170, 55);

    if (relative_pos == 0.0f) {
        switch (animation_timing) {
            case ANIMATION_TIMING_EASE:
                R_Get_AnimationTimingFunction("ease", &accelerator);
                break;
            case ANIMATION_TIMING_LINEAR:
                R_Get_AnimationTimingFunction("linear", &accelerator);
                break;
            case ANIMATION_TIMING_EASE_IN:
                R_Get_AnimationTimingFunction("ease-in", &accelerator);
                break;
            case ANIMATION_TIMING_EASE_OUT:
                R_Get_AnimationTimingFunction("ease-out", &accelerator);
                break;
            case ANIMATION_TIMING_EASE_IN_OUT:
            default:
                R_Get_AnimationTimingFunction("ease-in-out", &accelerator);
                break;
        }
    }

    /* Draw rectangle */
    canvas2d.fillStyle = "#0000FF";  /* blue */
    switch (animation_timing) {
        case ANIMATION_TIMING_EASE:
            canvas2d.fillRect(21, 562, 173, 13);
            break;
        case ANIMATION_TIMING_LINEAR:
            canvas2d.fillRect(220, 562, 173, 13);
            break;
        case ANIMATION_TIMING_EASE_IN:
            canvas2d.fillRect(422, 562, 173, 13);
            break;
        case ANIMATION_TIMING_EASE_OUT:
            canvas2d.fillRect(623, 562, 173, 13);
            break;
        case ANIMATION_TIMING_EASE_IN_OUT:
        default:
            canvas2d.fillRect(823, 562, 173, 13);
            break;
    }

    /* Draw button */
    canvas2d.drawImage(animetion_timing_jpg_File, 0, 496);

    position = R_ANIMATION_TIMING_FUNCTION_GetValue(accelerator, relative_pos, 505.0f, 6.0f);
    canvas2d.drawImage(Board_jpg_File, position, 140);

    /* Draw return button */
    canvas2d.drawImage(RetBtn_jpg_File, 792, 6);

    /* Complete drawing */
    R_GRAPHICS_Finish(canvas2d.c_LanguageContext);
}

void RGA_Func_Axel(frame_buffer_t* frmbuf_info, int animation_timing, float32_t relative_pos, graphics_image_t* image) {
	/* Do accelerated motion from right to left */
	    static int prv_timing = animation_timing;
	    static int crsr = 0;
		int position;

	    /* Clear */
	    canvas2d.clearRect(0, 140, frmbuf_info->width, IMG_DRAW_HEIGHT);

	    if (relative_pos == 0.0f) {
	        switch (animation_timing) {
	            case ANIMATION_TIMING_EASE:
	                R_Get_AnimationTimingFunction("ease", &accelerator);
	                break;
	            case ANIMATION_TIMING_LINEAR:
	                R_Get_AnimationTimingFunction("linear", &accelerator);
	                break;
	            case ANIMATION_TIMING_EASE_IN:
	                R_Get_AnimationTimingFunction("ease-in", &accelerator);
	                break;
	            case ANIMATION_TIMING_EASE_OUT:
	                R_Get_AnimationTimingFunction("ease-out", &accelerator);
	                break;
	            case ANIMATION_TIMING_EASE_IN_OUT:
	            default:
	                R_Get_AnimationTimingFunction("ease-in-out", &accelerator);
	                break;
	        }
	        if(prv_timing != animation_timing)
	        {
	        	prv_timing = animation_timing;
	        	crsr = 2;
	        }
	    }

	    /* Draw rectangle */
	    canvas2d.fillStyle = "#0000FF";  /* blue */
	    switch (animation_timing) {
	        case ANIMATION_TIMING_EASE:
	        	if(crsr)
	        	{
	        		canvas2d.clearRect(0, 563, frmbuf_info->width, 13);
	        		canvas2d.fillRect(21, 563, 173, 13);
	        	}
	            break;
	        case ANIMATION_TIMING_LINEAR:
	        	if(crsr)
	        	{
	        		canvas2d.clearRect(0, 563, frmbuf_info->width, 13);
	        		canvas2d.fillRect(220, 563, 173, 13);
	        	}
	            break;
	        case ANIMATION_TIMING_EASE_IN:
	        	if(crsr)
	        	{
	        		canvas2d.clearRect(0, 563, frmbuf_info->width, 13);
	        		canvas2d.fillRect(422, 563, 173, 13);
	        	}
	        	break;
	        case ANIMATION_TIMING_EASE_OUT:
	            if(crsr)
	            {
	            	canvas2d.clearRect(0, 563, frmbuf_info->width, 13);
	            	canvas2d.fillRect(623, 563, 173, 13);
	            }
	            break;
	        case ANIMATION_TIMING_EASE_IN_OUT:
	        default:
	        	if(crsr)
	        	{
	        		canvas2d.clearRect(0, 563, frmbuf_info->width, 13);
	        		canvas2d.fillRect(823, 563, 173, 13);
	        	}
	            break;
	    }

	    /* Draw button */
	    if(crsr)
	    {
	    	canvas2d.drawImage(animetion_timing_jpg_File, 0, 496);
    		crsr--;
	    }

	    position = R_ANIMATION_TIMING_FUNCTION_GetValue(accelerator, relative_pos, 505.0f, 6.0f);
	    canvas2d.drawImage(image, position, 140);

	    /* Complete drawing */
	    R_GRAPHICS_Finish(canvas2d.c_LanguageContext);
}

bool RGA_Func_CheckPorts(frame_buffer_t* frmbuf_info, bool rca_pluged, bool usb_pluged) {
	static bool rca_visible = false;
	static bool usb_visible = false;
	bool rca_changed = false;
	bool usb_changed = false;

	/* Detect Change in ports status */
	if(rca_pluged ^ rca_visible)
	{
		rca_visible = rca_pluged;
		rca_changed = true;
	}

	if(usb_pluged ^ usb_visible)
	{
		usb_visible = usb_pluged;
		usb_changed = true;
	}

	if(rca_changed | usb_changed)
	{
		frmbuf_info->draw_buffer_index = (frmbuf_info->draw_buffer_index)? 0 : 1;

		/* Draw a image Clear */
		canvas2d.drawImage(TopScrn_jpg_File, 0, 0);

		if(rca_visible)
			canvas2d.drawImage(RCA_jpg_File, (frmbuf_info->width/2)-160, 50);

		if(usb_visible)
			canvas2d.drawImage(Usb_jpg_File, frmbuf_info->width-100, (frmbuf_info->height/2)-73);

		/* Complete drawing */
		R_GRAPHICS_Finish(canvas2d.c_LanguageContext);
	}

    return (rca_changed | usb_changed);
}
#else

void RGA_Func_DrawRectangle(frame_buffer_t* frmbuf_info, draw_rectangle_pos_t * pos, int pos_num) {
    int cnt;

    /* Clear */
    canvas2d.clearRect(0, 0, frmbuf_info->width, frmbuf_info->height);

    /* Draw description */
    canvas2d.drawImage(decs_swipe_jpg_File, 138, 240);

    /* Draw rectangle */
    for (cnt = 0; cnt < pos_num; cnt++) {
        canvas2d.fillStyle = pos->style;
        canvas2d.fillRect(pos->x, pos->y, pos->w, pos->h);
        pos++;
    }

    /* Draw return button */
    canvas2d.drawImage(RetBtn_jpg_File, 368, 6);

    /* Complete drawing */
    R_GRAPHICS_Finish(canvas2d.c_LanguageContext);
}

void RGA_Func_DrawImage(frame_buffer_t* frmbuf_info, int x, int y) {
    int draw_pos_x = x - (CRYSTAL_DIAMETER / 2);
    int draw_pos_y = y - (CRYSTAL_DIAMETER / 2);

    /* Clear */
    canvas2d.clearRect(0, 0, frmbuf_info->width, frmbuf_info->height);

    /* Draw description */
    canvas2d.drawImage(decs_board_jpg_File, 80, 25);
    canvas2d.drawImage(decs_swipe_jpg_File, 138, 240);

    /* Draw the first image */
    canvas2d.drawImage(Board_jpg_File, 50, 65);

    /* Draw the second image */
    if ((frmbuf_info->pixel_format == PIXEL_FORMAT_ARGB8888) || (frmbuf_info->pixel_format == PIXEL_FORMAT_XRGB8888)) {
        canvas2d.drawImage(Crystal_png_ARGB8888, draw_pos_x, draw_pos_y);
    } else if (frmbuf_info->pixel_format != PIXEL_FORMAT_YUV422) {
        canvas2d.drawImage(Crystal_png_ARGB4444, draw_pos_x, draw_pos_y);
    } else {
        canvas2d.drawImage(Crystal_png_File, draw_pos_x, draw_pos_y);
    }

    /* Draw return button */
    canvas2d.drawImage(RetBtn_jpg_File, 368, 6);

    /* Complete drawing */
    R_GRAPHICS_Finish(canvas2d.c_LanguageContext);
}

void RGA_Func_Dissolve(frame_buffer_t* frmbuf_info, float32_t global_alpha, interact_t mode) {
    /* Dissolve = Fade Out + Fade In */
    /* Clear */
    canvas2d.clearRect(0, 0, frmbuf_info->width, frmbuf_info->height);

    /* Draw description */
    canvas2d.drawImage(decs_board_jpg_File, 80, 25);
    if(mode == AUTO_INTERACT)
    {
    	canvas2d.drawImage(ManualBtn_jpg_File, 398, 104);
        canvas2d.drawImage(decs_wait_jpg_File, 190, 240);
    }
    else
    {
    	canvas2d.drawImage(AutoBtn_jpg_File, 398, 104);
    	canvas2d.drawImage(decs_swipe_side_jpg_File, 84, 240);
    }

    /* fade Out */
    canvas2d.globalAlpha = 1.0f - global_alpha;
    canvas2d.drawImage(Board_jpg_File, 50, 65);

    /* fade In */
    canvas2d.globalAlpha = global_alpha;
    canvas2d.fillStyle = "#FFFFFF";
    canvas2d.fillRect(50, 65, IMG_DRAW_WIDTH, IMG_DRAW_HEIGHT);

    canvas2d.globalAlpha = 1.0f;

    /* Draw return button */
    canvas2d.drawImage(RetBtn_jpg_File, 368, 6);

    /* Complete drawing */
    R_GRAPHICS_Finish(canvas2d.c_LanguageContext);
}

void RGA_Func_Diss(frame_buffer_t* frmbuf_info, float32_t global_alpha, interact_t mode, graphics_image_t* image) {
    /* Dissolve = Fade Out + Fade In */
	static int prv_mode = mode;
	static int crsr = 0;

	if(!crsr)
		if(prv_mode != mode)
		{
			prv_mode = mode;
			crsr = 2;
		}

	if(crsr)
	{
		canvas2d.clearRect(0, 240, frmbuf_info->width, 20);
		if(mode == AUTO_INTERACT)
		{
			canvas2d.drawImage(ManualBtn_jpg_File, 398, 104);
			canvas2d.drawImage(decs_wait_jpg_File, 190, 240);
		}
		else
		{
			canvas2d.drawImage(AutoBtn_jpg_File, 398, 104);
			canvas2d.drawImage(decs_swipe_side_jpg_File, 84, 240);
		}

		crsr--;
	}

    /* fade Out */
    canvas2d.globalAlpha = 1.0f - global_alpha;
    canvas2d.drawImage(image, 50, 65);

    /* fade In */
    canvas2d.globalAlpha = global_alpha;
    canvas2d.fillStyle = "#FFFFFF";
    canvas2d.fillRect(50, 65, IMG_DRAW_WIDTH, IMG_DRAW_HEIGHT);

    canvas2d.globalAlpha = 1.0f;

    /* Complete drawing */
    R_GRAPHICS_Finish(canvas2d.c_LanguageContext);
}

void RGA_Func_Scroll(frame_buffer_t* frmbuf_info, int src_width_pos, interact_t mode) {
    /* Clear */
    canvas2d.clearRect(0, 0, frmbuf_info->width, frmbuf_info->height);

    /* Draw description */
    canvas2d.drawImage(decs_board_jpg_File, 80, 25);
    if(mode == AUTO_INTERACT)
    {
        canvas2d.drawImage(ManualBtn_jpg_File, 398, 104);
        canvas2d.drawImage(decs_wait_jpg_File, 190, 240);
    }
    else
    {
       	canvas2d.drawImage(AutoBtn_jpg_File, 398, 104);
       	canvas2d.drawImage(decs_swipe_side_jpg_File, 84, 240);
    }

    /* Scroll from left to right */
    canvas2d.drawImage(Board_wide_jpg_File,
                        /* src */       src_width_pos, 0, IMG_DRAW_WIDTH, IMG_DRAW_HEIGHT,
                        /* dst */       50, 65, IMG_DRAW_WIDTH, IMG_DRAW_HEIGHT);

    /* Draw return button */
    canvas2d.drawImage(RetBtn_jpg_File, 368, 6);

    /* Complete drawing */
    R_GRAPHICS_Finish(canvas2d.c_LanguageContext);
}

void RGA_Func_Scr(frame_buffer_t* frmbuf_info, int src_width_pos, interact_t mode, graphics_image_t* image) {
	static int prv_mode = mode;
	static int crsr = 0;

	if(!crsr)
		if(prv_mode != mode)
		{
			prv_mode = mode;
			crsr = 2;
		}

	if(crsr)
	{
		canvas2d.clearRect(0, 240, frmbuf_info->width, 20);
		if(mode == AUTO_INTERACT)
		{
			canvas2d.drawImage(ManualBtn_jpg_File, 398, 104);
			canvas2d.drawImage(decs_wait_jpg_File, 190, 240);
		}
		else
		{
			canvas2d.drawImage(AutoBtn_jpg_File, 398, 104);
			canvas2d.drawImage(decs_swipe_side_jpg_File, 84, 240);
		}
		crsr--;
	}

	/* Scroll from left to right */
    canvas2d.drawImage(image,
                        /* src */       src_width_pos, 0, IMG_DRAW_WIDTH, IMG_DRAW_HEIGHT,
                        /* dst */       50, 65, IMG_DRAW_WIDTH, IMG_DRAW_HEIGHT);

    /* Complete drawing */
    R_GRAPHICS_Finish(canvas2d.c_LanguageContext);
}

void RGA_Func_Zoom(frame_buffer_t* frmbuf_info, int src_height_pos, interact_t mode) {
    /* Clear */
    canvas2d.clearRect(0, 0, frmbuf_info->width, frmbuf_info->height);

    /* Draw description */
    canvas2d.drawImage(decs_board_jpg_File, 80, 25);
    if(mode == AUTO_INTERACT)
    {
        canvas2d.drawImage(ManualBtn_jpg_File, 398, 104);
        canvas2d.drawImage(decs_wait_jpg_File, 190, 240);
    }
    else
    {
      	canvas2d.drawImage(AutoBtn_jpg_File, 398, 104);
       	canvas2d.drawImage(decs_swipe_side_jpg_File, 84, 240);
    }

    /* Zoom out */
    canvas2d.drawImage(Board_jpg_File,
                        /* src X */     ZOOM_SRC_CENTER_X - (src_height_pos * IMG_DRAW_WIDTH / IMG_DRAW_HEIGHT),
                        /* src Y */     ZOOM_SRC_CENTER_Y - src_height_pos,
                        /* src W */     src_height_pos * 2 * IMG_DRAW_WIDTH / IMG_DRAW_HEIGHT,
                        /* src H */     src_height_pos * 2,
                        /* dst   */     50, 65, IMG_DRAW_WIDTH, IMG_DRAW_HEIGHT);

    /* Draw return button */
    canvas2d.drawImage(RetBtn_jpg_File, 368, 6);

    /* Complete drawing */
    R_GRAPHICS_Finish(canvas2d.c_LanguageContext);
}

void RGA_Func_Zo(frame_buffer_t* frmbuf_info, int src_height_pos, interact_t mode, graphics_image_t* image) {
	static int prv_mode = mode;
	static int crsr = 0;

	if(!crsr)
	if(prv_mode != mode)
	{
		prv_mode = mode;
		crsr = 2;
	}

	if(crsr)
	{
		canvas2d.clearRect(0, 240, frmbuf_info->width, 20);
		if(mode == AUTO_INTERACT)
		{
			canvas2d.drawImage(ManualBtn_jpg_File, 398, 104);
			canvas2d.drawImage(decs_wait_jpg_File, 190, 240);
		}
		else
		{
			canvas2d.drawImage(AutoBtn_jpg_File, 398, 104);
			canvas2d.drawImage(decs_swipe_side_jpg_File, 84, 240);
		}
		crsr--;
	}

    /* Zoom out */
    canvas2d.drawImage(image,
                        /* src X */     ZOOM_SRC_CENTER_X - (src_height_pos * IMG_DRAW_WIDTH / IMG_DRAW_HEIGHT),
                        /* src Y */     ZOOM_SRC_CENTER_Y - src_height_pos,
                        /* src W */     src_height_pos * 2 * IMG_DRAW_WIDTH / IMG_DRAW_HEIGHT,
                        /* src H */     src_height_pos * 2,
                        /* dst   */     50, 65, IMG_DRAW_WIDTH, IMG_DRAW_HEIGHT);

    /* Complete drawing */
    R_GRAPHICS_Finish(canvas2d.c_LanguageContext);
}

void RGA_Func_Rotation(frame_buffer_t* frmbuf_info, graphics_matrix_float_t image_angle, interact_t mode) {
    /* Rotate the image to the right */
    /* Clear */
    canvas2d.clearRect(0, 0, frmbuf_info->width, frmbuf_info->height);

    /* Draw description */
    canvas2d.drawImage(decs_board_jpg_File, 305, 65);
    if(mode == AUTO_INTERACT)
    {
    	canvas2d.drawImage(ManualBtn_jpg_File, 398, 104);
    	canvas2d.drawImage(decs_wait_jpg_File, 330, 240);
    }
    else
    {
    	canvas2d.drawImage(AutoBtn_jpg_File, 398, 104);
    	canvas2d.drawImage(decs_swipe_side_jpg_File,   0, 0, 112, 19, 330, 180, 112, 19);
    	canvas2d.drawImage(decs_swipe_side_jpg_File, 113, 0, 93,  19, 335, 210,  93, 19);
    	canvas2d.drawImage(decs_swipe_side_jpg_File, 206, 0, 110, 19, 325, 240, 110, 19);
    }

    /* Move to drawing position */
    canvas2d.translate((50 + (IMG_DRAW_WIDTH / 2)), (65 + (IMG_DRAW_HEIGHT / 2)));

    /* Rotate */
    canvas2d.rotate(image_angle * (3.14159 / 180));

    /* Move to center to rotate */
    canvas2d.translate((-IMG_DRAW_WIDTH / 2), (-IMG_DRAW_HEIGHT / 2));
    canvas2d.drawImage(Board_jpg_File, 0, 0);
    canvas2d.setTransform(1, 0, 0, 1, 0, 0);

    /* Draw return button */
    canvas2d.drawImage(RetBtn_jpg_File, 368, 6);

    /* Complete drawing */
    R_GRAPHICS_Finish(canvas2d.c_LanguageContext);
}

void RGA_Func_Rotor(frame_buffer_t* frmbuf_info, graphics_matrix_float_t image_angle, interact_t mode, graphics_image_t* image) {
	/* Rotate the image to the right */
	static int prv_mode = mode;
	static int crsr = 0;

	if(!crsr)
		if(prv_mode != mode)
		{
			prv_mode = mode;
			crsr = 2;
		}

	if(crsr)
	{
		//canvas2d.clearRect(0, 240, frmbuf_info->width, 20);
		if(mode == AUTO_INTERACT)
		{
			canvas2d.drawImage(ManualBtn_jpg_File, 398, 104);
			canvas2d.clearRect(330, 180, 160, 90);
			canvas2d.drawImage(decs_wait_jpg_File, 330, 240);
		}
		else
		{
			canvas2d.drawImage(AutoBtn_jpg_File, 398, 104);
			canvas2d.drawImage(decs_swipe_side_jpg_File,   0, 0, 112, 19, 330, 180, 112, 19);
			canvas2d.drawImage(decs_swipe_side_jpg_File, 113, 0, 93,  19, 335, 210,  93, 19);
			canvas2d.drawImage(decs_swipe_side_jpg_File, 206, 0, 110, 19, 325, 240, 110, 19);
		}
		crsr--;
	}

    /* Clear [6ms] */
    canvas2d.clearRect(0, 0, 50+(IMG_DRAW_WIDTH / 2)+(frmbuf_info->height / 2), frmbuf_info->height);

    /* Move to drawing position [1ms]*/
    canvas2d.translate((50 + (IMG_DRAW_WIDTH / 2)), (65 + (IMG_DRAW_HEIGHT / 2)));

    /* Rotate [2ms]*/
    canvas2d.rotate(image_angle * (3.14159 / 180));

    /* Move to center to rotate [0,5 + 120 + 1ms]*/
    canvas2d.translate((-IMG_DRAW_WIDTH / 2), (-IMG_DRAW_HEIGHT / 2));
    canvas2d.drawImage(image, 0, 0);
    canvas2d.setTransform(1, 0, 0, 1, 0, 0);

    /* Complete drawing [4ms]*/
    R_GRAPHICS_Finish(canvas2d.c_LanguageContext);
}

void RGA_Func_Accelerate(frame_buffer_t* frmbuf_info, int animation_timing, float32_t relative_pos) {
    /* Do accelerated motion from right to left */
    int position;

    /* Clear */
    canvas2d.clearRect(0, 0, frmbuf_info->width, frmbuf_info->height);

    /* Draw description */
    canvas2d.drawImage(decs_board_jpg_File, 80, 25);

    if (relative_pos == 0.0f) {
        switch (animation_timing) {
            case ANIMATION_TIMING_EASE:
                R_Get_AnimationTimingFunction("ease", &accelerator);
                break;
            case ANIMATION_TIMING_LINEAR:
                R_Get_AnimationTimingFunction("linear", &accelerator);
                break;
            case ANIMATION_TIMING_EASE_IN:
                R_Get_AnimationTimingFunction("ease-in", &accelerator);
                break;
            case ANIMATION_TIMING_EASE_OUT:
                R_Get_AnimationTimingFunction("ease-out", &accelerator);
                break;
            case ANIMATION_TIMING_EASE_IN_OUT:
            default:
                R_Get_AnimationTimingFunction("ease-in-out", &accelerator);
                break;
        }
    }

    /* Draw rectangle */
    canvas2d.fillStyle = "#0000FF";  /* blue */
    switch (animation_timing) {
        case ANIMATION_TIMING_EASE:
            canvas2d.fillRect(10, 255, 81, 6);
            break;
        case ANIMATION_TIMING_LINEAR:
            canvas2d.fillRect(103, 255, 81, 6);
            break;
        case ANIMATION_TIMING_EASE_IN:
            canvas2d.fillRect(198, 255, 81, 6);
            break;
        case ANIMATION_TIMING_EASE_OUT:
            canvas2d.fillRect(292, 255, 81, 6);
            break;
        case ANIMATION_TIMING_EASE_IN_OUT:
        default:
            canvas2d.fillRect(386, 255, 81, 6);
            break;
    }

    /* Draw button */
    canvas2d.drawImage(animetion_timing_jpg_File, 0, 225);

    position = R_ANIMATION_TIMING_FUNCTION_GetValue(accelerator, relative_pos, 240.0f, 6.0f);
    canvas2d.drawImage(Board_jpg_File, position, 65);

    /* Draw return button */
    canvas2d.drawImage(RetBtn_jpg_File, 368, 6);

    /* Complete drawing */
    R_GRAPHICS_Finish(canvas2d.c_LanguageContext);
}

void RGA_Func_Axel(frame_buffer_t* frmbuf_info, int animation_timing, float32_t relative_pos, graphics_image_t* image) {
	/* Do accelerated motion from right to left */
	    static int prv_timing = animation_timing;
	    static int crsr = 0;
		int position;

	    /* Clear */
	    canvas2d.clearRect(0, 65, frmbuf_info->width, IMG_DRAW_HEIGHT);

	    if (relative_pos == 0.0f) {
	        switch (animation_timing) {
	            case ANIMATION_TIMING_EASE:
	                R_Get_AnimationTimingFunction("ease", &accelerator);
	                break;
	            case ANIMATION_TIMING_LINEAR:
	                R_Get_AnimationTimingFunction("linear", &accelerator);
	                break;
	            case ANIMATION_TIMING_EASE_IN:
	                R_Get_AnimationTimingFunction("ease-in", &accelerator);
	                break;
	            case ANIMATION_TIMING_EASE_OUT:
	                R_Get_AnimationTimingFunction("ease-out", &accelerator);
	                break;
	            case ANIMATION_TIMING_EASE_IN_OUT:
	            default:
	                R_Get_AnimationTimingFunction("ease-in-out", &accelerator);
	                break;
	        }
	        if(prv_timing != animation_timing)
	        {
	        	prv_timing = animation_timing;
	        	crsr = 2;
	        }
	    }

	    /* Draw rectangle */
	    canvas2d.fillStyle = "#0000FF";  /* blue */
	    switch (animation_timing) {
	        case ANIMATION_TIMING_EASE:
	        	if(crsr)
	        	{
	        		canvas2d.clearRect(0, 255, frmbuf_info->width, 6);
	        		canvas2d.fillRect(10, 255, 81, 6);
	        	}
	            break;
	        case ANIMATION_TIMING_LINEAR:
	        	if(crsr)
	        	{
	        		canvas2d.clearRect(0, 255, frmbuf_info->width, 6);
	        		canvas2d.fillRect(103, 255, 81, 6);
	        	}
	            break;
	        case ANIMATION_TIMING_EASE_IN:
	        	if(crsr)
	        	{
	        		canvas2d.clearRect(0, 255, frmbuf_info->width, 6);
	        		canvas2d.fillRect(198, 255, 81, 6);
	        	}
	        	break;
	        case ANIMATION_TIMING_EASE_OUT:
	            if(crsr)
	            {
	            	canvas2d.clearRect(0, 255, frmbuf_info->width, 6);
	            	canvas2d.fillRect(292, 255, 81, 6);
	            }
	            break;
	        case ANIMATION_TIMING_EASE_IN_OUT:
	        default:
	        	if(crsr)
	        	{
	        		canvas2d.clearRect(0, 255, frmbuf_info->width, 6);
	        		canvas2d.fillRect(386, 255, 81, 6);
	        	}
	            break;
	    }

	    /* Draw button */
	    if(crsr)
	    {
	    	canvas2d.drawImage(animetion_timing_jpg_File, 0, 225);
    		crsr--;
	    }

	    position = R_ANIMATION_TIMING_FUNCTION_GetValue(accelerator, relative_pos, 240.0f, 6.0f);
	    canvas2d.drawImage(image, position, 65);

	    /* Complete drawing */
	    R_GRAPHICS_Finish(canvas2d.c_LanguageContext);
}

bool RGA_Func_CheckPorts(frame_buffer_t* frmbuf_info, bool rca_pluged, bool usb_pluged) {
	static bool rca_visible = false;
	static bool usb_visible = false;
	bool rca_changed = false;
	bool usb_changed = false;

	/* Detect Change in ports status */
	if(rca_pluged ^ rca_visible)
	{
		rca_visible = rca_pluged;
		rca_changed = true;
	}

	if(usb_pluged ^ usb_visible)
	{
		usb_visible = usb_pluged;
		usb_changed = true;
	}

	if(rca_changed | usb_changed)
	{
		frmbuf_info->draw_buffer_index = (frmbuf_info->draw_buffer_index)? 0 : 1;

		/* Draw a image Clear */
		canvas2d.drawImage(TopScrn_jpg_File, 0, 0);

		if(rca_visible)
			canvas2d.drawImage(RCA_jpg_File, (frmbuf_info->width/2)-80, 22);

		if(usb_visible)
			canvas2d.drawImage(Usb_jpg_File, frmbuf_info->width-50, (frmbuf_info->height/2)-36);

		/* Complete drawing */
		R_GRAPHICS_Finish(canvas2d.c_LanguageContext);
	}

    return (rca_changed | usb_changed);
}
#endif

