/**
 * @file timeslice.h
 * @Synopsis Implementa o gerenciamento do uso de timeslices, necessario para a comunicação no protocolo CBx->Header 
 * @author Jhonatan Casale
 * @version 1
 * @date 2014-10-31
 * \class Timeslice
 */

#ifndef __TIMESLICE_H__
#define __TIMESLICE_H__

#include <stdint.h>

const uint8_t MAX_TIMESLICES = 4;
///< Por decisão de projeto, este foi dimensionado para o uso de 4 timeslices

class Timeslice{
    private :
        uint8_t timeslice[ MAX_TIMESLICES ];
    public :
        /**
         * @Synopsis Cria e inicializa o objeto, pronto para o uso.
         *
         * Exemplo:
         * @code
         * ...
         *  Timeslice * ts = new Timeslice();
         * ...
         * @endcode
         */
        Timeslice();
        
        /**
         * @Synopsis Destroi o elemento.
         *
         * Exemplo:
         * @code
         * ...
         *  delete( ts );
         * ...
         * @endcode
         */
        ~Timeslice();
        
        /**
         * @Synopsis Busca por timeslices disponiveis.
         *
         * @return zero, caso todos os timeslices já estiverem em uso, ou retorna um
         * numero maior que zero que representa justamente o timeslice.
         *
         * Exemplo:
         * @code
         * ...
         *  ts->get_timeslice();
         * ...
         * @endcode
         */
        uint8_t get_timeslice();
        
        /**
         * @Synopsis Coloca o timeslice devolvido a disposição para novos usos. 
         *
         * @param i O timeslice que esta sendo devolvido.
         *
         * Exemplo:
         * @code
         * ...
         *  ts->return_timeslice( 0x03 );
         * @endcode
         */
        void return_timeslice( uint8_t i );
        
        /**
         * @Synopsis Reinicia a estrutura, deixando todos os timeslices a disposição para uso.
         *
         *
         * Exemplo:
         * @code
         * ...
         *  ts->reset();
         * ...
         * @endcode
         */
        void reset();
        
        /**
         * @Synopsis Representa quantos timeslices ainda estão disponiveis para uso.
         *
         * @return O numero de timeslices ainda desocupados. Este numero é sempre maior ou igual a zero.
         *
         * Exemplo:
         * @code
         * ...
         *  ts->remain_timeslices();
         * ...
         * @endcode
         */
        uint8_t remain_timeslices( void );
};

#endif