/**
 * @file sip.h
 * @Synopsis Implementa as funções utilizadas no tratamento SIP entre a Header e o server
 * @author Jhonatan Casale
 * @version 1
 * @date 2014-11-05
 * \class Sip
 */
#ifndef __SIP_H__
#define __SIP_H__

#include "EthernetInterface.h"
#include "mbed.h"
#include <stdint.h>
#include "call.h"
%: include "config_manager.h"
%: include "shared_variables.h"

const uint8_t INVITE_MAX_WAITING_TIME = 45;
///< Indica o timeout de espera de resposta de pedido de ligação para o servidor, após esse tempo responde ligação encerrado para o Call_box

const uint16_t SIP_MAXFIELDSIZE = 256;
///< Define o tamanho máximo de algumas mensagens usadas na negociação Sip.

const uint16_t SIP_MAXMSGSIZE = 1024;
///< Define o tamanho máximo das mensagens enviadas.

static const char SIP_ALLOW[] = "Allow: ACK, BYE, CANCEL, INFO, INVITE, NOTIFY, OPTIONS, REFER";
///< String de composição de pacotes enviados pela Header para o *

static const char fill_random_aux[] = "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789pP";

const uint8_t CALLBOX_STRING_SIZE = 32;

const uint8_t CONTENT_LENGTH_MSG_SIZE = 65;

const uint8_t sip_idle = 0;
const uint8_t sip_waiting_trying = 1 << 1;
const uint8_t sip_trying = 1 << 2;
const uint8_t sip_ringing = 1 << 3;
const uint8_t sip_busy = 1 << 4;
const uint8_t sip_ok = 1 << 5; // don't used
const uint8_t sip_on_call = 1 << 6;
const uint8_t sip_denied = 1 << 7;

class Sip{

private :
    uint8_t status; ///< Representa o status do objeto Sip em dado momento.

    int my_ext;
    int my_port;
    int peer_ext;
    int server_port;
    int my_rtp_port;
    
    int listen_SIP_server_return;
    int last_cseq;
    
    char server_ip[ 20 ];
    char my_ip[ 20 ];
    char my_display[ 20 ];
    char SVNREV[ 16 ];
    char buffer[ SIP_MAXMSGSIZE ];
    char last_invite_tag [ SIP_MAXFIELDSIZE ];
    char last_invite_callid [ SIP_MAXFIELDSIZE ];
    char last_branch [ SIP_MAXFIELDSIZE ];
    
    void __init_sock__ ( void );
    void __end_sock__ ( void );
    void __reconnect__ ( void );
    
    bool waiting;
    
    UDPSocket sock;
    
    Endpoint sip_server;
    
    VZ_call * call;
    
    Timer invite_timer;
    
public :
        /**
         * @Synopsis Cria um objeto Sip setando ramal e porta e o restando buscando valores default.
         *
         * @param id O identificador do objeto ( por definição de projeto o ramal ( ext ) ).
         * @param my_port A porta do objeto Sip para comunicação com o servidor.
         *
         * Exemplo:
         *
         * @code
         * ...
         *  Sip * sip = new Sip( 5002, 5002 );
         * ...
         * @endcode
         */
    Sip ( const int my_ext, const uint16_t my_port );

        /**
         * @Synopsis Destroi o objeto Sip
         *
         * Exemplo:
         *
         * @code
         * ...
         *  delete( sip );
         * ...
         * @endcode
         */
    ~Sip ( void );

        /**
         * @Synopsis Responsavel por montar encaminhar o pacote de registro para o servidor.
         *
         * Exemplo:
         * @code
         * ...
         *  sip->registry();
         * ...
         * @endcode
         */
    int registry ( void );

        /**
         * @Synopsis Efetivamente envia o pedido de ligação para o servidor.
         *
         * @return Uma referência para uma ligação, caso a negociação resulte na aceitação por parte do servidor, NULL, caso contrário.
         * \note Como por conveniência, foi configurado no servidor * que existiria musica de conforto, o que fez com que, assim que é 
         * feito o envio do pacote de invite, uma chamada é criada para se trocar os dados desse audio de conforto, ficando a Header na
         * espera do servidor aceitar ou não o pedido de forma definitiva, no caso em que o server dizer "ok", a mesma chamada é retornada,
         * porém, no caso de timeout ou negativa do server, a chamada é destruida e NULL é retornado.
         *
         * Exemplo:
         * @code
         * ...
         *  VZ_call * call = sip->invite();
         * ...
         * @endcode
         */
    VZ_call * invite ( void );

        /**
         * @Synopsis Gera um nro de cseq para preenchimento no cabeçalho dos pacotes Sip.
         *
         * @return O numero gerado.
         *
         * Exemplo:
         * @code
         * ...
         *  int cseq = sip->get_cseq();
         * ...
         * @endcode
         */
    int get_cseq ( void );
    
        /**
         * @Synopsis Efetivamente cria o pacote de registro.
         *
         * @param buffer Recebe um ponteiro para uma região de memória onde irá escrever o pacote de registro.
         *
         * @return Um ponteiro para o inicio do pacote montado ( mesmo endereço passado como parâmetro ).
         */
    char * build_registry_package ( void );
    
        /**
         * @Synopsis Efetivamente cria um pacote de invite ( pedido de chamada ) do Call_box para o server.
         *
         * @param s Um ponteiro para uma região de memória onde irá escrever o pacote de invite.
         * @param callbox_string Uma string de identifcação do Call_box.
         * @param cseq O nro de cseq que irá ser colocado no pacote.
         *
         * @return 
         */
    char * build_invite_package ( int * cseq, const bool retry = false );
    
        /**
         * @Synopsis Efetivamente cria um pacote de "despedida" para o servidor, usado para encerrar ligações.
         *
         * @param buffer Um ponteiro para uma região de memória onde irá escrever o pacote de bye.
         *
         * @return Um ponteiro para o inicio do pacote montado ( mesmo endereço passado como parâmetro ).
         */
    char * build_bye_package ( void );
    
        /**
         * @Synopsis Efetivamente cria um pacote de ack de mensagem recebida.
         *
         * @param buffer Um ponteiro para a posição de memória onde será criado o pacote ack
         * @param orig O pacote original recebido do servidor, serve como referência na criação do pacote de resposta,
         *
         * @return Um ponteiro para o inicio do pacote montado ( mesmo endereço passado como parâmetro ).
         */
    char * build_ack_package ( void );
    
        /**
         * @Synopsis 
         * 
         * @param buffer Um ponteiro para a posição de memória onde será montado o pacote.
         * @param orig O pacote de origem, serve como referência na criação do pacote.
         *
         * @return Um ponteiro para o inicio do pacote montado ( mesmo endereço passado como parâmetro ).
         */
    char * build_reply_package ( void );
    
        /**
         * @Synopsis Preenche de forma aleatótia 16 posições de memória.
         *
         * @param buffer Um ponteiro que aponta para o inicio das 16 ( bytes ) posições que serão preenchidas.
         *
         * @return Retorna um ponteiro para os dados preenchidos ( mesmo endereço passado como parâmetro.
         */
    char * fill_random16h ( char * buffer );
    
        /**
         * @Synopsis Preenche de forma aleatótia dados.
         *
         * @param buffer Um ponteiro que aponta para o inicio das posições a serem preenchidas.
         * @param size A quantidade ( em posições ) que serão preenchidas.
         *
         * @return Um ponteiro para o inicio das posições preenchidas ( mesmo endereço do parâmetro passado ).
         */
    char * fill_random ( char * buffer, const int size );
    
        /**
         * @Synopsis Gera aleatóriamente a porta RTP de onde os dados de audiao serão enviados da Header.
         *
         * @return O nro da porta que será usada na negociação Sip.
         */
    int fill_random_rtp_port ( void );
    
        /**
         * @Synopsis Dado um pacote recebido do server, decodifica o nro cseq.
         *
         * @param package O pacote do qual se tem interesse em decodificar.
         * @param cseq Um ponteiro que aponta para a posição de memória onde será escrito o cseq.
         *
         * @return Um ponteiro para o inicio da posição de memória que contem o cseq ( mesma passada como parâmetro ).
         */
    char * decode_cseq ( const unsigned char * package, char * cseq );
    
        /**
         * @Synopsis Dado um pacote recebido do server, decodifica o branch.
         *
         * @param package Um ponteiro para o pacote recebido do servidor o qual temos interesse em decodificar.
         * @param branch Um ponteiro para o inicio da posição de memória onde escreveremos o branch
         *
         * @return Um ponteiro para o inicio da posição de memória onde escrevemos o branch ( mesma passada como parâmentro ).
         */
    char * decode_branch ( const unsigned char * package, char * branch );
    
        /**
         * @Synopsis Busca por uma determinado substring em um pacote recebido do servidor. Copiando o restando desse pacote para
         * uma posição de interesse.
         *
         * @param package O pacote que se tem interesse em decodificar.
         * @param tag A substring que a função ira buscar em package.
         * @param out A região de memória para onde o restante dos dados contidos em package será copiado, caso a substring seja 
         * encontrada.
         *
         * @return 1, caso tenha encontrado a string tag contida no package, copia o restando do conteudo de package para out e retorna 1,
         * retorna 0 caso a substring não seja encontrada.
         */
    int decode_gettag ( const unsigned char * package, const char * tag, char * out );
    
        /**
         * @Synopsis Recebe um ramal de comunicação com o servidor, realizando solicitações futuras com esse ramal.
         *
         * @param new_server_ext O Ramal do servidor que será associado ao objeto Sip
         *
         * Exemplo:
         * @code
         * ...
         *  sip->set_server_ext( 913 );
         * ...
         * @endcode
         */
    void set_server_ext ( const int new_server_ext );
    
        /**
         * @Synopsis Recebe uma porta de comunicação com o servidor, setando e realizando comunicaões entre Header-Server atraves 
         * dessa porta
         *
         * @param new_server_port A porta do servidor que será associada ao objeto Sip
         *
         * Exemplo:
         * @code
         * ...
         *  sip->set_server_port( 818 );
         * ...
         * @endcode
         */
    void set_server_port ( const int new_server_port );
    
        /**
         * @Synopsis Seta a porta de comunicação Sip da Header.
         *
         * @param port Porta que será associada ao objeto Sip criado.
         *
         * Exemplo:
         * @code
         * ...
         *  sipset_port( 1028 );
         * ...
         * @endcode
         *
         *
         */
    void set_port ( const int port );
    
        /**
         * @Synopsis Seta o valor passado como parâmetro como sendo o endeço IP de contato do servidor.
         *
         * @param new_server_ip O endereço que será associado ao objeto Sip.
         *
         * Exemplo:
         * @code
         * ...
         *  sip->set_server_ip( "192.168.120.8" );
         * ...
         * @endcode
         */
    void set_server_ip ( const char * new_server_ip );
    
        /**
         * @Synopsis Responsavel por ouvir o servidor *, verificando se o mesmo esta mandando alguma mensagem, seja Sip ou qualquer outra
         * nas portas acordadas.
         *
         * @return 0, sucesso na execução, sem pendencia, ou retorna um nro maior que zero, representando o nro do ramal cuja ligação deve
         * ser encerrada.
         */
    int listen_SIP_server ( void );
    
        /**
         * @Synopsis Envia efetivamente o pacote de despedita com Call_box pro servidor
         *
         * Exemplo:
         * @code
         * ...
         *  sip->send_bye();
         * ...
         * @endcode
         */
    void send_bye ( void );
    
        /**
         * @Synopsis Retorna o status atual do objeto
         *
         * @return O valor correspondente ao status do objeto
         */
    int get_status ( void );
    
    /*------------------------------------------------------------------------------------------------*/
    int get_socket_fd ( void );
    
    int udp_incomming_pkg ( void );
    
    int get_ext ( void );
    
    int get_port ( void );
    
    int get_sip_rtp_port ( void );
    
    int print_yourself ( void );
    
    int retry_send_last_invite_pkg_to_ast ( void );
    
    
    void set_sip_rtp_port ( const int new_my_rtp_port );
    
    void sip_set_status ( const uint8_t status );
    
    void reset_call ( void );
    
    void update ( void );
};
#endif