/**
 * @file call_box.h
 * @Synopsis Implementa as funções de gerenciamento do Call_Box 
 * @author Jhonatan Casale
 * @version 1
 * @date 2014-11-06
 * \class Call_Box
 */
#ifndef __CALL_BOX_H__
#define __CALL_BOX_H__

#include <stdint.h>
#include "object.h"
#include "sip.h"
%: include "config_manager.h"
%: include "shared_variables.h"

const int __RESQUEST_TIME__ = 3;

const uint16_t MAX_INVITE_RETRY = 512 * 2;

///< O tempo que a Header demora para "pingar" o Call_Box para saber se esta tudo bem.
const uint8_t __RANGE_TO_REGISTER__ = 150;
///< Doc. later

const uint8_t RTP_REQUEST_PORT_TIMEOUT = 45;

const short int REQUEST_REGISTRY = -1;
const short int REQUEST_PEND = -2;
const short int REQUEST_PING = -4;
///< Valor usado como referencia para registro do equipamento CBx.

const uint8_t __STEP__ = 30;
///< Usado para aumentar o intervalo entre cada ping, fora de uso atualmente.

const uint8_t cb_idle = 1;
///< Representa que o Call_Box esta disponivel

const uint8_t cb_ringing = 2;
///< Representa o Call_Box no status, discando.

const uint8_t cb_trying = 3;
///< Representa o Call_Box no status tentando concluir o pedido de ligação.

const uint8_t cb_on_call = 4;
///< Representa que o Call_Box já esta em uma ligação.

const uint8_t cb_busy = 5;
///< Representa que o Call_Box esta ocupado.

const uint8_t cb_denied = 6;
///< Representa que o pedido de invite do Call_Box foi negado.

const uint8_t cb_bootloader = 7;
///< Representa que o Call_Box esta em atualizacao de firmware  


class Call_Box : public Object {
    private :
        /**
          * @Synopsis Objeto Sip que será usado para tratativas com o servidor.
          *
          * \note Esse objeto é criado no construtor da classe Call_Box, com os mesmos parâmetros passados para o 
          * construtor da classe Call_Box.
          */
        Sip * sip;
    
        bool invite_response;
        bool bye_response;
        bool overflow;
        
        int ext;  // ramal
        int port; // porta
        int sip_socket_fd;
        int shift_port;
        
        /**
          * @Synopsis Representa o status do Call_Box.
          *
          * \note Este valor sempre é atualizado ( por convenção ) com o uso da macro set_status(a,b), definida em utils.h
          */
        uint8_t status;
        
        uint8_t overflow_times;
        uint8_t msg_id;
        uint8_t timeslice;
        uint8_t next_aging_type;
        uint8_t  invite_try_number;
        
        uint16_t invite_retry_count;
        uint16_t invite_counter;
        
        //FIXME remove timeout
        float timeout;
        
        Timer t;
        Timer rtp_timer; // usado pra controlar o timeout dos pedidos de porta rtp
        Timer invite_timer;
        
        Call_Box * pair;
    public :
                /**
                 * @Synopsis Cria um objeto Call_Box
                 *
                 * @param ext Vincula o objeto ao ramal informado
                 * @param port Vincula o objeto a porto informada.
                 *
                 * Exemplo:
                 * @code
                 * ...
                 *  int ext = 5121;
                 *  int port = 5021;
                 *  Call_Box * cb = new Call_Box( ext, port );
                 * ...
                 * @endcode
                 */
        Call_Box ( const int ext, const int port );

                /**
                 * @Synopsis Destroi o objeto Call_Box
                 *
                 * Exemplo:
                 * @code
                 * ...
                 *  delete( cb );
                 * ...
                 * @endcode
                 * \note Deleta o objeto Sip nesse processo.
                 */
        ~Call_Box ( void );
        
                /**
                 * @Synopsis Informa o ramal vinculado ao objeto Call_Box.
                 *
                 * @return O numero do ramal contido atualmente no objeto Call_Box
                 *
                 * Exemplo:
                 * @code
                 * ...
                 *  int ext = cb->get_ext();
                 * ...
                 * @endcode
                 */
        int get_ext ( void );

                /**
                 * @Synopsis Informa o numero da porta vinculada ao objeto Call_Box.
                 *
                 * @return O numero da porta contido atualmente no objeto Call_Box
                 *
                 * Exemplo:
                 * @code
                 * ...
                 *  int port = cb->get_port();
                 * ...
                 * @endcode
                 */
        int get_port ( void );
       
                /**
                 * @Synopsis Informa o tempo decorrido.
                 * \note O timer de cada objeto é iniciado no momento da criação do objeto Call_Box.
                 *
                 * @return O tempo decorrido desde a ultima vez que o timer do objeto foi resetado.
                 *
                 * Exemplo:
                 * @code
                 * ...
                 *  float elapsed_time = cb->get_elapsed_time();
                 * ...
                 * @endcode
                 */
        uint16_t get_elapsed_time ( void );

                /**
                 * @Synopsis Reseta o timer do objeto Call_Box.
                 *
                 * Exemplo:
                 * @code
                 * ...
                 *  cb->reset_elapsed_time();
                 * ...
                 * @endcode
                 */
        void reset_elapsed_time ( void );

                /**
                 * @Synopsis Invoca o método de registro deste Call_Box ( via objeto Sip vinculado ).
                 *
                 * Exemplo:
                 * @code
                 * ...
                 *  cb->registry();
                 * ...
                 * @endcode
                 */
        int registry ( void );

                /**
                 * @Synopsis Invoca o método de pedido de ligação.
                 *
                 * @return Um ponteiro para um objeto VZ_call quando o pedido foi aceito pelo server, NULL caso em que o pedido de invite foi 
                 * negado pelo server ou aconteceu timeout do invite.
                 *
                 * Exemplo:
                 * @code
                 * ...
                 *  VZ_call * call = cb->invite();
                 * ...
                 * @endcode
                 */
        VZ_call * invite ( void );

                /**
                 * @Synopsis Valor inicial para preenchimento deste campo nos pacotes trocados entre Header/Call_Box.
                 *
                 * @param msg_id Seta o valor recebido como parâmetro na variável do objeto.
                 *
                 * Exemplo:
                 * @code
                 * ...
                 *  cb->set_msg_id( 0x12 );
                 * ...
                 * @endcode
                 */
        void set_msg_id ( const uint8_t msg_id );

                /**
                 * @Synopsis Informa o numero atual de msg_id que será enviado na próxima mensagem desse Call_Box
                 *
                 * @return O valor atual de msg_id.
                 *
                 * Exemplo:
                 * @code
                 * ...
                 *  uint8_t msg_id = cb->get_msg_id();
                 * ...
                 * @endcode
                 */
        uint8_t get_msg_id ( void );

        
                /**
                 * @Synopsis Armazena o valor de Timeslice atualmente em uso pelo Call_Box.
                 *
                 * @param timeslice O valor que corresponde ao Timeslice disponivel para comunição do Call_Box
                 *
                 * Exemplo:
                 * @code
                 * ...
                 *  Timeslice * ts = new Timeslice();
                 *  cb->set_timeslice( ts->get_timeslice() );
                 * ...
                 * @endcode
                 */
        void set_timeslice ( const uint8_t timeslice );

                /**
                 * @Synopsis Informa o timeslice ocupado atualmente pelo Call_Box.
                 *
                 * @return O valor do timeslice em uso pelo Call_Box.
                 *
                 * Exemplo:
                 * @code
                 * ...
                 *  uint8_t ts = cb->get_timeslice();
                 * ...
                 * @endcode
                 */
        uint8_t get_timeslice ( void );
                
                /**
                 * @Synopsis Faz chamada ( via objeto Sip vinculado ) a função que irá escutar a porta SIP associada neste Call_Box.
                 *
                 * @return Um valor menor que zero se a execução falhar, igual a zero se a execução for bem sucedida e nenhum dado foi
                 * recebido, ou quando a execução foi bem sucedida e nenhuma mensagem que demanda tramamento foi recebida;  um numero 
                 * maior do que zero, caso tenha recebido um pacote do tipo "bye" do servidor; este numero corresponde ao ramal do Call_Box.
                 * 
                 * Exemplo:
                 * @code
                 * ...
                 *  int returned_value = cb->listen_SIP_server();
                 * ...
                 * @endcode
                 */
        int listen_SIP_server ( void );

                /**
                 * @Synopsis Invoca ( via objeto Sip ) o método de envio do pacote de "bye" para o servidor.
                 *
                 * Exemplo:
                 * @code
                 * ...
                 *  cb->send_bye();
                 * ...
                 * @endcode
                 */
        void send_bye ( void );

                /**
                 * @Synopsis Altera o valor do status do objeto Sip vinculado.
                 *
                 * @param status O novo valor de status que será associado ao objeto Sip vinculado ao Call_Box.
                 *
                 * Exemplo:
                 * @code
                 * ...
                 *  cb->set_sip_status( 0 );
                 * ...
                 * @endcode
                 */
        void set_sip_status ( const int status );
        
        /*------------------------------------------------------------------------------------------------*/
        bool get_invite_response ( void );
        
        bool time_to_retry ( void );
        
        bool is_rtp_timer_timeout ( void );
        
        bool get_overflow_flag ( void );
        
        
        uint8_t get_invite_try_number ( void );
        
        uint8_t msg_id_update ( void );
        
        uint8_t get_overflow_times ( void );
        
        uint16_t get_invite_retry_count ( void );
        
        
        int get_sip_status ( void );
        
        void set_sip_status ( const uint8_t sip_status );
        
        int get_status ( void );
        
        int get_sip_ext ( void );
        
        int get_sip_port ( void );
        
        int get_timer ( void );
        
        int get_rtp_port ( void );
        
        int print_yourself ( void );
        
        int get_sip_socket_fd ( void );
        
        int sip_udp_incomming_pkg ( void );
        
        int call_init ( const int timeslice );
        
        int call_end ( const bool send_bye_to_ast = true  );
        
        int call_confirmed ( void );
        
        int retry_send_invite_pkg_to_ast ( void );
        
        int sip_print_yourself ( void );
        
        
        void cb_set_status ( const uint8_t status );
        
        void set_invite_response_ok ( void );
        
        void set_invite_response_pending ( void );
        
        void set_bye_response_ok ( void );
        
        void reset_cb_status ( void );
        
        void invite_retry_count_reset ( void );
              
        void set_rtp_port ( const int new_rtp_port );
        
        void init_rtp_timer ( void );
        
        void reset_rtp_timer ( void );
        
        void call_config ( void );
        
        void update_time ( void );
        
        Sip * get_sip ( void );
        
        uint16_t get_invite_counter ( void );
            
        uint16_t update_invite_counter ( void );
        
        int update ( void );
        
        Call_Box * get_pair_cbx ( void );
        
        int set_pair_cbx ( Call_Box * new_cbx );
};

#endif