/**
 * @file rtp.h
 * @Synopsis Esta classe visa gerenciar o tratamento dos pacotes de RTP, ou seja, os pacotes de audio trocados entre o Call_Box, Header
 * servidor.
 * @author Jhonatan Casale
 * @version 1
 * @date 2014-11-05
 * \class RTP
 */

#ifndef __RTP_H__
#define __RTP_H__

#include "mbed.h"
#include "shared_variables.h"
#include "vz_protocol.h"
#include "debug.h"

class RTP_Header{
    private :
        uint32_t ss_id;
        uint16_t seq_number;
        uint32_t timestamp;
    public  :
                /**
                 * @Synopsis Cria um objeto da classe RTP_Header. Esta classe visa gerenciar a criação do cabeçalhoo RTP
                 *
                 * @param cb_ext Vincula o objeto recem criado com o ramal do Call_Box solicitante
                 *
                 * Exemplo:
                 * @code
                 * ...
                 *  int ext = 0x54;
                 *  RTP_Header * rtp_header = new RTP_Header( ext );
                 * ...
                 * @endcode
                 */
        RTP_Header( int cb_ext );

                /**
                 * @Synopsis Destroi o objeto RTP_Header.
                 *
                 * Exemplo:
                 * @code
                 * ...
                 *  delete( rtp_header );
                 * ...
                 * @endcode
                 */
        ~RTP_Header();

                /**
                 * @Synopsis Informa o tamanho ( em bytes ) do cabeçalho rtp.
                 *
                 * @return O numero ( em bytes ) ocupados pelo cabeçãolho do rtp.
                 *
                 * Exemplo:
                 * @code
                 * ...
                 *  size_t size = rtp_header->size();
                 * ...
                 * @endcode
                 */
        size_t size( void );

            /**
                 * @Synopsis Cria o cabeçalho rtp usado neste aplicação. Porém, fora de uso no momento.
                 *
                 * @param pkg Um ponteiro para a posição de memória onde será montado o pacote.
                 *
                 * @return Um ponteiro para o inicio do pacote montado.
                 *
                 * @code
                 *  uint8_t * build_header( uint8_t * buffer_header ){
                 *          // função não implementada.
                 *  }
                 * @endcode
                 */
        uint8_t * build_header( uint8_t * buffer_header );

                /**
                 * @Synopsis Cria o cabeçalho rtp usado neste aplicação.
                 *
                 * @param pkg Um ponteiro para a posição de memória onde será montado o pacote.
                 *
                 * @return Um ponteiro para o inicio do pacote montado.
                 *
                 * Exemplo:
                 * @code
                 * ...
                 *  char pkg[ 512 ]; 
                 *  rtp_header->build_header( pkg );
                 * ...
                 * @endcode
                 */
        char * build_header( char * pkg );
        int get_ss_id ( void ) { return this->ss_id; }
        int get_seq_number ( void ) { return this->seq_number; }
        int get_timestamp ( void ) { return this->timestamp; }
};

class RTP_Body{
    private :
    public  :
            /**
             * @Synopsis Constroi um objeto RTP_Body.
             *
             * Exemplo:
             * @code
             * ...
             *  RTP_Body * rtp_body = new RTP_Body();
             * ...
             * @endcode
             */
        RTP_Body();

                /**
                 * @Synopsis Destroi o objeto RTP_Body.
                 *
                 * Exemplo:
                 * @code
                 * ...
                 *  delete( rtp_body );
                 * ...
                 * @endcode
                 */
        ~RTP_Body();

                /**
                 * @Synopsis Informa o tamanho ( em bytes ) ocupados pelo corpo da mensagem.
                 *
                 * @return Retorna o numero de bytes ocupados pelo corpo da mensagem.
                 *
                 * Exemplo:
                 * @code
                 * ...
                 *  size_t size = rtp_body->size();
                 * ...
                 * @endcode
                 */
        size_t size( void );

                /**
                 * @Synopsis Cria o corpo do pacote. 
                 *
                 * @param dest Ponteiro para posição de memória onde será montado o pacote.
                 * @param buffer Ponteiro para a posição de memória com os dados originias.
                 *
                 * @return Um ponteiro para posição de memória contendo o corpo da mensagem.
                 *
                 * Exemplo:
                 * @code
                 * ...
                 *  // assumindo que buffer é onde foi armazenado os dados que precisam ser enviados.
                 *  rtp_body->build_body( pkg , buffer ); 
                 * ...
                 * @endcode
                 */
        char * build_body( char * dest, char * buffer );
};
class RTP{
    private :
        uint8_t buffer[ 512 ];
        char pkg[ 512 ];
        RTP_Header * rtp_header;
        RTP_Body * rtp_body;
    public  :
                /**
                 * @Synopsis Cria um objeto RTP.
                 *
                 * @param ext Vincula o objeto RTP ao ramal passado como parâmetro.
                 *
                 * Exemplo:
                 * @code
                 * ...
                 *  int ext = 1918;
                 *  RTP * rtp = new RTP( ext );
                 * ...
                 * @endcode
                 */
        RTP( int ext );

                /**
                 * @Synopsis Destroi o objeto RTP.
                 *
                 * Exemplo:
                 * @code
                 * ...
                 *  delete( rtp );
                 * ...
                 * @endcode
                 */
        ~RTP();

                /**
                 * @Synopsis Informa o tamanho ocupado pelo cabeçalho.
                 *
                 * @return O numero em bytes ocupados pelo cabeçalho.
                 *
                 * Exemplo:
                 * @code
                 * ...
                 *  size_t size = rtp->header_size();
                 * ...
                 * @endcode
                 */
        size_t header_size( void );

                /**
                 * @Synopsis Constroi efetivamente o pacote seguindo o protocolo rtp.
                 *
                 * @param buffer Posição de memória onde será montado o pacote.
                 *
                 * @return Um ponteiro para o inicio do pacote montado, pronto para envio.
                 *
                 * Exemplo:
                 * @code
                 * ...
                 *  // assumindo que buffer é onde foi armazenado os dados que precisam ser enviados.
                 *  return this->rtp->build_eth_package( buffer );
                 * ...
                 * @endcode
                 */
        char * build_eth_package( char * buffer );

        int print_yourself ( void );
};
#endif