/**
 * @file clock.h
 * @Synopsis Implementa as funcionalidades de requisição e atualização do clock usado pela header e pelos CBx
 * @author Jhonatan Casale
 * @version 1
 * @date 2014-05-04
 */

#ifndef __CLOCK_H__
#define __CLOCK_H__

#include <stdint.h>
#include "EthernetInterface.h"
#include "mbed.h"
#include "debug.h"
%: include "shared_variables.h"
%: include "config_manager.h"

const uint16_t EXTERNAL_TIME_REQUEST_WAIT_SECONDS = 300;
///< Tempo que a Header espera para sincronizar o timer com o servidor.

const uint16_t CLOCK_HEADER_PORT = 8975;
///< Porta usada pela Header para essa comunicação.

const int TIME_MSG_SIZE = 64;
///< Tamanho ( em bytes ) utilizado nas sincronizações.

const int CLOCK_SERVER_PORT = 123;
///< Porta destino no servidor, para onde a Header irá encaminhar os pacotes de pedido de clock.

extern time_t current_time;
///< Representa o tempo em Unixtime

extern UDPSocket clock_sock;
///< Socket usado na comunicação com o servidor, representa o lado cliente ( Header )

extern Endpoint clock_server;
///< Socket usado na comunicação com o servidor, representa o lado servidor ( Server ( * ) )

extern Timer server_clock_timer;
///< Timer usado para gerenciar os pedidos de clock para o Servidor.

//*-----------------------------------------------------------------------------------------------------------------------------*//

/**
 * @Synopsis Monta e envia o pacote de pedido de clock para o servidor, em caso de falha no envio, tente reconectar o socket
 *
 * @return O numero de bytes efetivamente colocados na fila UDP de envio, -1 em caso de falha.
 *
 * Exemplo:
 * @code
 * ...
 *   int request_clock_to_server_ret = request_clock_to_server ();
 * ...
 * @endcode
 */
int request_clock_to_server( void );

/**
 * @Synopsis Controla o pedido de clock para o servidor, a cada EXTERNAL_TIME_REQUEST_WAIT_SECONDS segundos
 *
 * @return 0x30 nos casos em que não enviou o pacote, ou retorna o valor devolvido por request_clock_to_server ()
 *
 * Exemplo:
 * @code
 * ...
 *  check_clock ();
 * ...
 * @endcode
 */
int check_clock( void );

/**
 * @Synopsis Atualiza a referência de clock caso seja necessario ( se | tempo_local - tempo_recebido | > 2
 *
 * @return A diferença entre as referências de tempo ( local - recebido ).
 *
 * Exemplo:
 * @code
 * ...
 *  update_clock ();
 * ...
 * @endcode
 */
int update_clock ( void );

/**
 * @Synopsis Usada para iniciar o server_clock_timer, e demais configurações iniciais.
 *
 * @return Zero ( 0 ) em caso de sucesso, numeros diferentes de zero em caso de ocorrencia de erros.
 *
 * Exemplo:
 * @code
 * ...
 *  init_clock ();
 * ...
 * @endcode
 */
int init_clock ( void );

/**
 * @Synopsis Tenta reconectar o socket em caso na utilização
 *
 * @return O valor retornado pelo método sock.bind ();
 *
 * Exemplo:
 * @code
 * ...
 *  clock_sock_reconnect ();
 * ...
 * @endcode
 */
int clock_sock_reconnect ( void );

/**
 * @Synopsis Exibe uma mensagem no formato "current_time : [Unixtime]     clock() : [Vz_time]"
 *
 * @return O tamanho ( em bytes ) da mensagem que sera escrita na tela
 *
 * Exemplo:
 * @code
 * ...
 *  show_clock ();
 * ...
 * @endcode
 */
int show_clock ( void );

#endif