#ifndef __CALL_MANAGER_H__
#define __CALL_MANAGER_H__

#include "timeslice.h"
#include "vector.h"
#include "call.h"
#include "call_box.h"
#include "parallelcpld.h"
#include "call_box_manager.h"
#include "shared_variables.h"

extern Timer invite_pkg_retry_timer;
extern Timer bye_pkg_retry_timer;

/**
 * @Synopsis Busca por uma determinada ligação vz_call no vetor de v_calusando como chave de busca o ramal.
 *
 * @param v_call Um vetor contendo todas as vz_calls em andamento em um dado momento.
 * @param ext O ramal pelo qual se irá buscar a vz_call.
 *
 * @return NULL, caso essa chamada não tenha sido encontrada; um ponteiro para esta chamada, caso a mesma tenha sido encontrada.
 *
 * Exemplo:
 * @code
 * ...
 *  Vector * v_call = new Vector();
 *      //assumindo que esse vetor de VZ_call já foi populado.
 *  int ext = 5218;
 *  VZ_call * call = find_Call ( v_call, ext );
 * ...
 * @endcode
 */
VZ_call * find_Call ( Vector * v_call, const int ext );

/**
 * @Synopsis Responsável por remover calls por timeout.
 *
 * @param v_call Um vetor contendo todas as vz_calls em andamento em um dado momento.
 * @param v_cb Um vetor contendo todos os Call_box conhecidos pela Header em um determinado momento.
 * @param data Usado para a criação do pacote de "ping" que a Header irá enviar para o CBx.
 * @param write_buffer Local de memória onde efetivamente irá ser montado o pacote para envio ao CBx. 
 * @param ts O timeslice que será possivelmente devolvido.
 *
 * Exemplo:
 * @code
 * ...
 *  Vector * v_call = new Vector();
 *  Vector * v_cb = new Vector();
 *      // assumindo que os vetores já foram populados.
 *  uint8_t buffer[ 300 ];
 *      uint8_t write_buffer[ 300 ];
 *      Timeslice * ts = new Timeslice();
 *      ...
 *      call_manager( v_call, v_cb, buffer, write_buffer, ts ); 
 * ...
 * @endcode
 */
void call_manager( Vector * v_call, Vector * v_cb, Timeslice * ts );

/* verificando andamento de pedidos de ligação */
void invite_handler( Vector * v_call, Vector * v_cb, Timeslice * ts, Call_Box * cb );

#endif