/**
 * @file vz_protocol.h
 * @Synopsis Implementa as principais funcionalidades do protocolo de comunicação entre os CBx -> Header e Header -> CBx.
 * @author Jhonatan Casale
 * @version 1
 * @date 2014-11-05
 */

#ifndef __VZ_PROTOCOL_H__
#define __VZ_PROTOCOL_H__

#include "EthernetInterface.h"
#include "mbed.h"
#include <string.h>
#include <stdlib.h>
#include <stdint.h>
#include "debug.h"
#include "bits.h"
#include "clock.h"
#include "utils.h"
%: include "shared_variables.h"
%: include "config_manager.h"

extern int begin;
///< Registra o numero do menor ramal conhecido pela header até o momento

extern int end;
///< Registra o numero do maior ramal conhecido pela header até o momento

const uint8_t VZ_HEADER_OFFSET = 7;
///< Indica o inicio dos dados recebidos efetivamente no pacote VZ.

const uint8_t CLOCK_SYNC_SIZE = 14;
///< Indica o numero de bytes ocupados pelo relogio no pacote transmitido.

const uint8_t SEQ_NUM_SIZE = 1;
///< Indica o numero de bytes ocupados para uso de sequence number.

const uint8_t SEQ_NUM_PLACE = 7;
///< Indica o numero de bytes ocupados para uso de sequence number.

const uint8_t CB_AUDIO_DATA_SIZE = 240; 
///< Indica o numero de pacotes enviados pelo CBx referente a dados de audio.

const uint8_t TIMESLICE_PLACE = 22; 
///< Indica o local ( em relação ao começo do pacote ) onde se encontra o timeslice.

const uint8_t TYPE_PLACE = 6; 
///< Indica o local ( em relação ao começo do pacote ) onde se encontra o timeslice.

const uint16_t BASE_PORT = 5000;
///< Estabelece o menor ramal aceito para tratamento

const uint8_t u8_MAX_CB_IN_A_BRANCH = 52;
///< Estabelece o nro máximo de CBx em um mesmo ramo

const uint8_t BOOT = 0x00; 
///< Indica o tipo boot, enviado pelo CBx, assim que o mesmo liga ou sofre reboot.

const uint8_t REGISTRY = 0x02; 
///< Tipo de registro, enviado pelo CBx, quando o mesmo quer se registrar, enviado pela Header para verificar se determinado CBx ainda esta ativo.

const uint8_t REGISTRY_ACK = 0x02 bitor BIT7;
///< "ack de resposta" do tiop REGISTRY

const uint8_t INVITE = 0x04;
///< Representa o tipo de pedido de invite, enviado pelo CBx sempre quando o mesmo quer iniciar uma ligação com o server.

const uint8_t INVITE_ACK = 0x04 bitor BIT7;
///< "ack de resposta" do tiop INVITE

const uint8_t AUDIO = 0x08;
///< Pacotes do tipo audio são trocados entre Header e CBx durante a ligação, representam os dados RTP.

const uint8_t TELEMETRY = 0x10;
///< Define o tipo de pacote de telemetria enviado pelo CBx.

const uint8_t BOOTLOADER_CBX = 0x03;
///< Define o tipo de pacote para a gravação do CBx

const uint8_t CB_BYE = 0x20;
///< Representa o tipo de pacote que o CBx envia para a Header solicitando o final da ligação.

const uint8_t CB_BYE_ACK = 0x20 bitor BIT7;
///< "ack de resposta" do tiop CB_BYE

const uint8_t PROMPT = 0x01;
///< Identifica o tipo de pacote responsavel por mandar comandos executáveis no Cbx.

const uint8_t FLOOD = 0x40;
///< Representa os pacotes de flood, úteis para validação de comunicação Header-CBx.

const uint8_t FW = 0x50;
///< Tipo para redirecionamento semn tratamento para uma determinada porta UDP pré-configurada.

const uint8_t FW1 = 0x51;
///< Tipo para redirecionamento semn tratamento para uma determinada porta UDP pré-configurada.

const uint8_t FW2 = 0x52;
///< Tipo para redirecionamento semn tratamento para uma determinada porta UDP pré-configurada.

const uint8_t FW3 = 0x53;
///< Tipo para redirecionamento semn tratamento para uma determinada porta UDP pré-configurada.

const uint8_t FW4 = 0x54;
///< Tipo para redirecionamento semn tratamento para uma determinada porta UDP pré-configurada.

const uint8_t FW5 = 0x55;
///< Tipo para redirecionamento semn tratamento para uma determinada porta UDP pré-configurada.

const uint8_t FW6 = 0x56;
///< Tipo para redirecionamento semn tratamento para uma determinada porta UDP pré-configurada.

const uint8_t CB_STATS = 0x07;
///< Tipo de comunicação de estatisticas de rede, enviadas pela CBx

const uint8_t CB_STATS_ACK = 0x07 bitor BIT7;
///< "ack de resposta" do tiop CB_STATS_ACK

const uint16_t BROADCAST_EXT = 0xf0f3;
///< Ramal de broadcast, usado na comunicação Header -> CBx

const uint8_t DO_NOTHING = 0x7f;
///< Representa o tipo de idle, importante para algumas comunicações.

const uint16_t RX_CB_IDLE = 300;
///< Tempo maximo ( countdown em segundos ) que a Header espera por algum pacote no lado fibra, deixa de tickar o wdt.

const uint16_t ETH_CONNECT_TIMEOUT = 330;
///< Tempo maximo ( countdown em segundos ) que a Header espera tentando conectar na interface ETH, deixa de tickar o wdt quando chega em zero.

const uint16_t RTP_MSG_SIZE = 160;
///< Tamanho em bytes ocupados pelos pacotes de audio no sentido * -> Header -> CBx

extern uint32_t pkg_zero; 
///< Contador de pacotes contendo somente zeros

extern uint32_t pkg_ckserr; 
///< Contador de pacotes em que o checksum resultou divergente do calculado/recebido

extern uint32_t pkg_cksok; 
///< Contador de pacotes em que o checksum resultou o mesmo calculado e o recebido

/**
 * @Synopsis Calcula o checksum do pacote.
 *
 * @param buffer Um ponteiro para a região de memória onde os dados de interesse estão localizados.
 * @param length O numero de elementos que serão considerados no cálculo do checksum
 *
 * @return 0 - caso em que o vetor apontado por buffer estiver setado em NULL. O checksum propriamente
 * calculado caso contrário.
 *
 * Exemplo:
 * @code
 * ...
 *  // onde cb_buffer contém os dados vindos do Call_Box, Obs. as posições 4 e 5 contém o MSB e LSB do CC calculado pelo Call_Box.
 *  uint16_t cc = ( uint16_t )cb_buffer[ 4 ] << 8 or cb_buffer[ 5 ];
 *  if( cc != __checksum__( cb_buffer, __CB_BUFFER_SIZE__ ) ){
 *      //faça alguma coisa ...
 *  }
 * ...
 * @endcode
 */
uint16_t vz_checksum ( uint8_t * buffer, size_t length );

/* incorporar a parte de cc */
/**
 * @Synopsis Esta é a função responsável por, dado um pacote recebido dos CBx, quebra-lo em ramal( ext ), porta, type e dados.
 *
 * @param ext Um ponteiro para onde sera setado o ramal do CBx que enviou este pacote.
 * @param port Um ponteiro para onde sera setado a porta do CBx que enviou este pacote.
 * @param type Um ponteiro para onde sera setado o type de mensagem enviada pelo CBx.
 * @param cb_buffer O pacote que se deseja decodificar.
 *
 * @return NULL, caso em que o checksum( cc ) calculado não bater com o cc recebido no pacote, retorna NULL também em pacotes do
 * tipo flood, ou no caso em que o pacote recebido para processamento apontar para NULL; em todos esses casos os valores de ext,
 * port e type devem ser desconsiderados para manter a integridade do processamento; retorna um ponteiro para o inicio dos dados
 * enviados pelo CBx e seta ramal, porta e type nos casos em que o pacote recebido for válidado pelo protocolo.
 *
 * Exemplo:
 * @code
 * ...
 *  int ext, port, type;
 *  uint8_t * data, buffer[ __CB_BUFFER_SIZE__ ];
 *      //assumindo que os dados vindos do Call_Box estão armazenados em buffer;
 *  data = __parse_vz_pkg__( &ext, &port, &type, buffer );
 * ...
 * @endcode
 */

/**
 * \note Formato do pacote VZ :
 *
 * | E | E | P | P | C | C | T | [ Seq_num | Audio ] | 14[ Clock | Audio ] | [ TS | Audio ] |...|
 *
 * E = Ext = Ramal 
 *
 * P = Port = Porta
 *
 * C = Checksum 
 *
 * T = Type = Tipo 
 *
 * Seq_num = Sequence Number = Numero de sequencia 
 *
 * Clock = 14 bytes to sync 
 *
 * ...= demais __CB_BUFFER_SIZE__ - __VZ_HEADER_OFFSET__  bytes 
 *
 */
uint8_t * parse_vz_pkg ( int * ext, int * port, volatile uint8_t * type, uint8_t * cb_buffer );

/**
 * @Synopsis Função responsavel por montar o pacote para envio para o CBx seguindo o formato do protocolo VZ. 
 *
 * @param ext O ramal do CBx destino deste pacote.
 * @param port A porta do CBx destino deste pacote.
 * @param type O tipo do pacote que será enviado para o CBx.
 * @param cb_buffer Os dados que serão colocados nesse pacote.
 * @param seq_num O numero do sequência deste pacote.
 * @param length O tamanho ( em bytes ) dos dados que serão enviados.
 * @param pkg Um ponteiro que recebera o pacote montado e pronto para envio.
 *
 * @return O pacote montado e pronto para envio.
 *
 * Exemplo:
 * @code
 * ...
 *  int ext = 1011;
 *  port = 1011;
 *  type = __REGISTRY__;
 *  send2callboxes( __build_cb_package__( ext, port, type, 
 *              ( char * )data, cb->get_msg_id(), __CB_BUFFER_SIZE__ - __VZ_HEADER_OFFSET__, write_buffer ) ); 
 * ...
 * @endcode
 */
uint8_t * build_cb_package ( const int ext, const int port, const uint8_t type, const char * cb_buffer, const uint8_t seq_num, const int length,  uint8_t * pkg );

/**
 * @Synopsis Obtém uma referência de clock de um servidor.
 *
 * @param buffer Imprime a referência do relógio nesta posição de memória.
 *
 * \note Por definição de projeto, o relógio ocupa 14 bytes, seguindo o formato
 *
 * | ano | ano | ano | ano | mes | mes | dia | dia | hora | hora | minuto | minuto | segundo | segundo |
 *
 * Exemplo:
 *
 * 20141105101235
 */

int init_ranges ( void );

/**
 * @Synopsis Converte o current_time do sistema para string.
 *
 * @param buffer Buffer de escrita onde sera preenchido com o current_time no formato VZ
 *
 * Exemplo:
 * @code
 * ...
 *  char buffer [ 16 ];
 *  print_clock ( buffer );
 * ...
 * @endcode
 */
void print_clock ( uint8_t * buffer );

#endif