/**
 * @file call.h
 * @Synopsis Implementa as funcionalidades de uma ligação, incluindo troca de dados de audio em ambos os lados. 
 * @author Jhonatan Casale
 * @version 1
 * @date 2014-11-05
 * \class VZ_call
 */
 
#ifndef __CALL_H__
#define __CALL_H__

#include "shared_variables.h"
#include "object.h"
#include "rtp.h"
#include "debug.h"
#include "rtpbuf.h"
%: include "config_manager.h"

const uint8_t __CALL_MAX_IDLE_TIME__ = 5;
///< Define o tempo máximo sem receber dados em alguma chamada.

const uint8_t __MAX_CALL_TIME__ = 180;
///< Define o tempo máximo ( em segundos ) de uma ligação.

const uint8_t CBX_PKG_IDLE_MAX_TIME = 10;

class VZ_call : public Object{
    private :
        Timer t;
        Timer finish;
        Timer cbx_pkg_idle_timer;
        int cb_ext;
        int cb_port;
        uint8_t cb2server[ 320 ];
        char server2cb[ 320 ];
        uint8_t buffer[ 320 ];
        UDPSocket rtp_sock;
        int rtp_server_ext;
        int rtp_server_port;
        Endpoint rtp_server;
        RTP * rtp;
        rtpbuf_t rtpbuf;
    public :
    /**
     * @Synopsis Cria um objeto da Classe VZ_call.
     *
     * @param cb_ext O ramal do CBx que solicitou a ligação.
     * @param cb_port A porta RTP do CBx que solicitou a ligação.
     * @param rtp_server_ext O ramal do servidor.
     * @param rtp_server_port A porta RTP do servidor.
     *
     * Exemplo:
     * @code
     * ...
     *  VZ_call * call = new VZ_call( 5010, 11841, 913, 12902 );
     * ...
     * @endcode
     */
        VZ_call( int cb_ext, int cb_port, int rtp_server_ext, int rtp_server_port );

    /**
     * @Synopsis Destroi o objeto da classe.
     *
     * Exemplo:
     * @code
     * ...
     *  delete( call );
     * ...
     * @endcode
     */
        ~VZ_call();

    /**
     * @Synopsis Obtém o ramal do CBx.
     *
     * @return Um inteiro que representa o ramal do CBx que solicitou a ligação.
     *
     * Exemplo:
     * @code
     * ...
     *  int cb_ext = call->get_cb_ext();
     * ...
     * @endcode
     */
        int get_cb_ext();

    /**
     * @Synopsis Obtém a porta RTP do CBx que solicitou a ligação.
     *
     * @return Um inteiro que representa a porta do CBx. 
     *
     * Exemplo:
     * @code
     * ...
     *  int cb_port = call->get_cb_ext();
     * ...
     * @endcode
     */
        int get_cb_port();

    /**
     * @Synopsis Informa o ramal do servidor RTP.
     *
     * @return O numero do ramal do servidor RTP.
     *
     * Exemplo:
     * @code
     * ...
     *  int rtp_server_ext = call->get_rtp_server_ext();
     * ...
     * @endcode
     */
        int get_rtp_server_ext();

    /**
     * @Synopsis Informa o numero da porta do servidor RTP.
     *
     * @return O numero da porta usada na comunicação com o servidor RTP.
     * 
     * Exemplo:
     * @code
     * ...
     *  int rtp_server_port = call->get_rtp_server_port();
     * ...
     * @endcode
     */
        int get_rtp_server_port();

    /**
     * @Synopsis Coleta dados de audio vindos do servidor
     *
     * @param length Um ponteiro para posição de memória que recebera quantos bytes foram lidos no pacote recebido.
     *
     * @return Um ponteiro para o pacote disponivel para processamento.
     *
     * Exemplo:
     * @code
     * ...
     *  int length = 0;                                                                 
     *  char * eth_msg = call->get_eth_message( &length ); 
     * ...
     * @endcode
     */
        char * get_eth_message( int * length );
            
    /**
     * @Synopsis Responsavel por efetivamente montar o pacote que será enviado via rede.
     *
     * @param buffer Um ponteiro para o inicio dos dados recebidos do CBx que será encapsulado.
     *
     * @return Um ponteiro para o inicio do pacote pronto para o envio.
     *
     * Exemplo:
     * @code
     * ...
     *  char str_to_server[ 1024 ];
     *  // assumindo que em data_from_cb consta armazenados os dados vindos dos Call_Box
     *  strcpy( str_to_server, build_eth_package( data_from_cb ) );
     * ...
     * @endcode
     */
        char * build_eth_package( uint8_t * );

    /**
     * @Synopsis Responsavel por enviar efetivamente os dados via eth
     *
     * @param buffer Um ponteiro para o inicio dos dados que serão enviados.
     *
     * @return O numero de bytes efetivamente enviados.
     *
     * Exemplo:
     * @code
     * ...
     *  char * pkg = call->build_eth_package( data_from_cb );
     *  //assumindo que os dados vindos do Call_Box estão armazenados em data_from_cb
     *  int sent = call->send_message( pkg ); 
     * ...
     * @endcode
     */
        int send_message( char * );

    /**
     * @Synopsis Verifica se o tempo de ligação é menor que o tempo máximo permitido.
     *
     * @retval True - Caso o tempo da ligação tenha passado do limite definido em __MAX_CALL_TIME__.
     * @retval False - caso em que o tempo da ligação atual for menor que o máximo permitido..
     *
     * Exemplo:
     * @code
     * ...
     *  printf( "Call já deveria ter acabado ? %s \n\r", ( call->is_timeout() ) ? "Sim" : "Não" );
     * ...
     * @endcode
     */
        bool is_timetofinish ( void );
        
        
        bool cbx_pkg_is_idle ( void );
        
        uint16_t get_elapsed_time ( void );
        
        void cbx_pkg_idle_timer_reset ( void );
        void reconnect ( void );
        void * check_rtp ( void );
        void init_cbx_pkg_idle_timer ( void );
        
        int rtp_print_yourself ( void );
        int print_yourself ( void );
};
#endif