/**
 * @author Hugues Angelis
 *
 * @section LICENSE
 *
 * Copyright (c) 2010 ARM Limited
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 *
 * @section DESCRIPTION
 *
 * motion control.
 *
 */

#ifndef PID_H
#define PID_H

#define PI  3.1415926535898

/**
 * Includes : Mbed Library + Nucleo_Encoder_16_bits library
 */
#include "mbed.h"
#include "Nucleo_Encoder_16_bits.h"

/**
 * PID Motion control system (speed control).
 *  refere to wikipedia for more explainations : https://en.wikipedia.org/wiki/PID_controller
 *
 * @note This library require a tuning (mostly base on try/modify) and will not give a correct performance until you have tuned all 3 parameters Kp, Ki and Kd for each wheels of the robot
 *
 * @note You must use a QE (quadrature Encoder) connected to a 16 bits timer to get proper motion control. Pins A & B of the QE must be connected respectively to pin 1 and 2 of the timer. Typicaly on Nucleo F446RE TIM3 and TIM4 are perfect to do this job. In this case simply use TIM3 or TIM4 as timer parameter.
 *
 * @note You must also specify the number of pulses generated by the QE for a 1mm displacement of the wheel. This is the scale parameter
 *
 * @note Outputed PWM value evolve from 1 (full PWM fortward) to -1 (full PWM backward). This value can also be found in the global variable : _PwmValue. The PWM value is based on a 1.3 m/s maximum speed.
 *
 * @note As this motion control system is implemented in a microcontroler it is important to understand that there is a loop time for the motion control system and that this loop time MUST be constant. Kp, Ki and Kd are dependent of the loop time. Changing loop time means changing all the corrector's coefficients. Library use a Ticker to control loop time. The looptime can be set by software as long as it remain constant during the whole use of PID controler. Loop time is set to 1ms by default.
 *
 * @note The PID is initialized with Ki = 0, Kd = 0 and Kp = 1
 *       -  Increasing Kp will shorten your response time but will also create instability (at beggining overshoots, then instability).
 *       -  Adding a bit of Ki will allow your system to bring to 0 the static error (ie : will make the error, between the set point and your mesurment, tend to 0) but might create instability and increase setting time.
 *       -  Adding a bit of Kd will stabilize the response (with almost no bad effect, as long as Kd remains small).
 * @note More info can be found here : https://en.wikipedia.org/wiki/PID_controller
 */
class PID
{

public :

    /**
     * Constructor (standard) with full bridge control
     *
     * @param LTIM (TIM_TypeDef*) : the Mbed 16 bits timer used to connect the A&B output of left QE
     * @param RTIM (TIM_TypeDef*) : the Mbed 16 bits timer used to connect the A&B output of right QE
     * @param LPWM (PinName) : the Mbed pin used to send PWM for the left wheel's full bridge
     * @param RPWM (PinName) : the Mbed pin used to send PWM for the right wheel's full bridge
     * @param LDIRA (PinName) : the Mbed pin used to send direction to the Left wheel IN1 full bridge pin
     * @param LDIRB (PinName) : the Mbed pin used to send direction to the Left wheel IN2 full bridge pin
     * @param RDIRA (PinName) : the Mbed pin used to send direction to the Right wheel IN1 full bridge pin
     * @param RDIRB (PinName) : the Mbed pin used to send direction to the Right wheel IN2 full bridge pin
     */
    PID(TIM_TypeDef *LTIM, TIM_TypeDef *RTIM, PinName LPWM, PinName RPWM, PinName LDIRA, PinName LDIRB, PinName RDIRA, PinName RDIRB);

    /**
     * Set the Kp value
     *
     * @param  Kp (float) : value of Kp
     * @return The value of Kp (float)
     * @note  Can also be accessed using the global variable _Kp
     */
    float setProportionnalValue (float KpValue);

    /**
     * Set the Ki value
     *
     * @param  Ki (float) : value of Ki
     * @return The value of Ki (float)
     * @note  Can also be accessed using the global variable _Ki
     */
    float setintegralValue (float KiValue);

    /**
     * Set the Kd value
     *
     * @param  Kd (float) : value of Kd
     * @return The value of Kd (float)
     * @note  Can also be accessed using the global variable _Kd
     */
    float setDerivativeValue (float KdValue);

    /**
     * Set the Set point value of the speed for integrated full bridge
     *
     * @param left  (double) : Set point value for left  wheel speed (in mm/s)
     * @param right (double) : Set point value for right wheel speed (in mm/s)
     */
    void setSpeed (double left, double right);

    /**
     * Get position of the robot (in mm for X and Y and radian for Theta)
     *
     * @param x (double - passed by reference) : actual position of the center of the robot, in mm, along X axis (front of the robot at startup)
     * @param y (double - passed by reference) : actual position of the center of the robot, in mm, along Y axis (left of the robot at startup)
     * @param theta (double - passed by reference) : radian angle between the normal to the line passing through the 2 wheels and the angle of the robot at startup
     * @note the position is updated each time a motion computation take place (ie : each milliseconds)
     */
    void getPosition (double *x, double *y, double *theta);

    /**
     * Reset position of the robot (in mm for X and Y and radian for Theta)
     *
     * @note the positionis equal to 0 on all 3 axis (cannot be undone)
     */
    void resetPosition (void);

    /**
     * Get speed of the two wheels of the robot
     *
     * @param vG (double - passed by reference) : actual speed in mm/s of the left wheel
     * @param vD (double - passed by reference) : actual speed in mm/s of the right wheel
     */
    void getSpeed (double *vG, double *vD);
    
    
    double _traveled_distance_mm;
    double getTraveledDistance_mm() {return _traveled_distance_mm;}
    
    
    
    /**
     * Global Variable of corrective values
     * @note  works well with Kp = 2.0, Ki = 0.4 and Kd = 1
     */
    double                  _Kp, _Ki, _Kd;
    /**
     * Global Variable of speed
     */
    double                  _SpeedG, _SpeedD;
    /**
     * Global Variable of measured speed
     */
    double                  _measSpeedG, _measSpeedD;
    /**
     * Global Variable of measured displacement
     */
    double                  _measDistG, _measDistD;
    /**
     * Global Variable to indicate that required wheel speed is unreachable (set if speed is unreachable)
     * @note    Must be cleared by user
     */
    int                     _WheelStuckG, _WheelStuckD;
    /**
     * Global Variable of wheel PWM value
     */
    double                  _PwmValueG, _PwmValueD;
    /**
     * Global Variable of gravity center of robot position (odometric, error increase with time)
     */
    double                  _X, _Y, _THETA;

    /**
     * Global Variable to indicate that required speed is unreachable (=1 if speed is unreachable)
     * @note    Must be cleared by user
     */
    int                     RobotIsStuck;

protected :

    Nucleo_Encoder_16_bits  _Lencoder, _Rencoder;
    PwmOut                  _Lpwm, _Rpwm;
    DigitalOut              _LdirA, _LdirB, _RdirA, _RdirB;

private :

    Ticker      _tick;
    void    controlLoop();

};
#endif //PID_H