/**
 * @author Hugues Angelis
 *
 * @section LICENSE
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 *
 * @section DESCRIPTION
 *
 * CMUCAM 5 - Pixy
 *
 * Datasheet, FAQ and PC drivers :
 *
 * http://www.cmucam.org/projects/cmucam5
 */

#ifndef PIXY_H
#define PIXY_H

/**
 * Include : Mbed Library
 */
#include "mbed.h"

/**
 * Defines
 */
#define CC_BLOCSIZE         14
#define N_BLOCSIZE          12

#define N_BLOCCODE          0xAA55
#define CC_BLOCCODE         0xAA56

    /**
     *  \struct Byte    ->      Short hand for unsigned char
     *  \struct Word    ->      Short hand for unsigned short
     *  \struct lWord   ->      Short hand for unsigned long
     */
    typedef unsigned char   Byte;
    typedef unsigned short  Word;
    typedef unsigned long   lWord;
    /**
     *  \struct T_pixyCCBloc
     *  \brief Structured type to match pixy Color Code bloc organisation
     *  \param Checksum (Word) : modulus sum of all words of the message
     *  \param signature (Word) : identity (ie : numbers associated with colors) of the detected color code bloc
     *  \param x (Word) : X coordinate (in pixel) of the center of the color code bloc
     *  \param y (Word) : Y coordinate (in pixel) of the center of the color code bloc
     *  \param width (Word) : Width (in pixel) of the color code bloc
     *  \param height (Word) : Height (in pixel) of the color code bloc
     *  \param angle (Word) : Angle (in degree, referenced to horizontal) of the color code bloc  
     */
    typedef struct {
        Word    checksum;
        Word    signature;
        Word    x;
        Word    y;
        Word    width;
        Word    height;
        Word    angle;
    } T_pixyCCBloc;
    
    /**
     *  \struct T_pixyNMBloc
     *  \brief Structured type to match pixy normal bloc organisation
     *  \param Checksum (Word) : modulus sum of all words of the message
     *  \param signature (Word) : identity (ie : number associated with the color) of the detected bloc
     *  \param x (Word) : X coordinate (in pixel) of the center of the color code bloc
     *  \param y (Word) : Y coordinate (in pixel) of the center of the color code bloc
     *  \param width (Word) : Width (in pixel) of the color code bloc
     *  \param height (Word) : Height (in pixel) of the color code bloc
     */
    typedef struct {
        Word    checksum;
        Word    signature;
        Word    x;
        Word    y;
        Word    width;
        Word    height;
    } T_pixyNMBloc;
    
    /**
     *  \union T_pixyCCData
     *  \brief Overlaped type to access a structured pixy Color Code bloc type with both byte and structured 
     *  \param CCBloc (T_pixyCCBloc) : Color Code bloc structured element
     *  \param tab[14] (Byte) : Byte access to the Color Code bloc 
     */
    typedef union {
        Byte            tab[14];
        T_pixyCCBloc    CCbloc;
    } T_pixyCCData;
    
    /**
     *  \union T_pixyNMData
     *  \brief Overlaped type to access a structured pixy Normal bloc type with both byte and structured
     *  \param NMBloc (T_pixyNMBloc) : Normal bloc structured element
     *  \param tab[12] (Byte) : Byte access to the normal bloc 
     */
    typedef union {
        Byte            tab[12];
        T_pixyNMBloc    NMbloc;
    } T_pixyNMData;

    typedef enum {none, begin, normal, colorCode, doubleZero, waitForStart} T_pixyState;
    
    typedef union {
        lWord   motlong;
        Word    mot[2];
        Byte    octet[4];
    } T_msgBuffer;
    
    typedef union {
        Word    mot;
        Byte    octet[2];
    } T_wordBuffer;
    
/**
 * Pixy : CMU CAM 5 - Smart camera
 * More informations at http://cmucam.org/projects/cmucam5
 */
class PIXY {

protected :

    Serial*  _Pixy;

public :

/** Direct access to the CC FIFO
 * @note A FIFO is a circular buffer where data are stored in the order in witch they arrive and can be read in the same order. A FIFO need 2 variables (call read and write pointers) to point both read and write registers.
 * @note The library manage both read and write pointers with FIFO functions, but users can access the FIFO with their own code, users must then manage their own read pointers while write pointers (that are strictly private) can only be managed by the library functions.   
 *
 *  \var Pixy_CCFIFO[20]
 *  \brief is a FIFO where the Color Code objects are stored. FIFO can store up to 20 objects
 */

/** Direct access to the NM FIFO
 * @note A FIFO is a circular buffer where data are stored in the order in witch they arrive and can be read in the same order. A FIFO need 2 variables (call read and write pointers) to point both read and write registers.
 * @note The library manage both read and write pointers with FIFO functions, but users can access the FIFO with their own code, users must then manage their own read pointers while write pointers (that are strictly private) can only be managed by the library functions.   
 *  \var Pixy_NMFIFO[20]
 *  \brief is a FIFO where the Normal objects are stored. FIFO can store up to 20 objects
 */
 
/** 
 *  \var Pixy_CCObjet
 *  \brief is the number of CC object read by the library ISR from the CAM during the last frame
 *  @note not Thread Safe
 *  \var Pixy_NMObjet
 *  \brief is the number of normal object read by the library ISR from the CAM during the last frame
 *  @note not Thread Safe
 */
    T_pixyCCData        Pixy_CCFIFO[20];    // FIFO des objets ColorCode
    T_pixyNMData        Pixy_NMFIFO[20];    // FIFO des objets Normaux
    Byte                Pixy_CCObjet;       // Nombre d'objets de type Color Code
    Byte                Pixy_NMObjet;       // Nombre d'objets Normaux
    Byte                Pixy_FirstCCObjet;  // Position dans la FIFO du premier objet de la trame suivante
    Byte                Pixy_FirstNMObjet;  // Position dans la FIFO du premier objet de la trame suivante
    
    
    /** A new frame as been read
     *
     * @note Semaphore are global variable used to communicate from ISR with main routine. All semaphores must be cleared by user.
     *  \var FlagPixy
     *  \brief is automaticly set after each received image frame
     */

    /** Data have been lost
     *
     * @note Semaphore are global variable used to communicate from ISR with main routine. All semaphores must be cleared by user.
     *  \var FlagPixyOverflow
     *  \brief is automaticly set if any FIFO oveflows (more than 20 obj of a kind). 
     */
    int                 FlagPixy, FlagPixyOverflow;
    int                 Pixy_check;

    /**
     * Constructor of pixy object.
     *
     * @param tx : the Mbed pin used as TX
     * @param rx : the Mbed pin used as RX
     * @param debit : the bitrate of the serial (max value is 230400 b/s)
     */
    PIXY(PinName tx, PinName rx, int debit =230400);

    /**
     * Return the number of objects (normal and ColorCode) that have been
     * received from the PIXY and stored in the FIFO during the last frame
     *
     * @param nbNM (int - passed by reference) : number of normal object detected
     * @param nbCC (int - passed by reference) : number of color code object detected
     * @return   0 if sucessfull,
     *          -1 if no PIXY is talking,
     *          -2 sucessfull but object(s) have been lost from previous frames (missed frame) or from actual (too many objects) 
     */
    int detectedObject (int* nbNM, int* nbCC);

    /**
     * Reads the oldest Color Code object from the last frame received 
     *
     * @note this function also manage the read pointer of the Color Code FIFO (ie : after use function point the next object to be read)
     *
     * @return a T_PixyCCBloc (see .h for details)
     */
    T_pixyCCBloc getCCBloc (void); 
        
    /**
     * Reads the oldest Normal object from the last frame received 
     *
     * @note this function also manage the read pointer of the Normal FIFO (ie : after use function point the next object to be read)
     *
     * @return a T_PixyNMBloc (see .h for details)
     */
    T_pixyNMBloc getNMBloc (void); 
        
    /**
     * Set the Brightness of the Pixy
     *
     * @param   brightness level (between 0 and 255)
     */
    void setBrightness (Byte brightness); 

    /**
     * Gives information about a new image taken by Pixy.
     * @return  0 if no new image or  1 if there is a new image
     */
    int checkNewImage (void);

    /**
     * Gives information about a detected Pixy.
     * @return  -1 if no Pixy detected or  0 if Pixy detected
     */
    int checkPixy (void);

private :

    int Pixy_CCFrameIsNew, Pixy_NMFrameIsNew;

    /**
     * Fonction d'interruption de lecture sur la liaison série.
     */
    void getPixyByte();


};

#endif /* PIXY_H */
