/* mbed TextLCD Library, for a 4-bit LCD based on HD44780
 * Copyright (c) 2007-2010, sford, http://mbed.org
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

#include "myTextLCD.h"
#include "mbed.h"
#include "MODSERIAL.h"

extern MODSERIAL pc;                    // definiert in main

TextLCD::TextLCD(PinName rs, PinName e, PinName d4, PinName d5,
                 PinName d6, PinName d7, LCDType type) : _rs(rs),
        _e(e), _d(d4, d5, d6, d7),
        _type(type) {

    _e  = 1;
    _rs = 0;            // command mode
    _type = LCD16x2;
    
    // die Anzeige soll hier zwischengespeichert werden
    for(int i = 0; i < 16; i++)
    {
        zeile1_text[i] = 0x20; // Leerzeichen
        zeile2_text[i] = 0x20; // Leerzeichen
        zeile1_flag[i] = false;
        zeile2_flag[i] = false;   
    }
    
    lcd_flag = true;
    
    wait(0.015);        // Wait 15ms to ensure powered up

    // send "Display Settings" 3 times (Only top nibble of 0x30 as we've got 4-bit bus)
    for (int i=0; i<3; i++) {
        writeByte(0x3);
        wait(0.00164);  // this command takes 1.64ms, so wait for it
    }
    writeByte(0x2);     // 4-bit mode
    wait(0.000040f);    // most instructions take 40us

    writeCommand(0x28); // Function set 001 BW N F - -
    writeCommand(0x0C);
    writeCommand(0x6);  // Cursor Direction and Display Shift : 0000 01 CD S (CD 0-left, 1-right S(hift) 0-no, 1-yes
    cls();
}

//---------------------------------------------------
// in der Phase, wo die LCD Anzeige von einer Eingabe
// benötigt wird, kann mit diesem Falg die Ausgabe
// abgeschaltet werden.

void TextLCD::set_lcd_flag(bool flag)
{
   lcd_flag = flag; 
}

// hier werden die Daten in die LCD Anzeige geschrieben
// im neuen Ablauf werden hier die Daten in die Variable geschrieben

void TextLCD::character(int column, int row, int c) 
{
    
    if (row == 0)
    { 
      if (zeile1_text[column] != c)
      {
        zeile1_text[column] = c;
        zeile1_flag[column] = true;   
      }  
    }
    else
    {
      if (zeile2_text[column] != c)
      {
        zeile2_text[column] = c;
        zeile2_flag[column] = true;          
      }         
    }
}

// -----------------------------------------------------
// den Text im Speicher auf die LCD Anzeige ausgeben

void TextLCD::writeLCD()
{
    int i;
    
    if (lcd_flag)
    {
        // Text auf Zeile 1 ausgeben
        for (i = 0; i<16; i++)
        {
            if (zeile1_flag[i])
            {   
                int a = address(i, 0);
                writeCommand(a);
                writeData(zeile1_text[i]);
                zeile1_flag[i] = false;
            }
        }
        
        // Text auf Zeile 2 ausgeben
        for (i = 0; i<16; i++)
        {
            if (zeile2_flag[i])
            {   
                int a = address(i, 1);
                writeCommand(a);
                writeData(zeile2_text[i]);
                zeile2_flag[i] = false;
            }
        }
    }   
}

// --------------------------------------------------------
// Nachdem die LCD Anzeige wieder die Freigabe bekommt
// wird der ganze Speicher auf die LCD Anzeige ausgeben
//
void TextLCD::writeALL()
{
    int i, a;
    
    // Text auf Zeile 1 ausgeben
    a = address(0, 0);
    writeCommand(a);
    wait_us(40);
    
    for (i = 0; i<16; i++)
    {
        writeData(zeile1_text[i]);
        zeile1_flag[i] = false;
    }
    
    // Text auf Zeile 2 ausgeben
    
    a = address(0, 1);
    writeCommand(a);    
    wait_us(40);
    
    for (i = 0; i<16; i++)
    {
         writeData(zeile2_text[i]);
         zeile2_flag[i] = false;
    } 
    
    // Text auf Zeile 1 ausgeben
    a = address(0, 0);
    writeCommand(a);
    wait_us(40);  
}

void TextLCD::cls() 
{
    writeCommand(0x01); // cls, and set cursor to 0
    wait(0.00164f);     // This command takes 1.64 ms
    locate(0, 0);
    
}

void TextLCD::locate(int column, int row) 
{
    _column = column;
    _row = row;
}

int TextLCD::_putc(int value) 
{
    if (value == '\n') {
        _column = 0;
        _row++;
        if (_row >= rows()) {
            _row = 0;
        }
    } else {
        character(_column, _row, value);
        _column++;
        if (_column >= columns()) {
            _column = 0;
            _row++;
            if (_row >= rows()) {
                _row = 0;
            }
        }
    }
    return value;
}

int TextLCD::_getc() {
    return -1;
}

void TextLCD::writeByte(int value) 
{
    _d = value >> 4;
    wait_us(1); 
    _e = 0;
    wait_us(40);    
    _e = 1;
    _d = value >> 0;
    wait_us(1);
    _e = 0;
    wait_us(40); 
    _e = 1;
    wait_us(42);    // most instructions take 40us   
}

void TextLCD::writeCommand(int command) 
{
    _rs = 0;
    writeByte(command);
}

void TextLCD::writeData(int data) 
{
    _rs = 1;
    writeByte(data);
}

int TextLCD::address(int column, int row) 
{
    switch (_type) {
        case LCD20x4:
            switch (row) {
                case 0:
                    return 0x80 + column;
                case 1:
                    return 0xc0 + column;
                case 2:
                    return 0x94 + column;
                case 3:
                    return 0xd4 + column;
            }
        case LCD16x2B:
            return 0x80 + (row * 40) + column;
        case LCD16x2:
        case LCD20x2:
        default:
            return 0x80 + (row * 0x40) + column;
    }
}

int TextLCD::columns() 
{
    switch (_type) {
        case LCD20x4:
        case LCD20x2:
            return 20;
        case LCD16x2:
        case LCD16x2B:
        default:
            return 16;
    }
}

int TextLCD::rows() 
{
    switch (_type) {
        case LCD20x4:
            return 4;
        case LCD16x2:
        case LCD16x2B:
        case LCD20x2:
        default:
            return 2;
    }
}

void TextLCD::print_LCD(char* Pbuffer,char line_num, char offset)
{
    unsigned int i=0;
    
    int a = address(offset,line_num);
    writeCommand(a);
    
    while (Pbuffer[i] !=  0)
    {
        writeData(Pbuffer[i]);  
        i++;    
    }
}