#include "mbed.h"
#include "PM2_Libary.h"
#include <cstdint>
#include <cstdio>
#include "math.h"
//*******************************************************************************************************************************************************************
// Defined Variables in mm coming from Hardware-team. Need to be updated
const float wheel_diameter                = 30; // diameter of wheel with caterpillar to calculate mm per wheel turn (4)
const float arm_length                    = 118.5; // lenght of arm from pivotpoint to pivotpoint (3)
const float dist_arm_attach_distsensor    = 20; // distance between pivot point arm  on body to start distancesensor on top in horizontal (6)
const float dist_distsensors              = 200; // distance between the two distancesensors on top of Wall-E (9)
const float dist_arm_ground               = 51; // distance between pivotpoint arm and ground (5)
const float dist_arm_attach_OK_griparea   = 10.5 ; // Height of Grappler cutout to grapple Stair (8) (maybe add 1mm so gripper is a bit over the plate)
const float dist_grappleratt_grappler_uk  = 36.5; // distance between pivotpoint Grappler and bottom edge (?)
const float grip_area_depth               = 32.5;

const float height_stairs                 = 100; // height to top of next stairstep in mm
//***********************************************************************************************************************************************************
// declaration of Input - Output pins

// user button on nucleo board
Timer user_button_timer;            // create Timer object which we use to check if user button was pressed for a certain time (robust against signal bouncing)
InterruptIn user_button(PC_13);     // create InterruptIn interface object to evaluate user button falling and rising edge (no blocking code in ISR)
void user_button_pressed_fcn();     // custom functions which gets executed when user button gets pressed and released, definition below
void user_button_released_fcn();

// Sharp GP2Y0A41SK0F, 4-40 cm IR Sensor
// define variable to store measurement from infrared distancesensor in mm
float ir_distance_mm_L;
float ir_distance_mm_R;  

AnalogIn ir_analog_in_Distance_L(PC_2);  
AnalogIn ir_analog_in_Distance_R(PC_3);
AnalogIn ir_analog_in_Lookdown_B(PC_5);
AnalogIn ir_analog_in_Lookdown_F(PB_1);
// create AnalogIn object to read in infrared distance sensor, 0...3.3V are mapped to 0...1

// 78:1, 100:1, ... Metal Gearmotor 20Dx44L mm 12V CB
DigitalOut enable_motors(PB_15);    // create DigitalOut object to enable dc motors
const float   pwm_period_s = 0.00005f;    // define pwm period time in seconds and create FastPWM objects to command dc motors

//motor pin declaration
FastPWM pwm_M_right (PB_13);    //motor pin decalaration for wheels right side         
FastPWM pwm_M_left  (PA_9);     //motor pin decalaration for wheels left side 
FastPWM pwm_M_arm   (PA_10);    //motor pin decalaration for arm 

//Encoder pin declaration
EncoderCounter  encoder_M_right (PA_6, PC_7); //encoder pin decalaration for wheels right side
EncoderCounter  encoder_M_left  (PB_6, PB_7); //encoder pin decalaration for wheels left side
EncoderCounter  encoder_M_arm   (PA_0, PA_1); //encoder pin decalaration for arm
//***********************************************************************************************************************************************************
// Hardware controll Setup and functions (motors and sensors)

// create SpeedController and PositionController objects, default parametrization is for 78.125:1 gear box
const float max_voltage               = 12.0f;     // define maximum voltage of battery packs, adjust this to 6.0f V if you only use one batterypack
const float counts_per_turn_wheels    = 20.0f * 78.125f;    // define counts per turn at gearbox end (counts/turn * gearratio) for wheels
const float counts_per_turn_arm       = 20.0f * 78.125f * 100.0f;    // define counts per turn at gearbox end (counts/turn * gearratio) for arm
const float kn                        = 180.0f / 12.0f;      // define motor constant in rpm per V
const float k_gear                    = 100.0f / 78.125f;   // define additional ratio in case you are using a dc motor with a different gear box, e.g. 100:1 (DC with 100:1 has 2'000 turns for 360°)
const float kp                        = 0.1f;     // define custom kp, this is the default speed controller gain for gear box 78.125:1

//motors for tracks
PositionController positionController_M_right(counts_per_turn_wheels * k_gear, kn / k_gear, kp * k_gear, max_voltage, pwm_M_right, encoder_M_right); // parameters adjusted to 100:1 gear, we need a different speed controller gain here
PositionController positionController_M_left(counts_per_turn_wheels * k_gear, kn / k_gear, kp * k_gear, max_voltage, pwm_M_left, encoder_M_left); // parameters adjusted to 100:1 gear, we need a different speed controller gain here
//Arm Motor
PositionController positionController_M_Arm(counts_per_turn_arm * k_gear, kn / k_gear, kp * k_gear, max_voltage, pwm_M_arm, encoder_M_arm); // parameters adjusted to 100:1 gear, we need a different speed controller gain here

// PositionController positionController_M3(counts_per_turn, kn, max_voltage, pwm_M3, encoder_M3); // default 78.125:1 gear with default contoller parameters
//PositionController positionController_M3(counts_per_turn * k_gear, kn / k_gear, kp * k_gear, max_voltage, pwm_M3, encoder_M3); // parameters adjusted to 100:1 gear, we need a different speed controller gain here
//***********************************************************************************************************************************************************
// calculations for basic movment and controll

//placeholder variables for prototype testing
const int   drive_straight_mm = 200;  // placeholder for testing drives amount forward
const int   drive_back_mm = -200;    // placeholder for testing drives amount backwards
int         ToNextFunction = 0;      // current state of the system (which function is beeing executed)

// definition variables for calculations
const float   pi = 2 * acos(0.0); // definiton of pi
const float   start_deg_arm = -(asin((dist_arm_ground - dist_grappleratt_grappler_uk) / arm_length) * 180.0/pi); //calculates the starting degree of the arm (gripper has to touch ground in frotn of Wall-E)
const float   end_pos_lift_deg = 180 + (asin((dist_arm_ground - (dist_grappleratt_grappler_uk - dist_arm_attach_OK_griparea)) / arm_length) * 180 / pi) - start_deg_arm;  // calculates the degree which the arm has to have when lift_up has been executed.

// definition of rotation speeds for motors 0 = none 1.0 = max.
const float   max_speed_rps_wheel = 0.5f;  // define maximum speed that the position controller is changig the speed for the wheels, has to be smaller or equal to kn * max_voltage
const float   max_speed_rps_arm = 0.3f; // define maximum speed that the position controller is changig the speed for the arm, has to be smaller or equal to kn * max_voltage

// calculates the deg which the arm has to take to reach a certain height (the input height has to be the height of OK Gripper area)
// PARAM: height_mm = height which OK Gripperarea has to reach.
// RETURN: deg_arm = absolut Position in deg that the arm has to take.
float calc_arm_deg_for_height(int height_mm)
{
    float height_arm = height_mm - (dist_arm_ground - dist_arm_attach_OK_griparea); // calculates the height which only the arm has to cover (- attachement height (arm to robot) etc.)
    float deg_arm = asin(height_arm / arm_length) * 180.0/pi; // calculates the absolute degrees which the arm has to reach
    return deg_arm;
}

//calculates the deg which the wheels have to turn in order to cover specified distance in mm
//RETURN: deg_wheel = degree which the motor has to turn in order to cover distance(mm)
float wheel_dist_to_deg(int distance) // distance has to be in mm.
{
    float deg_wheel = distance/(wheel_diameter * pi) * 360 ;
    return deg_wheel;
}

// increments the Motor for defined degree from the current one
// PARAM: deg_to_turn = degree to turn the Motor
// PARAM: current_rotation = the current rotation of the Motor (Motor.getRotation())
// RETURN: new_turn_rotation = new Rotation value in rotations
float turn_relative_deg(float deg_to_turn, float current_rotation)
{
    float new_turn_rotation = current_rotation + deg_to_turn/360.0;
    return new_turn_rotation;
}

// sets the Motor to a specified degree in one rotation 
// PARAM: end_deg = new position of the arm in degree 0 <= value >=360
// PARAM: current_rotation = the current rotation of the Motor (Motor.getRotation())
// RETURN: new_partial_rotation = new deg value in rotations
float turn_absolut_deg(float end_deg, float current_rotations)
{
    int full_rotations;
	if(current_rotations > 0)
	{
		full_rotations = round(current_rotations - 0.5);	
	}
	else if(current_rotations < 0)
	{
		full_rotations = round(current_rotations + 0.5);	
	} 
	else
	{
		full_rotations = 0;
	}
    float new_partial_rotation = full_rotations - start_deg_arm/360 + end_deg/360;
    return new_partial_rotation;
}

//***********************************************************************************************************************************************************
// important calculatet constants for Wall-E
const double deg_up_from_horizon_to_stair = calc_arm_deg_for_height(height_stairs);

// import functions from file mapping
extern double powerx(double base, double pow2);
extern double mapping (float adc_value_mV);

//simple check if there is an object in proximity
//returns 0 if there is NO object present
//returns 1 if there is an object present
//returns 2 if the distance isn't in the expected range
uint8_t StepDetection(double distance)
{
    double d_valueMM = distance;
    if(d_valueMM >= 4) return 0;
    else if(d_valueMM < 4)  return 1;
    else if(d_valueMM <= 0 || d_valueMM > 100 ) return 2;
    else return 2;
}

// bring arm in starting position. Height of stairs.
void set_arm_stair_height()
{
    float initial_full_rotation = floor(positionController_M_Arm.getRotation());
    float rotations = (deg_up_from_horizon_to_stair - start_deg_arm) / 360 + initial_full_rotation;
    float diff;

    positionController_M_Arm.setDesiredRotation(rotations, max_speed_rps_arm); // command to turn motor to desired deg.
}

//Drives forward into the next step 
//Prameter:distance in milimeter
void drive_straight(float distance)
{
    float deg_to_turn = wheel_dist_to_deg(distance);

    float relativ_turns_rightmotor = turn_relative_deg(deg_to_turn, positionController_M_right.getRotation());
    float relativ_turns_leftmotor = turn_relative_deg(deg_to_turn, positionController_M_left.getRotation());
  
    positionController_M_right.setDesiredRotation(relativ_turns_rightmotor, max_speed_rps_wheel);
    positionController_M_left.setDesiredRotation(relativ_turns_leftmotor, max_speed_rps_wheel); 

}

//turns the arm until the robot is on the next step
void lift_up()
{
    float absolut_pos_arm = turn_absolut_deg(end_pos_lift_deg, positionController_M_Arm.getRotation()-1);
    positionController_M_Arm.setDesiredRotation(absolut_pos_arm, max_speed_rps_arm);
}
//***********************************************************************************************************************************************************

//Function which checks if sensors and motors have been wired correctly and the expectet results will happen. otherwise Wall-E will ???.
void check_start()
{

}

// while loop gets executed every main_task_period_ms milliseconds
int main_task_period_ms = 30;   // define main task period time in ms e.g. 30 ms -> main task runns ~33,33 times per second
Timer main_task_timer;          // create Timer object which we use to run the main task every main task period time in ms
//***********************************************************************************************************************************************************

int main(void)
{
    // attach button fall and rise functions to user button object
    user_button.fall(&user_button_pressed_fcn);
    user_button.rise(&user_button_released_fcn);

    while (true)
    {
        if (ToNextFunction>=1)
        {
            enable_motors = 1;
        }

        switch (ToNextFunction) 
        {

            case 1: 
            set_arm_stair_height();
            printf("Case 1: Position ARM (rot): %3.3f\n",positionController_M_Arm.getRotation());
            break;

            case 2: 
            drive_straight(drive_straight_mm);
            printf("Case 2: Position Right(rot): %3.3f;    Position Left (rot): %3.3f\n",
            positionController_M_right.getRotation(),positionController_M_left.getRotation());
            break;

            case 3: 
            lift_up();
            printf("Case 3: Position ARM (rot): %3.3f\n",positionController_M_Arm.getRotation());
            break;

            case 4: 
            drive_straight(drive_back_mm);
            printf("Case 4: Position Right(rot): %3.3f;    Position Left (rot): %3.3f\n",
            positionController_M_right.getRotation(),positionController_M_left.getRotation());
            ToNextFunction = 0;
            break;

            default:  ;
        } 
    }
    // read timer and make the main thread sleep for the remaining time span (non blocking)
    int main_task_elapsed_time_ms = std::chrono::duration_cast<std::chrono::milliseconds>(main_task_timer.elapsed_time()).count();
    thread_sleep_for(main_task_period_ms - main_task_elapsed_time_ms);
    return 0;
}


void user_button_pressed_fcn()
{
    user_button_timer.start();
    user_button_timer.reset();
}

void user_button_released_fcn() 
{
    // read timer and toggle do_execute_main_task if the button was pressed longer than the below specified time
    int user_button_elapsed_time_ms = std::chrono::duration_cast<std::chrono::milliseconds>(user_button_timer.elapsed_time()).count();
    user_button_timer.stop();
    if (user_button_elapsed_time_ms > 200) 
    {
       ToNextFunction += 1;
    }
}