/**
@file protocol.h
@brief Protocol definitions.
*/

/*
Copyright 2016 Erik Perillo <erik.perillo@gmail.com>

This file is part of piranha-ptc.

This is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

This is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
See the GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this. If not, see <http://www.gnu.org/licenses/>.
*/


#ifndef __PIRANHA_PROTOCOL_H__
#define __PIRANHA_PROTOCOL_H__

//@{
///PID parameters range.
#define PID_PARAMS_MIN 0.0
#define PID_PARAMS_MAX 100.0
//@}

//@{
///Ground velocity range.
#define GND_VEL_MIN -100.0
#define GND_VEL_MAX 100.0
//@}

//@{
///Angle reference range (in radians).
#define PI 3.141593
#define ABS_ANG_REF_MIN -PI
#define ABS_ANG_REF_MAX PI
//@}

//@{
///Angle reference from robot range (in radians).
#define REL_ANG_REF_MIN -PI
#define REL_ANG_REF_MAX PI
//@}

//@{
///Angle reference from magnetometer range (in radians).
#define MAG_ANG_REF_MIN -PI
#define MAG_ANG_REF_MAX PI
//@}

//@{
///Break intensity
#define BRAKE_INTENSITY_MIN 0.0
#define BRAKE_INTENSITY_MAX 100.0
//@}

//@{
///Jogging speed ratio.
#define BRAKE_PERIOD_MIN 0.0
#define BRAKE_PERIOD_MAX 100.0
//@}

//@{
///Jogging speed period (in seconds).
#define JOG_VEL_PERIOD_MIN 0.0
#define JOG_VEL_PERIOD_MAX 300.0
//@}

//@{
///Jogging speed ratio.
#define JOG_VEL_RATIO_MIN 0.0
#define JOG_VEL_RATIO_MAX 1.0
//@}

//@{
///Magnetometer calibration values.
#define MAG_CALIB_MIN -750.0
#define MAG_CALIB_MAX 750.0
//@}

///Messages to send via protocol.
enum
{
	///Do nothing.
	NONE,

	///Brake the robot.
	BRAKE,

	///Reset gyroscope.
	GYRO_ZERO,

	///Set zero axis to current angle measure.
	ANG_SET,

	///Reset angle zero axis.
	ANG_RST,

	///Turn on leds
	LED_ON,

	///Turn off leds
	LED_OFF,

	///Set new angle reference relative to zero axis.
	ABS_ANG_REF,

	///Set new angle reference relative to robot axis.
	REL_ANG_REF,

	///Set new angle reference relative to north using magnetometer.
	MAG_ANG_REF,

	///Set new ground velocity for robot.
	GND_VEL,

	///Set new jogging speed for robot.
	JOG_VEL,

	///Magnetometer calibration (min_x, max_x, min_y, max_y).
	MAG_CALIB,

	///Send PID control parameters (P, I, D, N).
	PID_PARAMS
};

#define MSG_HEADER_SIZE 1
#define MSG_VAL_SIZE 2
#define MSG_MAX_NUM_VALS 4
#define MSG_BUF_LEN (MSG_HEADER_SIZE + MSG_VAL_SIZE*MSG_MAX_NUM_VALS)
#define MSG_HEADER_IDX 0
#define MSG_VALS_START_IDX (MSG_HEADER_IDX + 1)

#define SENDER_PORT 7532
#define SENDER_IFACE_ADDR "192.168.7.2"
#define SENDER_NETMASK_ADDR "255.255.255.0"
#define SENDER_GATEWAY_ADDR "0.0.0.0"

#define RECEIVER_PORT 7533
#define RECEIVER_IFACE_ADDR "192.168.7.3"
#define RECEIVER_NETMASK_ADDR "255.255.255.0"
#define RECEIVER_GATEWAY_ADDR "0.0.0.0"

#endif

