#include "mbed.h"
#include <stdint.h>
#include <stdlib.h>
#include <stdio.h>
#include <time.h>
#include <string.h>
#include <ctype.h>

#include <string>
#include <vector>
#include "jtagPinState.h"


#define CURRENT_TIME 1456081493   //from http://www.epochconverter.com/

// Length of the Instruction register
#define IR_LENGTH       3            

// Supported Instructions
#define CMD_EXTEST          3'b000
#define CMD_SAMPLE_PRELOAD  3'b001
#define CMD_IDCODE          3'b010
#define CMD_INTSCAN         3'b101
#define CMD_BYPASS          3'b111
 
#define IDCODE                0x00001003
#define BOUNDARY_SCAN_LENGTH  143
#define SCAN_CHAIN_LENGTH     509
    

#define ACTIVE_RESET 0

#define SUBMENU 2
#define ZERO      48        // 48 = 0x30 = ascii 0
#define ONE       49
#define TWO       50
#define THREE     51
#define FOUR      52
#define RETURN 13   // 13 = 0x0d = ascii carriage return

#define OFF    0
#define ON     1
#define TOGGLE 2
#define FAIL   0
#define PASS   1

#define TDO_BIT_POS           9    // PC9

#define TDI_BIT_POS           5    // PA5 
#define TCK_BIT_POS           6    // PA6
#define TMS_BIT_POS           7    // PA7
#define TRST_BIT_POS          8    // PA8
#define CAPTURE_TDO_BIT_POS   13   // PA13

#define STATE_TEST_LOGIC_RESET 0
#define STATE_RUN_TEST_IDLE    1
#define STATE_SELECT_DR_SCAN   2
#define STATE_CAPTURE_DR       3
#define STATE_SHIFT_DR         4
#define STATE_EXIT1_DR         5
#define STATE_PAUSE_DR         6
#define STATE_EXIT2_DR         7
#define STATE_UPDATE_DR        8
#define STATE_SELECT_IR_SCAN   9
#define STATE_CAPTURE_IR      10
#define STATE_SHIFT_IR        11
#define STATE_EXIT1_IR        12
#define STATE_PAUSE_IR        13
#define STATE_EXIT2_IR        14
#define STATE_UPDATE_IR       15

#define IN_MASK      0x00000200   // TDO
#define OUT_MASK     0x000021e0   // CAPTURE_TDO, TRST, TMS, TCK, TDI
//#define OUT_MASK     0x000001e0   // TRST, TMS, TCK, TDI


void goto_STATE_TEST_LOGIC_RESET(void);
void printTestResultLED1(int, string);


string stateNames[STATE_UPDATE_IR + 1];

time_t theTime;

Serial pc(SERIAL_TX, SERIAL_RX);
SPI adc1_spi(SPI_MOSI, SPI_MISO, SPI_SCK);
DigitalOut myLED1(LED1);
InterruptIn  goButton(PC_13);

//const uint16_t NUM_SAMPLES = 100;

bool chFlag = false;

unsigned char ch = '0';
bool loopContinuously   = false;
bool printCommandResults = false;
bool printStates        = false;
bool verbose = false;


uint8_t state_menu=0;
uint8_t state_exit =0;
double speed=0.1;
int pat = 1,tog=0;

int TMS = 0;

int state, next_state;

PortOut outPins(PortA, OUT_MASK);
PortIn   inPins(PortC, IN_MASK);



//------------------------------------------------------------------------------
// stateTransition
//------------------------------------------------------------------------------
void stateTransition(void) {
    next_state = state;
    switch (state) {
        case STATE_TEST_LOGIC_RESET:
            if (TMS == 0) {
                next_state = STATE_RUN_TEST_IDLE;
            }
            break;
        case  STATE_RUN_TEST_IDLE:
            if (TMS == 1) {
                next_state = STATE_SELECT_DR_SCAN;
            }
            break;
        case  STATE_SELECT_DR_SCAN:
            if (TMS == 0) {
                next_state = STATE_CAPTURE_DR;
            } else {
                next_state = STATE_SELECT_IR_SCAN;
            }
            break;
        case  STATE_CAPTURE_DR:
            if (TMS == 0) {
                next_state = STATE_SHIFT_DR;
            } else {
                next_state = STATE_EXIT1_DR;
            }
            break;
        case  STATE_SHIFT_DR:
            if (TMS == 1) {
                next_state = STATE_EXIT1_DR;
            }
            break;
        case  STATE_EXIT1_DR:
            if (TMS == 0) {
                next_state = STATE_PAUSE_DR;
            } else {
                next_state = STATE_UPDATE_DR;
            }
            break;
        case  STATE_PAUSE_DR:
            if (TMS == 1) {
                next_state = STATE_EXIT2_DR;
            }
            break;
        case  STATE_EXIT2_DR:
            if (TMS == 0) {
                next_state = STATE_SHIFT_DR;
            } else {
                next_state = STATE_UPDATE_DR;
            }
            break;
        case  STATE_UPDATE_DR:
            if (TMS == 0) {
                next_state = STATE_RUN_TEST_IDLE;
            } else {
                next_state = STATE_SELECT_DR_SCAN;
            }
            break;
        case  STATE_SELECT_IR_SCAN:
            if (TMS == 0) {
                next_state = STATE_CAPTURE_IR;
            } else {
                next_state = STATE_TEST_LOGIC_RESET;
            }
            break;
        case  STATE_CAPTURE_IR:
            if (TMS == 0) {
                next_state = STATE_SHIFT_IR;
            } else {
                next_state = STATE_EXIT1_IR;
            }
            break;
        case  STATE_SHIFT_IR:
            if (TMS == 1) {
                next_state = STATE_EXIT1_IR;
            }
            break;
        case  STATE_EXIT1_IR:
            if (TMS == 0) {
                next_state = STATE_PAUSE_IR;
            } else {
                next_state = STATE_UPDATE_IR;
            }
            break;
        case  STATE_PAUSE_IR:
            if (TMS == 1) {
                next_state = STATE_EXIT2_IR;
            }
            break;
        case  STATE_EXIT2_IR:
            if (TMS == 0) {
                next_state = STATE_SHIFT_IR;
            } else {
                next_state = STATE_UPDATE_IR;
            }
            break;
        case  STATE_UPDATE_IR:
            if (TMS == 0) {
                next_state = STATE_RUN_TEST_IDLE;
            } else {
                next_state = STATE_SELECT_DR_SCAN;
            }
            break;
        default:
            pc.printf("\n\rERROR!  Illegal state trasition.");
            break;
    }
    state = next_state;
    if (printStates) {
        pc.printf("\n\rTMS = %0d ::  state = %s", TMS, stateNames[state]);
        fflush(stdout);
    }
}


//------------------------------------------------------------------------------
// initTime
//------------------------------------------------------------------------------
static void initTime(void){
    set_time(CURRENT_TIME); //from http://www.epochconverter.com/
}

//------------------------------------------------------------------------------
// toggleLED1
//------------------------------------------------------------------------------
void toggleLED1(int choice = TOGGLE){
    switch (choice) {
        case TOGGLE:
            myLED1 = !myLED1;
            break;
        case ON:
            myLED1 = ON;
            break;
        case OFF:
            myLED1 = OFF;
            break;
        default:
            printf("\n\nERROR!  Invalid value sent to toggleLED1().");
    }
 }
 

//------------------------------------------------------------------------------
// JTAG
//------------------------------------------------------------------------------
vector<int> JTAG(vector<jtagPinState> inData) {

    vector <uint32_t> outVals;
    vector <bool> captureTDO;
    uint32_t outVal = 0x00000000;
    vector<int> TDO_value;
    
    // Set nth bit of number to x.
    //number ^= (-x ^ number) & (1UL << n);
    // Set nth bit of number to 1.
    //number |= 1UL << n;
    // Clear nth bit of number to 0.
    //number &= ~(1UL << n);
    // Read nth bit of number.
    //bit = (number >> n) & 1U;
    for (int i = 0; i < inData.size(); i++) {
        // TCK FALLING EDGE
        // Set input bits to their new value, but set TCK low so that the 
        // inputs are changed on the falling edge of TCK.
        outVal ^= (-(inData[i].tdi()) ^ outVal) & (1UL << TDI_BIT_POS);
        outVal ^= (-(inData[i].tms()) ^ outVal) & (1UL << TMS_BIT_POS);
        outVal ^= (-(inData[i].trst()) ^ outVal) & (1UL << TRST_BIT_POS);        
        outVal ^= (-(inData[i].capture_tdo()) ^ outVal) & (1UL << CAPTURE_TDO_BIT_POS);
        outVals.push_back(outVal);
        
        if (inData[i].tck()) {
            // TCK RISING EDGE
            // If new TCK data is high, repeat the same input data and create
            // rising edge of TCK.
            outVal ^= (-(inData[i].tck()) ^ outVal) & (1UL << TCK_BIT_POS);
            outVals.push_back(outVal);
            // Set TCK data to zero for next falling edge.
            outVal &= ~(1UL << TCK_BIT_POS);
        } else {
            // If new TCK data is low (probably not normal case), just repeat 
            // the same input data with TCK still low.
            outVals.push_back(outVal);
        }
    }
    
    LOOP:
    int tdoIndex = 0;
    for (int i = 0; i < (2 * inData.size()); i++) {
        outPins = outVals[i];
        // if TCK is high, capture TMS value and update state.
        if ((outVals[i] >> TCK_BIT_POS) & 1U) {
            TMS = ((outVals[i] >> TMS_BIT_POS) & 1U);
            stateTransition();
        } else {
            // TCK is low, so capture TDO value if requested.
            if ((outVals[i] >> CAPTURE_TDO_BIT_POS) & 1U) {
                TDO_value.push_back((inPins >> TDO_BIT_POS) & 1U);
                //printf("\n\rTDO_value[%0d] = %d", tdoIndex, TDO_value[tdoIndex]);
                fflush(stdout);
                tdoIndex++;
            }
        }
    }
    if (loopContinuously) goto LOOP;
    //printf("\n\rTDO_value = 0x%.8x", TDO_value);
    //fflush(stdout);
    return TDO_value;
}


//------------------------------------------------------------------------------
// goto_STATE_RUN_TEST_IDLE
//------------------------------------------------------------------------------
 void goto_STATE_RUN_TEST_IDLE(void) {

    vector<int> readVector;
    jtagPinState tdata;
    vector<jtagPinState> tdArr;
    
    goto_STATE_TEST_LOGIC_RESET();
    
    // Drive TMS low for 1 TCK to get to STATE_RUN_TEST_IDLE.
    // TDI, TDO, TCK, TMS, TRST, CAPTURE_TDO
    tdArr.push_back(jtagPinState(0,0,1,0,!ACTIVE_RESET,0));
    readVector = JTAG(tdArr);
}


//------------------------------------------------------------------------------
// resetJTAG
//------------------------------------------------------------------------------
 void goto_STATE_TEST_LOGIC_RESET(void) {

    vector<int> readVector;
    jtagPinState tdata;
    vector<jtagPinState> tdArr;
    
    // Reset JTAG for 8 TCK cycles.
    // TDI, TDO, TCK, TMS, TRST, CAPTURE_TDO
    for (int i = 0; i < 8; i++) {
        tdArr.push_back(jtagPinState(0,0,1,1,ACTIVE_RESET,0));
    }
    
    readVector = JTAG(tdArr);
}


//------------------------------------------------------------------------------
// run_EXTEST
//------------------------------------------------------------------------------
void run_EXTEST() {
    
    vector<int> readVector;
    jtagPinState tdata;
    vector<jtagPinState> tdArr;
    
    // STATE_RUN_TEST_IDLE
    // TDI, TDO, TCK, TMS, TRST, CAPTURE_TDO
    tdArr.push_back(jtagPinState(0,0,1,1,!ACTIVE_RESET,0));
    tdArr.push_back(jtagPinState(0,0,1,1,!ACTIVE_RESET,0));
    tdArr.push_back(jtagPinState(0,0,1,0,!ACTIVE_RESET,0));
    tdArr.push_back(jtagPinState(0,0,1,0,!ACTIVE_RESET,0));

    // STATE_SHIFT_IR
    // TDI, TDO, TCK, TMS, TRST, CAPTURE_TDO
    tdArr.push_back(jtagPinState(0,0,1,0,!ACTIVE_RESET,0));
    tdArr.push_back(jtagPinState(0,0,1,0,!ACTIVE_RESET,0));
    tdArr.push_back(jtagPinState(0,0,1,1,!ACTIVE_RESET,0));

    // STATE_EXIT1_IR
    // TDI, TDO, TCK, TMS, TRST, CAPTURE_TDO
    tdArr.push_back(jtagPinState(0,0,1,1,!ACTIVE_RESET,0));
    tdArr.push_back(jtagPinState(0,0,1,1,!ACTIVE_RESET,0));
    tdArr.push_back(jtagPinState(0,0,1,0,!ACTIVE_RESET,0));
    tdArr.push_back(jtagPinState(0,0,1,0,!ACTIVE_RESET,0));
    
    // STATE_SHIFT_DR
    // TDI, TDO, TCK, TMS, TRST, CAPTURE_TDO
    for (int  j = 0; j < 154; j++) {
        tdArr.push_back(jtagPinState(0,0,1,0,!ACTIVE_RESET,1));
    }
    tdArr.push_back(jtagPinState(0,0,1,1,!ACTIVE_RESET,1));

    // STATE_EXIT1_DR
    // TDI, TDO, TCK, TMS, TRST, CAPTURE_TDO
    tdArr.push_back(jtagPinState(0,0,1,1,!ACTIVE_RESET,0));
    tdArr.push_back(jtagPinState(0,0,1,0,!ACTIVE_RESET,0));

    readVector = JTAG(tdArr);

    pc.printf("\n\rEXTEST: \n\r");
    for (int i = 0; i < readVector.size(); i++) {
        pc.printf("%0d", readVector[i]);
    }
    fflush(stdout);
}
    

//------------------------------------------------------------------------------
// read_SAMPLE_PRELOAD
//------------------------------------------------------------------------------
vector<int> read_SAMPLE_PRELOAD(vector<int> writeVector) {
        
    vector<int> readVector;
    jtagPinState tdata;
    vector<jtagPinState> tdArr;
    int j;

    // STATE_RUN_TEST_IDLE
    // TDI, TDO, TCK, TMS, TRST, CAPTURE_TDO
    tdArr.push_back(jtagPinState(0,0,1,1,!ACTIVE_RESET,0));
    tdArr.push_back(jtagPinState(0,0,1,1,!ACTIVE_RESET,0));
    tdArr.push_back(jtagPinState(0,0,1,0,!ACTIVE_RESET,0));
    tdArr.push_back(jtagPinState(0,0,1,0,!ACTIVE_RESET,0));

    // STATE_SHIFT_IR
    // TDI, TDO, TCK, TMS, TRST, CAPTURE_TDO
    tdArr.push_back(jtagPinState(1,0,1,0,!ACTIVE_RESET,0));
    tdArr.push_back(jtagPinState(0,0,1,0,!ACTIVE_RESET,0));
    tdArr.push_back(jtagPinState(0,0,1,1,!ACTIVE_RESET,0));

    // STATE_EXIT1_IR
    // TDI, TDO, TCK, TMS, TRST, CAPTURE_TDO
    tdArr.push_back(jtagPinState(0,0,1,1,!ACTIVE_RESET,0));
    tdArr.push_back(jtagPinState(0,0,1,1,!ACTIVE_RESET,0));
    tdArr.push_back(jtagPinState(0,0,1,0,!ACTIVE_RESET,0));
    tdArr.push_back(jtagPinState(0,0,1,0,!ACTIVE_RESET,0));
    
    // STATE_SHIFT_DR
    // TDI, TDO, TCK, TMS, TRST, CAPTURE_TDO
    for (j = 0; j < (writeVector.size() - 1); j++) {
        tdArr.push_back(jtagPinState(writeVector[j],0,1,0,!ACTIVE_RESET,1));
    }
    tdArr.push_back(jtagPinState(writeVector[j],0,1,1,!ACTIVE_RESET,1));

    // STATE_EXIT1_DR
    // TDI, TDO, TCK, TMS, TRST, CAPTURE_TDO
//    tdArr.push_back(jtagPinState(0,0,1,1,!ACTIVE_RESET,0));
//    tdArr.push_back(jtagPinState(0,0,1,0,!ACTIVE_RESET,0));

    readVector = JTAG(tdArr);
    
    return readVector;

}


//------------------------------------------------------------------------------
// read_IDCODE
//------------------------------------------------------------------------------
uint32_t read_IDCODE() {
    
    vector<int> readVector;
    jtagPinState tdata;
    vector<jtagPinState> tdArr;
    
    // STATE_RUN_TEST_IDLE
    // TDI, TDO, TCK, TMS, TRST, CAPTURE_TDO
    tdArr.push_back(jtagPinState(0,0,1,1,!ACTIVE_RESET,0));
    tdArr.push_back(jtagPinState(0,0,1,1,!ACTIVE_RESET,0));
    tdArr.push_back(jtagPinState(0,0,1,0,!ACTIVE_RESET,0));
    tdArr.push_back(jtagPinState(0,0,1,0,!ACTIVE_RESET,0));

    // STATE_SHIFT_IR
    // TDI, TDO, TCK, TMS, TRST, CAPTURE_TDO
    tdArr.push_back(jtagPinState(0,0,1,0,!ACTIVE_RESET,0));
    tdArr.push_back(jtagPinState(1,0,1,0,!ACTIVE_RESET,0));
    tdArr.push_back(jtagPinState(0,0,1,1,!ACTIVE_RESET,0));

    // STATE_EXIT1_IR
    // TDI, TDO, TCK, TMS, TRST, CAPTURE_TDO
    tdArr.push_back(jtagPinState(0,0,1,1,!ACTIVE_RESET,0));
    tdArr.push_back(jtagPinState(0,0,1,1,!ACTIVE_RESET,0));
    tdArr.push_back(jtagPinState(0,0,1,0,!ACTIVE_RESET,0));
    tdArr.push_back(jtagPinState(0,0,1,0,!ACTIVE_RESET,0));
    
    // STATE_SHIFT_DR
    // TDI, TDO, TCK, TMS, TRST, CAPTURE_TDO
    for (int j = 0; j < 31; j++) {
        tdArr.push_back(jtagPinState(0,0,1,0,!ACTIVE_RESET,1));
    }
    tdArr.push_back(jtagPinState(0,0,1,1,!ACTIVE_RESET,1));

    // STATE_EXIT1_DR
    // TDI, TDO, TCK, TMS, TRST, CAPTURE_TDO
    tdArr.push_back(jtagPinState(0,0,1,1,!ACTIVE_RESET,0));
    tdArr.push_back(jtagPinState(0,0,1,0,!ACTIVE_RESET,0));

    readVector = JTAG(tdArr);

    // Convert readVector to a 32-bit integer.
    uint32_t val = 0x00000000;
    // Set nth bit of number to x.
    //number ^= (-x ^ number) & (1UL << n);
    for (int n = 0; n < readVector.size(); n++) {
        val ^= (-(readVector[n]) ^ val) & (1UL << n);
    }
    return val;
}


//------------------------------------------------------------------------------
// read_INTSCAN
//------------------------------------------------------------------------------
vector<int> read_INTSCAN(vector<int> writeVector) {
    
    vector<int> readVector;
    vector<jtagPinState> tdArr;
    int j;
    
    // STATE_RUN_TEST_IDLE
    // TDI, TDO, TCK, TMS, TRST, CAPTURE_TDO
    tdArr.push_back(jtagPinState(0,0,1,1,!ACTIVE_RESET,0));
    tdArr.push_back(jtagPinState(0,0,1,1,!ACTIVE_RESET,0));
    tdArr.push_back(jtagPinState(0,0,1,0,!ACTIVE_RESET,0));
    tdArr.push_back(jtagPinState(0,0,1,0,!ACTIVE_RESET,0));

    // STATE_SHIFT_IR
    // TDI, TDO, TCK, TMS, TRST, CAPTURE_TDO
    tdArr.push_back(jtagPinState(1,0,1,0,!ACTIVE_RESET,0));
    tdArr.push_back(jtagPinState(0,0,1,0,!ACTIVE_RESET,0));
    tdArr.push_back(jtagPinState(1,0,1,1,!ACTIVE_RESET,0));

    // STATE_EXIT1_IR
    // TDI, TDO, TCK, TMS, TRST, CAPTURE_TDO
    tdArr.push_back(jtagPinState(0,0,1,1,!ACTIVE_RESET,0));
    tdArr.push_back(jtagPinState(0,0,1,1,!ACTIVE_RESET,0));
    tdArr.push_back(jtagPinState(0,0,1,0,!ACTIVE_RESET,0));
    tdArr.push_back(jtagPinState(0,0,1,0,!ACTIVE_RESET,0));
    
    // STATE_SHIFT_DR
    // TDI, TDO, TCK, TMS, TRST, CAPTURE_TDO
    for (j = 0; j < (writeVector.size() - 1); j++) {
        tdArr.push_back(jtagPinState(writeVector[j],0,1,0,!ACTIVE_RESET,1));
    }
    tdArr.push_back(jtagPinState(writeVector[j],0,1,1,!ACTIVE_RESET,1));

    // STATE_EXIT1_DR
    // TDI, TDO, TCK, TMS, TRST, CAPTURE_TDO
    tdArr.push_back(jtagPinState(0,0,1,1,!ACTIVE_RESET,0));
    tdArr.push_back(jtagPinState(0,0,1,0,!ACTIVE_RESET,0));

    readVector.clear();
    readVector = JTAG(tdArr);

    return readVector;
}


//------------------------------------------------------------------------------
// read_BYPASS
//------------------------------------------------------------------------------
vector<int> read_BYPASS(vector<int> writeVector) {
    
    vector<int> readVector;
    vector<jtagPinState> tdArr;
    int j;
    
    // STATE_RUN_TEST_IDLE
    // TDI, TDO, TCK, TMS, TRST, CAPTURE_TDO
    tdArr.push_back(jtagPinState(0,0,1,1,!ACTIVE_RESET,0));
    tdArr.push_back(jtagPinState(0,0,1,1,!ACTIVE_RESET,0));
    tdArr.push_back(jtagPinState(0,0,1,0,!ACTIVE_RESET,0));
    tdArr.push_back(jtagPinState(0,0,1,0,!ACTIVE_RESET,0));

    // STATE_SHIFT_IR
    // TDI, TDO, TCK, TMS, TRST, CAPTURE_TDO
    tdArr.push_back(jtagPinState(1,0,1,0,!ACTIVE_RESET,0));
    tdArr.push_back(jtagPinState(1,0,1,0,!ACTIVE_RESET,0));
    tdArr.push_back(jtagPinState(1,0,1,1,!ACTIVE_RESET,0));


    // STATE_EXIT1_IR
    // TDI, TDO, TCK, TMS, TRST, CAPTURE_TDO
    tdArr.push_back(jtagPinState(0,0,1,1,!ACTIVE_RESET,0));
    tdArr.push_back(jtagPinState(0,0,1,1,!ACTIVE_RESET,0));
    tdArr.push_back(jtagPinState(0,0,1,0,!ACTIVE_RESET,0));
    tdArr.push_back(jtagPinState(0,0,1,0,!ACTIVE_RESET,0));
    
    // STATE_SHIFT_DR
    // TDI, TDO, TCK, TMS, TRST, CAPTURE_TDO
    for (j = 0; j < (writeVector.size() - 1); j++) {
        tdArr.push_back(jtagPinState(writeVector[j],0,1,0,!ACTIVE_RESET,0));
        tdArr.push_back(jtagPinState(writeVector[j],0,1,0,!ACTIVE_RESET,1));
    }
    tdArr.push_back(jtagPinState(writeVector[j],0,1,1,!ACTIVE_RESET,0));
    
    readVector = JTAG(tdArr);
    
    return readVector;
}


//------------------------------------------------------------------------------
// callback
//------------------------------------------------------------------------------
 void callback() {
    // Note: you need to actually read from the serial to clear the RX interrupt
    //pc.printf("rcvd %c\n", pc.getc());
    ch = pc.getc();
    chFlag = true;
 
//    pc.printf("%c :: ascii value 0x%x = %d\n\r\n", ch, ch, ch);
    pc.printf("%c\n\r", ch);
}

/*
//------------------------------------------------------------------------------
//  getInitialBoundaryScanData
//------------------------------------------------------------------------------
vector<int> getInitialBoundaryScanData(void) {
    
    vector<int> initialVector;
    
    // Put the unique initial scan out data for the STHENO device into the  
    // initial vector.
    for (int j = 0; j < 15; j++) {
        initialVector.push_back(0);
    }
    bool bdat = true;
    for (int j = 0; j < BOUNDARY_SCAN_LENGTH - 15; j++) {
        initialVector.push_back(bdat);
        bdat = !bdat;
    }
    
    return initialVector;
}    
*/  
 
//------------------------------------------------------------------------------
//  runSAMPLE_PRELOAD
//------------------------------------------------------------------------------
void runSAMPLE_PRELOAD(void) {
    
    vector<int> readVector;
    vector<int> writeVector;
    
    
    for (int j = 0; j < BOUNDARY_SCAN_LENGTH; j++) {
        writeVector.push_back(0);
    }
    for (int j = 0; j < BOUNDARY_SCAN_LENGTH; j++) {
        writeVector.push_back(1);
    }
    for (int j = 0; j < BOUNDARY_SCAN_LENGTH; j++) {
        writeVector.push_back(0);
    }
    for (int j = 0; j < BOUNDARY_SCAN_LENGTH; j++) {
        writeVector.push_back(rand() % 2);
    }
    for (int j = 0; j < BOUNDARY_SCAN_LENGTH; j++) {
        writeVector.push_back(rand() % 2);
    }     
     
    readVector = read_SAMPLE_PRELOAD(writeVector);

    pc.printf("\n\rSAMPLE_PRELOAD:  (Vector size = %0d)  BOUNDARY_SCAN_LENGTH = %0d\n\r", readVector.size(), BOUNDARY_SCAN_LENGTH);

    for (int i = 0; i < readVector.size(); i++) {
        if ((i > 0) && ((i % BOUNDARY_SCAN_LENGTH) == 0)) { printf("\n\r"); fflush(stdout); }
        pc.printf("%0d", readVector[i]);
    }
    fflush(stdout);
}

//------------------------------------------------------------------------------
//  runINTSCAN
//------------------------------------------------------------------------------
void runINTSCAN(void) {
    
    vector<int> readVector;
    vector<int> writeVector;
    
    
    for (int j = 0; j < SCAN_CHAIN_LENGTH; j++) {
        writeVector.push_back(0);
    }
    for (int j = 0; j < SCAN_CHAIN_LENGTH; j++) {
        writeVector.push_back(1);
    }
    for (int j = 0; j < SCAN_CHAIN_LENGTH; j++) {
        writeVector.push_back(0);
    }
    for (int j = 0; j < SCAN_CHAIN_LENGTH; j++) {
        writeVector.push_back(rand() % 2);
    }
//    for (int j = 0; j < SCAN_CHAIN_LENGTH; j++) {
//        writeVector.push_back(rand() % 2);
//    }     
     
    readVector = read_INTSCAN(writeVector);    
        
    pc.printf("\n\rINTSCAN:  (Vector size = %0d)  SCAN_CHAIN_LENGTH = %0d\n\r", readVector.size(), SCAN_CHAIN_LENGTH);
    for (int i = 0; i < readVector.size(); i++) {
//        if ((i > 0) && ((i % 100) == 0)) { printf("\n\r"); fflush(stdout); }
        if ((i > 0) && ((i % SCAN_CHAIN_LENGTH) == 0)) { printf("\n\r"); fflush(stdout); }
        pc.printf("%0d", readVector[i]);
    }
    fflush(stdout);
}

//------------------------------------------------------------------------------
//  runBYPASS
//------------------------------------------------------------------------------
void runBYPASS(void) {
    
    vector<int> readVector;
    vector<int> writeVector;
    
    for (int j = 0; j < (rand() % 4096); j++) {
        writeVector.push_back(rand() % 2);
    }
     
    readVector = read_BYPASS(writeVector);

    pc.printf("\n\rBYPASS:  (Vector size = %0d)\n\r", readVector.size());

    for (int i = 0; i < readVector.size(); i++) {
        pc.printf("%0d", readVector[i]);
    }
    fflush(stdout);
}
    
//------------------------------------------------------------------------------
//  runTestBYPASS
//------------------------------------------------------------------------------
bool runTestBYPASS(void) {

    vector<int> readVector;
    vector<int> writeVector;
    bool didTestPass = true;
    
    for (int j = 0; j < (rand() % 4096); j++) {
        writeVector.push_back(rand() % 2);
    }
     
    readVector = read_BYPASS(writeVector);

    if (verbose) {
        pc.printf("\n\rBYPASS:  (Vector size = %0d)\n\r", readVector.size());
        for (int j = 0; j < readVector.size(); j++) {
            pc.printf("%0d", readVector[j]);
        }
    }
    for (int j = 0; j < readVector.size(); j++) {
        if (readVector[j] != writeVector[j]) {
            didTestPass = false;
            if (verbose) {
                printf("\n\rFailed!  Bypass data %0d:  actual %0d != %0d expect", j, readVector[j], writeVector[j]);
              fflush(stdout);
            }
        }
    }    
    
    return didTestPass;
}

//------------------------------------------------------------------------------
//  runTestSAMPLE_PRELOAD
//------------------------------------------------------------------------------
bool runTestSAMPLE_PRELOAD(void) {
//    vector<int> initialVector;
    vector<int> readVector;
    vector<int> writeVector;
    bool didTestPass = true;
    
    for (int j = 0; j < BOUNDARY_SCAN_LENGTH; j++) {
        writeVector.push_back(0);
    }
    for (int j = 0; j < BOUNDARY_SCAN_LENGTH; j++) {
        writeVector.push_back(1);
    }
    for (int j = 0; j < BOUNDARY_SCAN_LENGTH; j++) {
        writeVector.push_back(0);
    }
    for (int j = 0; j < BOUNDARY_SCAN_LENGTH; j++) {
        writeVector.push_back(rand() % 2);
    }
    for (int j = 0; j < BOUNDARY_SCAN_LENGTH; j++) {
        writeVector.push_back(rand() % 2);
    }
    
    readVector = read_SAMPLE_PRELOAD(writeVector);
    
/*    
    // Get the initial boundary scan data.
    initialVector = getInitialBoundaryScanData();
    //  Check initial boundary scan data.    
    for (int j = 0; j < BOUNDARY_SCAN_LENGTH; j++) {
        if (readVector[j] != initialVector[j]) {
            didTestPass = false;
            if (verbose) {
                printf("\n\rFailed!  Initial boundary scan data %0d:  %0d != %0d", j, readVector[j], initialVector[j]);
                fflush(stdout);
            }
        }
    }
*/

    for (int j = BOUNDARY_SCAN_LENGTH; j < readVector.size(); j++) {
        if (readVector[j] != writeVector[j - BOUNDARY_SCAN_LENGTH]) {
            didTestPass = false;
            if (verbose) {
                printf("\n\rFailed!  Boundary scan data %0d != %0d", readVector[j], writeVector[j - BOUNDARY_SCAN_LENGTH]);
                fflush(stdout);
            }
        }
    }
    
    if (verbose) {
        pc.printf("\n\rSAMPLE_PRELOAD:  (Vector size = %0d)  BOUNDARY_SCAN_LENGTH = %0d\n\r", readVector.size(), BOUNDARY_SCAN_LENGTH);

        for (int i = 0; i < readVector.size(); i++) {
            if ((i > 0) && ((i % BOUNDARY_SCAN_LENGTH) == 0)) { printf("\n\r"); fflush(stdout); }
            pc.printf("%0d", readVector[i]);
        }
        fflush(stdout);
    }
    
    return didTestPass;
}


//------------------------------------------------------------------------------
//  runTestINTSCAN
//------------------------------------------------------------------------------
bool runTestINTSCAN(void) {
//    vector<int> initialVector;
    vector<int> readVector;
    vector<int> writeVector;
    bool didTestPass = true;
    
    for (int j = 0; j < SCAN_CHAIN_LENGTH; j++) {
        writeVector.push_back(0);
    }
    for (int j = 0; j < SCAN_CHAIN_LENGTH; j++) {
        writeVector.push_back(1);
    }
//    for (int j = 0; j < SCAN_CHAIN_LENGTH; j++) {
//        writeVector.push_back(0);
//    }
    for (int j = 0; j < SCAN_CHAIN_LENGTH; j++) {
        writeVector.push_back(rand() % 2);
    }
    for (int j = 0; j < SCAN_CHAIN_LENGTH; j++) {
        writeVector.push_back(rand() % 2);
    }
    
    readVector = read_INTSCAN(writeVector);

/*  Ignore the initial internal scan data for now.
    
    // Get the initial internal scan data.
    initialVector = getInitialInternalScanData();  
    //  Check initial internal scan data.    
    for (int j = 0; j < SCAN_CHAIN_LENGTH; j++) {
        if (readVector[j] != initialVector[j]) {
            didTestPass = false;
            printf("\n\rFailed!  Initial internal scan data %0d != %0d", readVector[j], initialVector[j]);
            fflush(stdout);
        }
    }
*/

    for (int j = SCAN_CHAIN_LENGTH; j < readVector.size(); j++) {
        if (readVector[j] != writeVector[j - SCAN_CHAIN_LENGTH]) {
            didTestPass = false;
            if (verbose) {
                printf("\n\rFailed!  Internal scan data %0d != %0d", readVector[j], writeVector[j - SCAN_CHAIN_LENGTH]);
                fflush(stdout);
            }
        }
    }
    
    if (verbose) {
        pc.printf("\n\rINTSCAN:  (Vector size = %0d)  SCAN_CHAIN_LENGTH = %0d\n\r", readVector.size(), SCAN_CHAIN_LENGTH);

        for (int i = 0; i < readVector.size(); i++) {
            if ((i > 0) && ((i % SCAN_CHAIN_LENGTH) == 0)) { printf("\n\r"); fflush(stdout); }
            pc.printf("%0d", readVector[i]);
        }
        fflush(stdout);
    }
    
    return didTestPass;
}


//------------------------------------------------------------------------------
//  runTestIDCODE
//------------------------------------------------------------------------------
bool runTestIDCODE(void) {
    uint32_t val;
    bool didTestPass = true;
    
    val = read_IDCODE();
    if (val != IDCODE) { 
        if (verbose) { 
            printf("\n\n\rIDCODE actual 0x%.8x  !=  0x%.8x expect", val, IDCODE);
        }
        didTestPass = false;
    } else {
        if (verbose) {
            printf("\n\n\rIDCODE actual 0x%.8x  ==  0x%.8x expect", val, IDCODE);    
        }
    }
    fflush(stdout);
    return didTestPass;
}


//------------------------------------------------------------------------------
//  runAllTests
//------------------------------------------------------------------------------
void runAllTests(void) {
    bool resultEXTEST = true;
    bool resultSAMPLE_PRELOAD = true;
    bool resultIDCODE = true;
    bool resultINTSCAN = true;
    bool resultBYPASS = true;

    uint32_t val;


    toggleLED1(OFF);
    wait_ms(500);
    toggleLED1(ON);
    wait_ms(250);

    goto_STATE_RUN_TEST_IDLE();
    resultSAMPLE_PRELOAD = runTestSAMPLE_PRELOAD();

    goto_STATE_RUN_TEST_IDLE();
    resultIDCODE = runTestIDCODE();

    goto_STATE_RUN_TEST_IDLE();
    resultINTSCAN = runTestINTSCAN();

    goto_STATE_RUN_TEST_IDLE();
    resultBYPASS = runTestBYPASS();

    printTestResultLED1(resultSAMPLE_PRELOAD, "Test SAMPLE_PRELOAD");
    printTestResultLED1(resultIDCODE,         "Test IDCODE");
    printTestResultLED1(resultINTSCAN,        "Test INTSCAN");
    printTestResultLED1(resultBYPASS,         "Test BYPASS");

    if ((resultEXTEST) && (resultSAMPLE_PRELOAD) && (resultIDCODE) && (resultINTSCAN) && (resultBYPASS)) {
        printTestResultLED1(PASS, "Overall test result.");
    } else {
        printTestResultLED1(FAIL, "Overall test result.");
    }
}
    
//------------------------------------------------------------------------------
//  printTestResultLED1
//------------------------------------------------------------------------------
void printTestResultLED1(int result, string description) {
    if (result) {
        pc.printf("\n\rPASSED!   %s", description);
        toggleLED1(ON);
    } else {
        pc.printf("\n\r------------------------------------------------------");
        pc.printf("\n\r-     FAILED!   %s", description);
        pc.printf("\n\r------------------------------------------------------");
        toggleLED1(OFF);
    }
    fflush(stdout);
}


//------------------------------------------------------------------------------
// JTAGMenu
//------------------------------------------------------------------------------
void JTAGMenu(void) {    
    bool needToPrintMenu = true;
    bool continueLoop = true;
        
    while(continueLoop) {    

        if (needToPrintMenu) {
            needToPrintMenu = false;
            pc.printf("\n\rCurrent state = %s", stateNames[state]);                    
            pc.printf("\n\n\r\t\tJTAG Menu");
            pc.printf("\n\r\t1.  Go to STATE_TEST_LOGIC_RESET.");
            pc.printf("\n\r\t2.  Go to STATE_RUN_TEST_IDLE.");
            pc.printf("\n\r\t3.  Read IDCODE.");
            pc.printf("\n\r\t4.  BYPASS mode.");
            pc.printf("\n\r\t5.  SAMPLE_PRELOAD");
            pc.printf("\n\r\t6.  INTSCAN");
            pc.printf("\n\r\t7.  EXTEST");
            pc.printf("\n\rEnter number of choice or <Enter> to exit menu: ");
        }
        fflush(stdout);

        if(chFlag) {

            chFlag = false;
            switch(ch){
                case '1':
                    goto_STATE_TEST_LOGIC_RESET();
                    needToPrintMenu = true;
                    break;
                case '2':
                    goto_STATE_RUN_TEST_IDLE();
                    needToPrintMenu = true;
                    break;
                case '3':
                    uint32_t val;
                    val = read_IDCODE();
                    printf("\n\rIDCODE = 0x%.8x", val);
                    fflush(stdout);
                    needToPrintMenu = true;
                    break;
                case '4':
                    runBYPASS();
                    needToPrintMenu = true;
                    break;
                case '5':
                    runSAMPLE_PRELOAD();
                    needToPrintMenu = true;
                    break;
                case '6':
                    runINTSCAN();
                    needToPrintMenu = true;
                    break;
                case '7':
                    run_EXTEST();
                    needToPrintMenu = true;
                    break;
                case RETURN:
                    continueLoop = false;
                    break;
                default:
                    pc.printf("\n\rInvalid value entered.  Please try again.");
                    needToPrintMenu = true;
                    break;
            }
        }
    }
}
  

//------------------------------------------------------------------------------
// configurationMenu
//------------------------------------------------------------------------------
void configurationMenu(void) {    
    bool needToPrintMenu = true;
    bool continueLoop = true;
        
    while(continueLoop) {    

        if (needToPrintMenu) {
            needToPrintMenu = false;
            pc.printf("\n\n\r\t\tConfiguration Menu");
            if (!printStates) {
                pc.printf("\n\r\t1.  Enable printStates.");
            } else {
                pc.printf("\n\r\t1.  Disable printStates.");
            }
            if (!printCommandResults) {
                pc.printf("\n\r\t2.  Enable printCommmandResults.");
            } else {
                pc.printf("\n\r\t2.  Disable printCommandResults.");
            }
            if (!loopContinuously) {
                pc.printf("\n\r\t3.  Enable continously loop on command.");
            } else {
                pc.printf("\n\r\t3.  Disable continously loop on command.");
            }
            if (!verbose) {
                pc.printf("\n\r\t4.  Enable verbose printing.");
            } else {
                pc.printf("\n\r\t4.  Disable verbose printing.");
            }
            pc.printf("\n\rEnter number of choice or <Enter> to exit menu: ");
        }
        fflush(stdout);

        if(chFlag) {
            chFlag = false;
            switch(ch){
                case '1':
                    printStates = !printStates;
                    needToPrintMenu = true;
                    break;
                case '2':
                    printCommandResults = !printCommandResults;
                    needToPrintMenu = true;
                    break;
                case '3':
                    loopContinuously = !loopContinuously;
                    needToPrintMenu = true;
                    break;
                case '4':
                    verbose = !verbose;
                    needToPrintMenu = true;
                    break;
                case RETURN:
                    continueLoop = false;
                    break;
                default:
                    pc.printf("\n\rInvalid value entered.  Please try again.");
                    needToPrintMenu = true;
                    break;
            }
        }
    }
}

//------------------------------------------------------------------------------
// mainMenu
//------------------------------------------------------------------------------
void mainMenu(void) {
    
    bool needToPrintMenu = true;
    bool continueLoop = true;
        
    while(continueLoop) {    

        if (needToPrintMenu) {
            needToPrintMenu = false;
            pc.printf("\n\n\r\t\tMain Menu");
            pc.printf("\n\r\t0.  Exit Program.");
            pc.printf("\n\r\t1.  JTAG Menu.");
            pc.printf("\n\r\t2.  Run all tests.");
            pc.printf("\n\r\t3.  Configuration Menu.");
            pc.printf("\n\rEnter Number of Menu Choice: ");
        }
        fflush(stdout);

        if(chFlag) {

            chFlag = false;
            switch(ch){
                case '0':
                    continueLoop = false;
                    break;
                case '1':
                    JTAGMenu();
                    needToPrintMenu = true;
                    break;
                case '2':
                    runAllTests();
                    needToPrintMenu = true;
                    break;
                case '3':
                    configurationMenu();
                    needToPrintMenu = true;
                    break;
                default:
                    pc.printf("\n\rInvalid value entered.  Please try again.");
                    needToPrintMenu = true;
                    break;
            }
        }
    }
}


//------------------------------------------------------------------------------
// main
//------------------------------------------------------------------------------
int main() {
/*    
    stateNames.push_back("STATE_TEST_LOGIC_RESET");
    stateNames.push_back("STATE_RUN_TEST_IDLE");
    stateNames.push_back("STATE_SELECT_DR_SCAN");
    stateNames.push_back("STATE_CAPTURE_DR");
    stateNames.push_back("STATE_SHIFT_DR");
    stateNames.push_back("STATE_EXIT1_DR");
    stateNames.push_back("STATE_PAUSE_DR");
    stateNames.push_back("STATE_EXIT2_DR");
    stateNames.push_back("STATE_UPDATE_DR");
    stateNames.push_back("STATE_SELECT_IR_SCAN");
    stateNames.push_back("STATE_CAPTURE_IR");
    stateNames.push_back("STATE_SHIFT_IR");
    stateNames.push_back("STATE_EXIT1_IR");
    stateNames.push_back("STATE_PAUSE_IR");
    stateNames.push_back("STATE_EXIT2_IR");
    stateNames.push_back("STATE_UPDATE_IR");
*/    
    stateNames[STATE_TEST_LOGIC_RESET] = "STATE_TEST_LOGIC_RESET";
    stateNames[STATE_RUN_TEST_IDLE]= "STATE_RUN_TEST_IDLE";
    stateNames[STATE_SELECT_DR_SCAN]= "STATE_SELECT_DR_SCAN";
    stateNames[STATE_CAPTURE_DR]= "STATE_CAPTURE_DR";
    stateNames[STATE_SHIFT_DR]= "STATE_SHIFT_DR";
    stateNames[STATE_EXIT1_DR]= "STATE_EXIT1_DR";
    stateNames[STATE_PAUSE_DR]= "STATE_PAUSE_DR";
    stateNames[STATE_EXIT2_DR]= "STATE_EXIT2_DR";
    stateNames[STATE_UPDATE_DR]= "STATE_UPDATE_DR";
    stateNames[STATE_SELECT_IR_SCAN]= "STATE_SELECT_IR_SCAN";
    stateNames[STATE_CAPTURE_IR]= "STATE_CAPTURE_IR";
    stateNames[STATE_SHIFT_IR]= "STATE_SHIFT_IR";
    stateNames[STATE_EXIT1_IR]= "STATE_EXIT1_IR";
    stateNames[STATE_PAUSE_IR]= "STATE_PAUSE_IR";
    stateNames[STATE_EXIT2_IR]= "STATE_EXIT2_IR";
    stateNames[STATE_UPDATE_IR]= "STATE_UPDATE_IR";

    // Setup the spi for 16 bits data, high steady state clock,
    // second edge capture, with a 1MHz clock rate
    adc1_spi.format(16,3);
    adc1_spi.frequency(2000000);
    
    pc.baud(9600);
    //pc.format(8,SerialBase::None,1);
    
    pc.attach(&callback);
    
    printf("\r\n--- Reset ---\r\n\r\n");
    
    initTime();
    
    struct tm * timeinfo;

    time (&theTime);
    timeinfo = localtime (&theTime);
    printf ("Current local time and date: %s\r\n", asctime(timeinfo));

    myLED1 = 0;
    goButton.rise(&runAllTests);
          
    TMS = 1;
    state = STATE_TEST_LOGIC_RESET;
    stateTransition();
          

    mainMenu();
    
    printf("\n\rExiting program!\n\n\r");
    fflush(stdout);
}


