#include "mbed.h"
#include "rtos.h"
#include "EthernetInterface.h"
#include "CommChannel.h"
#include "LokerFraming.h"
#include "const_defines.h"
#include <vector>

/*******************************************************************************
PRIVATE PinOut
*******************************************************************************/
Serial dbg(USBTX, USBRX);
DigitalOut led(LED_BLUE);


/*******************************************************************************
PRIVATE VARIABLE
*******************************************************************************/
int device_status;
int kirim_status;
int COL_SCAN, ROW_SCAN;
const char* SERVER_ADDRESS = "192.168.0.34";
const char* IP = "192.168.0.111";
const char* SM = "255.255.255.0";
const char* GW = "192.168.0.1";

TCPSocketConnection socket;
CommChannel COM;
bool comStat;

std::vector<int> RCV_BUFF;
int timerRxTOutCnt;
bool tmr10msTick;
Ticker timer10ms;
void timer10msTick();

bool tmr10sTick;
Ticker timer10s;
void timer10stick();

void rxRoutine(void const *args)
{
    while(true)
    {
        int n=0;
        char buf[256];
        n = socket.receive(buf, 256);
        if(n > 0)
        {
            buf[n] = '\0';
//            dbg.printf("Received message from server: '%s'\n\r", buf);
            for(int i=0; i<n; i++)
            {
                RCV_BUFF.push_back((int)buf[i]);
            }
            timerRxTOutCnt = 0;
        }
        Thread::wait(100);
    }   
}

//const int ECHO_SERVER_PORT = 5002;

Ticker run;
void runled();

int main()
{
    tmr10msTick = false;
    tmr10sTick = false;
    kirim_status = false;
    
    EthernetInterface eth;
    eth.init(IP, SM, GW); //Use DHCP
    int n = eth.connect();
    if(n)dbg.printf(GRE"Connected with IP : %s\r\n"DEF, IP);
    else dbg.printf(RED"Not Connected with IP : %s\r\n"DEF, IP);
    
    comStat = false;
    COM.connect(SERVER_ADDRESS);
    dbg.printf(GRE"Connected to Server at %s\r\n"DEF, SERVER_ADDRESS);
    
    timer10ms.attach_us(&timer10msTick, 10000);
    
    set_time(1483228800);
    run.attach(&runled, 1.0f);
    Thread thread1(rxRoutine);
    
    int Loop = 0;
    
    while(true)
    {
        char timeBuf[32];
        time_t seconds = time(NULL);
        strftime(timeBuf, 32, "%Y-%m-%d+%H:%M:%S", localtime(&seconds));
        
        comStat = socket.is_connected();
        if(!comStat)
        {
            socket.close();
            COM.connect(SERVER_ADDRESS);
            dbg.printf(GRE"Connected to Server at %s\r\n"DEF, SERVER_ADDRESS);
        }
        
        if(tmr10msTick)
        {
            tmr10msTick = false;
            COM.Tick10ms();
        }
        
        // Receive message from server
        if(COM.dataAvailable() > 0)
        {
            dbg.printf(YEL"%s <<\t", timeBuf);
            std::vector<int> r = COM.read();
            for(int i=0; i<r.size(); i++)
                dbg.printf("0x%02X ", r[i]);
            dbg.printf("\r\n"DEF);
            
            LokerFraming frm;
            frm.ProcessFromHost(r);
            size_t sz = frm.RCV_QUE.size();
            if(sz > 0)
            {
                LokerFrm lf = frm.RCV_QUE.front();
                frm.RCV_QUE.pop();
                
                switch(lf.status)
                {
                    case '0' :
                        device_status = idle;
                        break;
                    case '1' :
                    case '2' :
                        ROW_SCAN = lf.Data[0];
                        COL_SCAN = lf.Data[1];
                        dbg.printf(YEL"++ Scanning : %c %c\r\n\r\n"DEF, ROW_SCAN, COL_SCAN);
                        device_status = lf.status;
                        break;
                    case '3' : // for send status of
                        kirim_status = true;
                        break;
                    case '4' : // for setting the rtc
                        break;
                    default : break;
                }
            }
        }
        
        // Send message to server
        if(kirim_status)
        {
            dbg.printf(YEL"%s >>\t", timeBuf);
            if(comStat)
            {
                srand(seconds);
                std::vector<int>kirim;
                for(int i=0; i<8; i++)kirim.push_back(rand() % 0xFF);
                TxFrm tData(kirim, device_status);
                
                std::vector<int>tMsg = LokerFraming::CreateFrame(tData);
                for(int i=0; i<tMsg.size(); i++)
                    dbg.printf("0x%02X ", tMsg[i]);
                dbg.printf("\r\n\r\n"DEF);
                
                COM.write(tMsg);
                kirim_status = false;
                Loop++;
                if(Loop > 4)
                {
                    Loop = 0;
                    device_status = idle;
                }
            }
            else
            {
                socket.close();
                COM.connect(SERVER_ADDRESS);
            }
        }
    }
}

/*******************************************************************************
PRIVATE FUNCTION
*******************************************************************************/
void runled()
{
    led = !led;
}

void timer10stick()
{
    tmr10sTick = true;
}

void timer10msTick()
{
    tmr10msTick = true;
}
