/** mDot_DS18B20 - Simple mDot temperature sensor using Dallas Semiconductors DS18B20 OneWire temperature sensor.
 * It used the OTA_AUTO join mode using saved parameters. If the config is to be reset then pin A2 on the
 * dev board must be held low during a reset or power up.
 *
 * Uses MultiTech mDot developer board http://www.multitech.com/models/94558010LF
 * Requires a MultiTech MultiConnect Conduit http://www.multitech.com/models/94557203LF
 *
 * Example JSON received on Conduit:
 { "chan": 5, "codr": "4/5", "datr": "SF9BW125", "freq": "869.5",
 "lsnr": "8.8", "modu": "LORA", "rfch": 1, "rssi": -41, "seqn": 13,
 "size": 12, "timestamp": "2015-07-22T21:19:11Z", "tmst": 517590990,
 "payload": "{\"tmp\":21.3}", "eui": "00:80:00:00:00:00:9a:63", "_msg
 id": "73bcd8dd.8c4328" }
 *
 */

#include "mbed.h"
#include "DS1820.h"
#include "mDot.h"
#include "MTSLog.h"
#include "MTSText.h"
#include <string>
#include <vector>

using namespace mts;

#define MIN(a,b) (((a)<(b))?(a):(b))
#define MAX(a,b) (((a)>(b))?(a):(b))

// these options must match the settings on your Conduit in
// /var/config/lora/lora-network-server.conf
static std::string config_network_name = "xxx";
static std::string config_network_pass = "xxx";

// Ignoring sub band for EU modules.
//static uint8_t config_frequency_sub_band = 1;

// mDot/dev board activity LED
//#define ACTIVITY_LED PA_0

// DS18B20 OneWire pin
// D13 on Dev Board, pin 18 on mDot
#define DATA_PIN     PA_5
// A0 on Dev Board, pin 20 on mDot
//#define DATA_PIN     PB_1

// A2 - input to reset LoRaWAN config. Pin 15 om mDot.
#define CONFIG_RESET PC_1

// Config Reset intput
DigitalIn configReset(CONFIG_RESET);

// Temperature sensor object
DS1820 probe(DATA_PIN);

// Serial via USB for debugging only
Serial pc(USBTX,USBRX);


int main()
{
    int32_t ret;
    mDot* dot;
    std::vector<uint8_t> send_data;
    std::vector<uint8_t> recv_data;

    float temperature = 0.0;

    // Enable internal pullup on input pin
    configReset.mode(PullUp);

    pc.baud(115200);
    pc.printf("Private LoRaWAN Temperature sensor\n\r");
    pc.printf("Test\n\r");

    // get a mDot handle
    dot = mDot::getInstance();

//    dot->setLogLevel(MTSLog::WARNING_LEVEL);
    dot->setLogLevel(MTSLog::TRACE_LEVEL);

    logInfo("Checking Config");

    // Test if we've already saved the config
    std::string configNetworkName = dot->getNetworkName();
    
    pc.printf("Name: %s\r\n",config_network_name.c_str());

    // Reset config if network name is different or pin is low then reset config.
    if( config_network_name.compare(configNetworkName) != 0 || !configReset ) {
        // Not saved config, reset
        logInfo("Setting Config");

        // reset to default config so we know what state we're in
        dot->resetConfig();

        // Set byte order - AEP less than 1.0.30 needs LSB
//        dot->setJoinByteOrder(mDot::LSB);
//        dot->setJoinByteOrder(mDot::MSB);       // This is default for > 1.0.30 Conduit

        dot->setPublicNetwork(true);
        logInfo("setting Join mode");
        if ((ret = dot->setJoinMode(mDot::AUTO_OTA)) != mDot::MDOT_OK) {
            logError("failed to set Join Mode %d:%s", ret, mDot::getReturnCodeString(ret).c_str());
        }

        // If on developer board then you can enable activity LED
        // Currently no spare pins that LEDs are connected too.
//    dot->setActivityLedPin( ACTIVITY_LED );
//    dot->setActivityLedEnable(false);

        // Have a decent nubmer of retries in connecting to LoRaWAN
        dot->setJoinRetries( 3 );

        // Set Spreading Factor, higher is lower data rate, smaller packets but longer range
        // Lower is higher data rate, larger packets and shorter range.
//        dot->setTxDataRate( mDot::SF_9 );
        dot->setTxDataRate( mDot::SF_12 );
        dot->setTxPower( 14 );
        dot->setAck( 0 );       // 1 retries on Ack, 0 to disable

//    Not applicable for 868MHz in EU
//    if ((ret = dot->setFrequencySubBand(config_frequency_sub_band)) != mDot::MDOT_OK) {
//        initStatus = false;
//        logError(dot, "failed to set frequency sub band", ret);
//    }

        if ((ret = dot->setNetworkName(config_network_name)) != mDot::MDOT_OK) {
            logError("failed to set network name %d:%s", ret, mDot::getReturnCodeString(ret).c_str());
        }

        if ((ret = dot->setNetworkPassphrase(config_network_pass)) != mDot::MDOT_OK) {
            logError("failed to set network password %d:%s", ret, mDot::getReturnCodeString(ret).c_str());
        }

        logInfo("Saving Config");

        // Save config
        if (! dot->saveConfig()) {
            logError("failed to save configuration");
        }
    } else {
        logInfo("Using existing Config");
    }
 
    logInfo("Joining Network");

    while ((ret = dot->joinNetwork()) != mDot::MDOT_OK) {
        logError("failed to join network [%d][%s]", ret, mDot::getReturnCodeString(ret).c_str());
        wait_ms(dot->getNextTxMs() + 1);
    }

    logInfo("Joined Network");

    probe.setResolution(9);

    char dataBuf[50];
    while( 1 ) {
        // This takes upto 750mS, way too long. Change to 9 bit resolution if not already used.

        probe.convertTemperature(true, DS1820::all_devices);         //Start temperature conversion, wait until ready
//        printf("It is %3.1fC\r\n", probe.temperature());
        // Output data as JSON e.g. {"tmp":21.3}
        temperature = probe.temperature();
        sprintf(dataBuf, "{\"tmp\":%3.1f}", temperature );
        send_data.clear();
        // probably not the most efficent way to do this
        for( int i=0; i< strlen(dataBuf); i++ )
            send_data.push_back( dataBuf[i] );

        if ((ret = dot->send(send_data)) != mDot::MDOT_OK) {
            logError("failed to send: [%d][%s]", ret, mDot::getReturnCodeString(ret).c_str());
        } else {
            logInfo("send data: %s", Text::bin2hexString(send_data).c_str());
        }

        // Should  sleep here and wakeup after a set interval.
        uint32_t sleep_time = MAX((dot->getNextTxMs() / 1000), 60);
        logInfo("going to sleep for %d seconds", sleep_time);

        // go to sleep and wake up automatically sleep_time seconds later
        dot->sleep(sleep_time, mDot::RTC_ALARM);

//        wait_ms(sleep_time * 1000);
    }

    return 0;
}
