#include "mbed.h"
#include "mDot.h"
#include "MTSLog.h"
#include <string>
#include <vector>
#include <algorithm>

static uint8_t config_network_addr[] = { 0x01, 0x02, 0x03, 0x04 };
static uint8_t config_network_nskey[] = {0x2B, 0x7E, 0x15, 0x16, 0x28, 0xAE, 0xD2, 0xA6, 0xAB, 0xF7, 0x15, 0x88, 0x09, 0xCF, 0x4F, 0x3C};
static uint8_t config_network_dskey[] = {0x2B, 0x7E, 0x15, 0x16, 0x28, 0xAE, 0xD2, 0xA6, 0xAB, 0xF7, 0x15, 0x88, 0x09, 0xCF, 0x4F, 0x3C};
static uint8_t config_frequency_sub_band = 7;

mDot* dot;

void setupNetwork();

int main() {
    // Set up the network
    setupNetwork();
    
    // Message you want to send
    std::string data_str = "Hello!";

    // Copy the message in an array of bytes
    std::vector<uint8_t> data;
    for (std::string::iterator it = data_str.begin(); it != data_str.end(); it++)
        data.push_back((uint8_t) *it);
    
    // Start the loop
    int32_t ret;
    while (true) {
        // Send the data
        if ((ret = dot->send(data)) != mDot::MDOT_OK) {
            // Oops, there was an error, check the debug screen
            logError("Failed to send", ret, mDot::getReturnCodeString(ret).c_str());
        } else {
            // Sent the data
            logInfo("Successfully sent data");
        }
        
        // Wait 5 seconds
        osDelay(std::max((uint32_t)5000, (uint32_t)dot->getNextTxMs()));
    }

    return 0;
}

void setupNetwork() {
    int32_t ret;

    // get a mDot handle
    dot = mDot::getInstance();
    
    // print library version information
    logInfo("Version: %s", dot->getId().c_str());

    // reset to default config so we know what state we're in
    dot->resetConfig();
    
    dot->setLogLevel(mts::MTSLog::INFO_LEVEL);

    dot->setJoinMode(mDot::MANUAL);

    // set up the mDot with our network information: frequency sub band, network name, and network password
    // these can all be saved in NVM so they don't need to be set every time - see mDot::saveConfig()
    
    // frequency sub band is only applicable in the 915 (US) frequency band
    logInfo("Setting frequency sub band");
    if ((ret = dot->setFrequencySubBand(config_frequency_sub_band)) != mDot::MDOT_OK) {
        logError("Failed to set frequency sub band %d:%s", ret, mDot::getReturnCodeString(ret).c_str());
    }
    
    std::vector<uint8_t> temp;
    
    if ((ret = dot->setPublicNetwork(true)) != mDot::MDOT_OK) {
        logError("Failed to enable public network %d:%s", ret, mDot::getReturnCodeString(ret).c_str());
    }
    
    for (int i = 0; i < 4; i++) {
        temp.push_back(config_network_addr[i]);    
    }

    logInfo("setting network addr");
    if ((ret = dot->setNetworkAddress(temp)) != mDot::MDOT_OK) {
        logError("failed to set network name %d:%s", ret, mDot::getReturnCodeString(ret).c_str());
    }
    
    temp.clear();
    for (int i = 0; i < 16; i++) {
        temp.push_back(config_network_nskey[i]);    
    }
    
    logInfo("setting network password");
    if ((ret = dot->setNetworkSessionKey(temp)) != mDot::MDOT_OK) {
        logError("failed to set network password %d:%s", ret, mDot::getReturnCodeString(ret).c_str());
    }
    
    temp.clear();
    for (int i = 0; i < 16; i++) {
        temp.push_back(config_network_dskey[i]);    
    }
    
    logInfo("setting network password");
    if ((ret = dot->setDataSessionKey(temp)) != mDot::MDOT_OK) {
        logError("failed to set network password %d:%s", ret, mDot::getReturnCodeString(ret).c_str());
    }
    
    // a higher spreading factor allows for longer range but lower throughput
    // in the 915 (US) frequency band, spreading factors 7 - 10 are available
    // in the 868 (EU) frequency band, spreading factors 7 - 12 are available
    logInfo("Setting TX spreading factor");
    if ((ret = dot->setTxDataRate(mDot::SF_8)) != mDot::MDOT_OK) {
        logError("Failed to set TX datarate %d:%s", ret, mDot::getReturnCodeString(ret).c_str());
    }
    
    // request receive confirmation of packets from the gateway
    logInfo("Disabling ACKs");
    if ((ret = dot->setAck(0)) != mDot::MDOT_OK) {
        logError("Failed to enable ACKs %d:%s", ret, mDot::getReturnCodeString(ret).c_str());
    }
    
    // save this configuration to the mDot's NVM
    logInfo("Saving config");
    if (! dot->saveConfig()) {
        logError("Failed to save configuration");
    }

    // attempt to join the network
    //logInfo("Joining network");
//    while ((ret = dot->joinNetwork()) != mDot::MDOT_OK) {
//        logError("Failed to join network %d:%s", ret, mDot::getReturnCodeString(ret).c_str());
//        // in the 868 (EU) frequency band, we need to wait until another channel is available before transmitting again
//        osDelay(std::max((uint32_t)1000, (uint32_t)dot->getNextTxMs()));
//    }
}