#include "mbed.h"
#include "mDot.h"
#include "MTSLog.h"
#include <string>
#include <vector>
#include <algorithm>

static uint8_t devAddr[] = { <insert DevAddr> }; //for example: { 0x44, 0x08, 0xCC, 0x9A };
static uint8_t nwkSKey[] = { <insert nwkSKey> }; //for example: { 0x99, 0xC8, 0xA0, 0x4C, 0x50, 0x25, 0x94, 0xD2, 0xEB, 0xFD, 0xD9, 0x37, 0x56, 0x3C, 0x83, 0x32 };
static uint8_t appSKey[] = { <insert appSKey> }; //for example: { 0x07, 0xA5, 0x9B, 0xAD, 0x11, 0x0B, 0x94, 0x6E, 0xEE, 0xE5, 0xF0, 0x87, 0xD0, 0x10, 0x35, 0x07 };

static uint8_t config_frequency_sub_band = 7;

mDot* dot;

void setupNetwork();

int main() {
    
    Serial pc(USBTX, USBRX);
    // Set up the network
    setupNetwork();
    
    // Message you want to send
    std::string data_str = "Hello!";

    // Copy the message in an array of bytes
    std::vector<uint8_t> data;
    for (std::string::iterator it = data_str.begin(); it != data_str.end(); it++)
        data.push_back((uint8_t) *it);
    
    // Start the loop
    int32_t ret;
    while (true) {
        // Send the data
        if ((ret = dot->send(data)) != mDot::MDOT_OK) {
            // Oops, there was an error, check the debug screen
            logError("Failed to send", ret, mDot::getReturnCodeString(ret).c_str());
            //pc.printf("Failed to send");
        } else {
            // Sent the data
            logInfo("Successfully sent data");
            //pc.printf("Successfully sent data");
        }
    
        // Wait 5 seconds
        osDelay(std::max((uint32_t)1000, (uint32_t)dot->getNextTxMs()));
    }

    return 0;
}

void setupNetwork() {
    
    //Serial pc(USBTX, USBRX);
    
    int32_t ret;

    // get a mDot handle
    dot = mDot::getInstance();
    
    // print library version information
    logInfo("Version: %s", dot->getId().c_str());

    // reset to default config so we know what state we're in
    dot->resetConfig();
    
    dot->setLogLevel(mts::MTSLog::INFO_LEVEL);
    
    //joinmode: [MANUAL = ABP]/[OTA]/[AUTO_OTA]
    dot->setJoinMode(mDot::MANUAL);
        
    // set up the mDot with our network information: frequency sub band, network name, and network password
    // these can all be saved in NVM so they don't need to be set every time - see mDot::saveConfig()
    
    // frequency sub band is only applicable in the 915 (US) frequency band
    logInfo("Setting frequency sub band");
    if ((ret = dot->setFrequencySubBand(config_frequency_sub_band)) != mDot::MDOT_OK) {
        logError("Failed to set frequency sub band %d:%s", ret, mDot::getReturnCodeString(ret).c_str());
    }
    
    std::vector<uint8_t> temp;
    

    if ((ret = dot->setPublicNetwork(true)) != mDot::MDOT_OK) {
        logError("Failed to enable public network %d:%s", ret, mDot::getReturnCodeString(ret).c_str());
    }
    

    for (int i = 0; i < 4; i++) {
        temp.push_back(devAddr[i]);    
    }
 
    logInfo("setting Device Address");
    if ((ret = dot->setNetworkAddress(temp)) != mDot::MDOT_OK) {
        logError("failed to set network name %d:%s", ret, mDot::getReturnCodeString(ret).c_str());
    }
    
    temp.clear();
    for (int i = 0; i < 16; i++) {
        temp.push_back(nwkSKey[i]);    
    }
    
    logInfo("setting Network Session Key ");
    if ((ret = dot->setNetworkSessionKey(temp)) != mDot::MDOT_OK) {
        logError("failed to set network password %d:%s", ret, mDot::getReturnCodeString(ret).c_str());
    }
    
    temp.clear();
    for (int i = 0; i < 16; i++) {
        temp.push_back(appSKey[i]);    
    }
    
    logInfo("setting App Session Key");
    if ((ret = dot->setDataSessionKey(temp)) != mDot::MDOT_OK) {
        logError("failed to set network password %d:%s", ret, mDot::getReturnCodeString(ret).c_str());
    }       
   
    // a higher spreading factor allows for longer range but lower throughput
    // in the 915 (US) frequency band, spreading factors 7 - 10 are available
    // in the 868 (EU) frequency band, spreading factors 7 - 12 are available
    logInfo("Setting TX spreading factor");
    if ((ret = dot->setTxDataRate(mDot::SF_8)) != mDot::MDOT_OK) {
        logError("Failed to set TX datarate %d:%s", ret, mDot::getReturnCodeString(ret).c_str());
    }
    
    // request receive confirmation of packets from the gateway
    logInfo("Disabling ACKs");
    if ((ret = dot->setAck(0)) != mDot::MDOT_OK) {
        logError("Failed to enable ACKs %d:%s", ret, mDot::getReturnCodeString(ret).c_str());
    }
    
    // save this configuration to the mDot's NVM
    logInfo("Saving config");
    if (! dot->saveConfig()) {
        logError("Failed to save configuration");
    }
}