#include "mbed.h"
#include "MS5607SPI.h"
#include "arm_math.h"

#define M_PI 3.14159265358979323846

#define N_SAMPLES 1024

#define PEAK_SEARCH_START 1 //first bin to include in peak search
#define PEAK_SEARCH_END 50 //last bin to include in peak search
#define SUM_AROUND_PEAK 1 //+/- this many bins

#define REGRESSION_SLOPE 53.77986f //from Excel fit to test data
#define REGRESSION_INTERCEPT -80.07f

float thresholds[] = {0.f, 10.f, 30.f, 50.f, 70.f}; //estimated t90 thresholds to turn on LEDs

//mbed class def
Serial pc(USBTX, USBRX); // tx, rx
MS5607SPI pressure_sensor(p5, p6, p7, p8); // mosi, miso, sclk, cs

#define N_LEDS 4
PwmOut leds[N_LEDS] = {LED1, LED2, LED3, LED4};

DigitalOut motor(p26);

Timer sample_timer;
Timer iter_timer;

float p_data[N_SAMPLES];
float p_data_raw[N_SAMPLES];
float f_data[N_SAMPLES];
float mag_data[N_SAMPLES / 2 + 1];

unsigned int tData[N_SAMPLES];

arm_rfft_fast_instance_f32 rfft;

float duty = 1;
float sampleRate;

int round(float number)
{
    return (number >= 0) ? (int)(number + 0.5) : (int)(number - 0.5);
}

int main()
{
    printf("Turrentine\n");
    
    //Configure PC serial link
    pc.baud(115200);
    
    //Configure motor PWM
    motor = 0;

    printf("Pump On\n");
    //turn pump on
    motor = duty;

    printf("Start loop\n");
    int iter_counter = 0; //Iteration counter
    iter_timer.start();
    float temperature_raw = pressure_sensor.getRawTemperature();
    printf("fsmpl titer mean_press fmax sum_peak    t90\n");
    int a = 0;
    //program loop    
    while(1) {
        temperature_raw = pressure_sensor.getRawTemperature();
        
        while (a<2000){
            pressure_sensor.getRawPressure();
            a++;
            }
        a = 0;
        sample_timer.reset();
        sample_timer.start();
            
        //Capture raw pressure samples
        for(int i = 0; i < N_SAMPLES; i++) {
            p_data_raw[i] = pressure_sensor.calculatePressure(pressure_sensor.getRawPressure(), temperature_raw);
            p_data[i] = p_data_raw[i];
        }
        
        sample_timer.stop();
        
        //Output raw data
        printf("$RAW\n");
        for(int i = 0; i < N_SAMPLES; i++) {
            printf("$%f\n", p_data_raw[i]);
            //pressure_sensor.calculatePressure(pressure_sensor.getRawPressure(), temperature_raw);
        }
        printf("$ENDRAW\n");
        
        //http://www.keil.com/pack/doc/CMSIS/DSP/html/group__RealFFT.html
        //Compute the RFFT
        //
        //p_data is trashed in the process
        //Result is packaged as [real_0, real_N/2, real_1, imag_1, real_2, imag_2 ... real_N/2-1, imag_N/2-1]
        arm_rfft_fast_init_f32(&rfft, N_SAMPLES);
        arm_rfft_fast_f32(&rfft, p_data, f_data, 0);
        
        //http://www.keil.com/pack/doc/CMSIS/DSP/html/group__cmplx__mag.html
        //Convert to magntiude, skip over the DC and fundamental terms
        
        arm_cmplx_mag_f32(&f_data[2], &mag_data[1], (N_SAMPLES / 2) - 1);
        //Fill in the first and last terms from the first entry in f_data
        mag_data[0] = f_data[0];
        mag_data[N_SAMPLES / 2] = f_data[1];
        
        
        
        //Find peak in spectrum
        float max_mag = -1;
        int max_mag_i = -1;
        for(int i = PEAK_SEARCH_START + SUM_AROUND_PEAK; i < PEAK_SEARCH_END; i++){
            //printf("%10f ", mag_data[i]);
            if(mag_data[i] > max_mag) {
                max_mag_i = i;
                max_mag = mag_data[i];
            }
        }
        //printf("\n");
        
        //Sum surrounding
        float sum = 0.f;
        for(int i = max_mag_i - SUM_AROUND_PEAK; i < (max_mag_i + SUM_AROUND_PEAK + 1); i++) {
            sum += mag_data[i];
            //printf("%10f ", mag_data[i]);
        }
        
        //printf("\n");
        
        //Scale by N_SAMPLES to recover pressure in Pa
        sum /= N_SAMPLES;
        
        //Apply inverse of logarithmic regression from test data
        float t90 = exp((sum + 80.07) / 53.77986);
        
        
        //Output iteration data
        iter_counter++;
        sampleRate = 1000000 / (sample_timer.read_us() / N_SAMPLES);
        printf("$DATA %.2f,%.2f,%.1f,%.1f,%1.5e,%.1f\n", iter_timer.read(), sampleRate , f_data[0] / N_SAMPLES, (max_mag_i * sampleRate / N_SAMPLES), sum, t90);
        
        //Set LEDs based on thresholds
        for(int i = 0; i < N_LEDS; i++) {
            leds[i] = 0;
            
            if(t90 >= thresholds[i + 1]) {
                leds[i] = 1;
            } else if(i < (N_LEDS - 1)) {
                leds[i] = (t90 - thresholds[i]) / (thresholds[i + 1] - thresholds[i]);
            }
        }
        
        //break;
    }
}