/*------------------------------------------------------------------------------
Library header file for heater operations
Date: 16/07/2018


------------------------------------------------------------------------------*/

#ifndef Heater_H
#define Heater_H
#include "mbed.h"
#include "ADS8568_ADC.h"
#include "FastPWM.h"

#define MEAS_DELAY          60     // measurement delay for ADC in us
#define WAIT_DELAY          3      // wait delay for ADC in ms

#define N_ROLL_AVG          1      // rolling average for R values
#define ALL_CH              15     //value of convst bus to read all chanels simultaneosly
#define Kp                  0.1f   //proportional gain
#define Ti                  0.2f       //Integration time
#define Kd                  0.1f   //Differentiator gain
#define WIND_UP_LIMIT       0.1f //The change in error which turns off the integral term
#define TICK_CYCLE          1000 //the number of tics per duty cycle
#define LOG_LIM             30       //Number of reads before the result is logged

#define GUARD_PWM_RATIO     21.0/82.0    //Ratio of guard duty cycle to heater duty cycle


class Heater
{
    //This class provides the interface through which each heater can be controlled
    public:
        /** Constructor
                 * @param i_port, the current port in the ADC
                 * @param v_port, the voltage port in the ADC
                 * @param * drive, a pointer to the heater drive
                 * @param * guard, a pointer to the guard
                 * @param * intercept, the gradient of the linear relationship between resistance and temperature
                 * @param * slope, the intercept of the linear relationship between resistance and temperature
                 * @param R_ref (default value 1), optional parameter sets the target value for R                 
        **/         
        Heater(const int i_port, const int v_port, FastPWM * drive, FastPWM * guard, const float intercept, const float slope, float R_ref = 1);
        
        //Public member functions

        void read();            //Updates the resistance and error values for the heater
        void hold(const int hold_time);  //Holds R_ref for hold_time miliseconds
        void ramp_R(const int ramp_time, const float R_final, const float R_start);   //Ramps for ramp_time miliseconds from R_start to R_final
        void output() const;  //Prints the current state of the heater
        void turn_off();    //Turns the heater off

        
        //Getters and setters
        void Set_R_ref(float R);
        float Get_R() const;
        
        
    protected:
        

        int curr;       //Latest current reading from ADC
        int v;          //Latest voltage reading from ADC
        float R;        //Latest resistance calculated from ADC current and voltage
        float R_ref;    //Current referance for resistance
        float error;    //R_ref - R
        
        
        //float error_diff; //Differential error
        float error_integrated; //Integrated error
        
        const int i_port;     //ADC port which corresponds to current measurements
        const int v_port;     //ADC port which corresponds to voltage measurements
        const float intercept;
        const float slope;
        
        FastPWM * drive;    //Pointer to the driver
        FastPWM * guard;    //Pointer to the guard
};
    
#endif